﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_VoiceCommand.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_Voice.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe.h>
#else
  #include <winext/cafe.h>
#endif

namespace nw {
namespace snd {
namespace internal {

namespace
{
struct WaveBufferPacket
{
    WaveBuffer waveBuffer;
    AdpcmContext adpcmContext;
};
static const unsigned int WAVE_BUFFER_POOL_COUNT = 512;
static WaveBufferPacket s_WaveBufferArray[ WAVE_BUFFER_POOL_COUNT ];
static const unsigned int NWVOICE_WAVE_BUFFER_POOL_COUNT = 48;
static WaveBufferPacket s_NwVoiceWaveBufferArray[ NWVOICE_WAVE_BUFFER_POOL_COUNT ];

WaveBufferPacket* GetFreeWaveBuffer()
{
    for( unsigned int i=0; i < WAVE_BUFFER_POOL_COUNT ; i++ )
    {
        WaveBufferPacket* packet = &s_WaveBufferArray[i];
        if ( packet->waveBuffer.status == WaveBuffer::STATUS_FREE ||
             packet->waveBuffer.status == WaveBuffer::STATUS_DONE )
        {
            return packet;
        }
    }
    return NULL;
}
WaveBufferPacket* GetFreeNwVoiceWaveBuffer()
{
    for( unsigned int i=0; i < NWVOICE_WAVE_BUFFER_POOL_COUNT ; i++ )
    {
        WaveBufferPacket* packet = &s_NwVoiceWaveBufferArray[i];
        if ( packet->waveBuffer.status == WaveBuffer::STATUS_FREE ||
             packet->waveBuffer.status == WaveBuffer::STATUS_DONE )
        {
            return packet;
        }
    }
    return NULL;
}
}

//===========================================================================
//
// AxVoiceCommand
//
//===========================================================================

AxVoiceCommand& AxVoiceCommand::GetInstance()
{
    static AxVoiceCommand instance;
    return instance;
}

void AxVoiceCommand::AxVoiceDisposeCallback( AxVoice* /* voice */, void* arg )
{
    u32 voiceId = reinterpret_cast<u32>(arg);
    VirtualVoiceManager::GetInstance().SetAxVoice( voiceId, NULL );
}

void AxVoiceCommand::ProcessCommandList( Command* commandList )
{
    Command* command = commandList;

    while( command != NULL )
    {
        switch( command->id )
        {
        case VOICE_COMMAND_ALLOC_VOICE:
        {
            VoiceCommandAlloc* c = reinterpret_cast<VoiceCommandAlloc*>(command);
            AxVoice* axVoice = AxVoice::AllocVoice(
                c->priority,
                AxVoiceDisposeCallback,
                reinterpret_cast<void*>(c->voiceId)
            );
            if ( axVoice == NULL ) {
                break;
            }

            axVoice->SetRenderer( c->rendererType );
            axVoice->m_pVoice = reinterpret_cast<Voice*>(c->userId);

            VirtualVoiceManager::GetInstance().SetAxVoice( c->voiceId, axVoice );
            break;
        }
        case VOICE_COMMAND_FREE_VOICE:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->Free();
            VirtualVoiceManager::GetInstance().SetAxVoice( c->voiceId, NULL );
            break;
        }
        case VOICE_COMMAND_SET_PRIORITY:
        {
            VoiceCommandPriority* c = reinterpret_cast<VoiceCommandPriority*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->SetPriority( c->priority );
            break;
        }
        case VOICE_COMMAND_PLAY:
        {
            VoiceCommandPlay* c = reinterpret_cast<VoiceCommandPlay*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->SetSampleRate( c->sampleRate );
            axVoice->SetSampleFormat( c->sampleFormat );
            axVoice->SetAdpcmParam( c->adpcmParam );
            axVoice->SetState( VOICE_STATE_PLAY );
            break;
        }
        case VOICE_COMMAND_PAUSE:
        {
            VoiceCommandPause* c = reinterpret_cast<VoiceCommandPause*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->SetState( VOICE_STATE_PAUSE );
            break;
        }
        case VOICE_COMMAND_STOP:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->FreeAllWaveBuffer();
            axVoice->SetState( VOICE_STATE_STOP );
            break;
        }
        case VOICE_COMMAND_APPEND_WAVE_BUFFER:
        {
            VoiceCommandAppendWaveBuffer* c = reinterpret_cast<VoiceCommandAppendWaveBuffer*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            WaveBufferPacket* packet = GetFreeWaveBuffer();
            if ( packet == NULL ) break;

            WaveBuffer* waveBuffer = &(packet->waveBuffer);
            waveBuffer->bufferAddress = c->bufferAddress;
            waveBuffer->sampleLength = c->sampleLength;
            waveBuffer->sampleOffset = c->sampleOffset;
            if ( c->adpcmContextEnable )
            {
                packet->adpcmContext = c->adpcmContext;
                waveBuffer->pAdpcmContext = &(packet->adpcmContext);
            }
            else
            {
                waveBuffer->pAdpcmContext = NULL;
            }
            waveBuffer->loopFlag = c->loopFlag;
            waveBuffer->userParam = c->tag;
            axVoice->AppendWaveBuffer( waveBuffer );
            break;
        }
        case VOICE_COMMAND_UPDATE_PARAM:
        {
            VoiceCommandParam* c = reinterpret_cast<VoiceCommandParam*>(command);
            AxVoice* axVoice = VirtualVoiceManager::GetInstance().GetAxVoice( c->voiceId );
            if ( axVoice == NULL ) break;

            axVoice->SetVoiceParam( c->voiceParam );
            break;
        }

        } // switch

        command = command->next;
    } // while

}

//===========================================================================
//
// NwVoiceCommand
//
//===========================================================================

NwVoiceCommand& NwVoiceCommand::GetInstance()
{
    static NwVoiceCommand instance;
    return instance;
}

void NwVoiceCommand::NwVoiceDisposeCallback( NwVoice* /* voice */, void* arg )
{
    u32 voiceId = reinterpret_cast<u32>(arg);
    VirtualVoiceManager::GetInstance().SetNwVoice( voiceId, NULL );
}

void NwVoiceCommand::ProcessCommandList( Command* commandList )
{
    Command* command = commandList;

    while( command != NULL )
    {
        switch( command->id )
        {
        case VOICE_COMMAND_ALLOC_VOICE:
        {
            VoiceCommandAlloc* c = reinterpret_cast<VoiceCommandAlloc*>(command);
            NwVoice* nwVoice = c->nwVoiceManager->AllocVoice(
                c->priority,
                NwVoiceDisposeCallback,
                reinterpret_cast<void*>(c->voiceId)
            );
            if ( nwVoice == NULL ) {
                break;
            }

//            nwVoice->SetRednerer( c->rendererType );
            nwVoice->m_pVoice = reinterpret_cast<Voice*>(c->userId);

            VirtualVoiceManager::GetInstance().SetNwVoice( c->voiceId, nwVoice );
            break;
        }
        case VOICE_COMMAND_FREE_VOICE:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->Free();
            VirtualVoiceManager::GetInstance().SetNwVoice( c->voiceId, NULL );
            break;
        }
        case VOICE_COMMAND_SET_PRIORITY:
        {
            VoiceCommandPriority* c = reinterpret_cast<VoiceCommandPriority*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->SetPriority( c->priority );
            break;
        }
        case VOICE_COMMAND_PLAY:
        {
            VoiceCommandPlay* c = reinterpret_cast<VoiceCommandPlay*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->SetSampleRate( c->sampleRate );
            nwVoice->SetSampleFormat( c->sampleFormat );
            nwVoice->SetAdpcmParam( c->adpcmParam );
            nwVoice->SetState( VOICE_STATE_PLAY );
            break;
        }
        case VOICE_COMMAND_PAUSE:
        {
            VoiceCommandPause* c = reinterpret_cast<VoiceCommandPause*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->SetState( VOICE_STATE_PAUSE );
            break;
        }
        case VOICE_COMMAND_STOP:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->FreeAllWaveBuffer();
            nwVoice->SetState( VOICE_STATE_STOP );
            break;
        }
        case VOICE_COMMAND_APPEND_WAVE_BUFFER:
        {
            VoiceCommandAppendWaveBuffer* c = reinterpret_cast<VoiceCommandAppendWaveBuffer*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            WaveBufferPacket* packet = GetFreeNwVoiceWaveBuffer();
            if ( packet == NULL ) break;

            WaveBuffer* waveBuffer = &(packet->waveBuffer);
            waveBuffer->bufferAddress = c->bufferAddress;
            waveBuffer->sampleLength = c->sampleLength;
            waveBuffer->sampleOffset = c->sampleOffset;
            if ( c->adpcmContextEnable )
            {
                packet->adpcmContext = c->adpcmContext;
                waveBuffer->pAdpcmContext = &(packet->adpcmContext);
            }
            else
            {
                waveBuffer->pAdpcmContext = NULL;
            }
            waveBuffer->loopFlag = c->loopFlag;
            waveBuffer->userParam = c->tag;
            nwVoice->AppendWaveBuffer( waveBuffer );
            break;
        }
        case VOICE_COMMAND_UPDATE_PARAM:
        {
            VoiceCommandParam* c = reinterpret_cast<VoiceCommandParam*>(command);
            NwVoice* nwVoice = VirtualVoiceManager::GetInstance().GetNwVoice( c->voiceId );
            if ( nwVoice == NULL ) break;

            nwVoice->SetVoiceParam( c->voiceParam );
            break;
        }

        } // switch

        command = command->next;
    } // while

}

//===========================================================================
//
// VoiceReplyCommand
//
//===========================================================================

VoiceReplyCommand& VoiceReplyCommand::GetInstance()
{
    static VoiceReplyCommand instance;
    return instance;
}

void VoiceReplyCommand::ProcessCommandList( Command* commandList )
{
    Command* command = commandList;

    while( command != NULL )
    {
        switch( command->id )
        {
        case VOICE_REPLY_COMMAND_WAVE_BUFFER_UPDATE:
        {
            break;
        }
        } // switch

        command = command->next;
    } // while

}


}}}
