﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Voice.h>
#include <nw/snd/snd_Config.h>
#include <nw/types.h>
#include <nw/assert.h>
#include <nw/ut.h>
#if !defined(NW_PLATFORM_ANDROID) && !defined(NW_PLATFORM_IOS)
#include <nw/dbg.h>
#endif
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_VoiceCommand.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe.h>
#else
  #include <winext/cafe.h>
#endif

namespace nw {
namespace snd {
namespace internal {

NwVoiceManager* Voice::s_pNwVoiceManager = NULL;

namespace
{

const BiquadFilterCoefficients BIQUAD_FILTER_COEF_ZERO = { 0 };
const OutputMix DEFAULT_TV_MIX =
{
    1.0f, 1.0f, 0.0f, 0.0f,  // mainBus
    0.0f, 0.0f, 0.0f, 0.0f,  // auxA
    0.0f, 0.0f, 0.0f, 0.0f,  // auxB
    0.0f, 0.0f, 0.0f, 0.0f   // auxC
};
const OutputMix DEFAULT_DRC_MIX =
{
    0.0f, 0.0f, 0.0f, 0.0f,  // mainBus
    0.0f, 0.0f, 0.0f, 0.0f,  // auxA
    0.0f, 0.0f, 0.0f, 0.0f,  // auxB
    0.0f, 0.0f, 0.0f, 0.0f   // auxC
};

}

void VoiceParam::Initialize()
{
    m_Volume = 1.0f;
    m_Pitch = 1.0f;
    m_TvMix = DEFAULT_TV_MIX;
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_DrcMix[i] = DEFAULT_DRC_MIX;
    }
    m_RemoteMix.Initialize();

    m_MonoFilterFlag = false;
    m_BiquadFilterFlag = false;
    m_RemoteFilterFlag = false;
    m_DrcFrontBypass = false;

    m_BiquadFilterCoefficients = BIQUAD_FILTER_COEF_ZERO;
    m_MonoFilterCutoff = 0;
    m_RemoteFilter = 0;
    m_InterpolationType = 0;
}

//==================================================================
//
// Voice
//
//==================================================================
Voice::Voice()
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
: m_VoiceId( VirtualVoiceManager::INVALID_VOICE_ID )
#else
    : m_pAxVoice( NULL )
    , m_pNwVoice( NULL )
#endif
{
}

Voice::~Voice()
{
}

void Voice::Initialize( u32 prioriry )
{
    m_State = VOICE_STATE_STOP;
    m_Priority = prioriry;

    m_VoiceParam.Initialize();

    m_SampleRate = 32000;
    m_SampleFormat = SAMPLE_FORMAT_PCM_S16;

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;

    m_VoiceInfoEnableFlag = false;
#endif

}

bool Voice::IsAvailable() const
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_VoiceId != VirtualVoiceManager::INVALID_VOICE_ID )
    {
        if ( m_AxVoiceFlag )
        {
            AxVoice* pVoice = VirtualVoiceManager::GetInstance().GetAxVoice(m_VoiceId);
            if (pVoice != NULL)
            {
                return pVoice->IsAvailable();
            }
        }
        else
        {
            NwVoice* pVoice = VirtualVoiceManager::GetInstance().GetNwVoice(m_VoiceId);
            if (pVoice != NULL)
            {
                return pVoice->IsAvailable();
            }
        }
        return true;
    }
#else
    if ( m_pAxVoice != NULL )
    {
        return m_pAxVoice->IsAvailable();
    }
    if ( m_pNwVoice != NULL )
    {
        return m_pNwVoice->IsAvailable();
    }
#endif

    return false;
}

bool Voice::AllocVoice( u32 prioriry, VoiceRendererType type )
{
    NW_MINMAXLT_ASSERT(type, 0, VOICE_RENDERER_COUNT);

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    m_VoiceId = VirtualVoiceManager::GetInstance().AllocVirtualVoice();
    if ( m_VoiceId == VirtualVoiceManager::INVALID_VOICE_ID ) return false;

    switch ( type )
    {
    case VOICE_RENDERER_SDK:
    case VOICE_RENDERER_SDK_DSP:
    case VOICE_RENDERER_SDK_PPC:
    {
        AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
        VoiceCommandAlloc* command = cmdmgr.AllocCommand<VoiceCommandAlloc>();
        //@@@ NW_LOG("__ALLOC__\n");
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return false;
        }
        command->id = VOICE_COMMAND_ALLOC_VOICE;
        command->voiceId = m_VoiceId;
        command->rendererType = type;
        command->priority = prioriry;
        command->userId = this;
        m_CommandTag = cmdmgr.PushCommand(command);

        m_AxVoiceFlag = true;
        break;
    }
    case VOICE_RENDERER_NW:
    {
        if ( s_pNwVoiceManager == NULL )  return false;

        NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
        VoiceCommandAlloc* command = cmdmgr.AllocCommand<VoiceCommandAlloc>();
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return false;
        }
        command->id = VOICE_COMMAND_ALLOC_VOICE;
        command->voiceId = m_VoiceId;
        command->rendererType = type;
        command->priority = prioriry;
        command->userId = this;  // for develop
        command->nwVoiceManager = s_pNwVoiceManager;
        m_CommandTag = cmdmgr.PushCommand(command);

        m_AxVoiceFlag = false;
        break;
    }
    default:
        return false;
    }
    Initialize( prioriry );

#else // NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    switch ( type )
    {
    case VOICE_RENDERER_SDK:
    case VOICE_RENDERER_SDK_DSP:
    case VOICE_RENDERER_SDK_PPC:
    {
        AxVoice* axVoice = AxVoice::AllocVoice( prioriry, AxVoiceDisposeCallback, this );
        if ( axVoice == NULL ) {
            return false;
        }

        axVoice->SetRenderer( type );

        axVoice->m_pVoice = this; // for develop

        m_pAxVoice = axVoice;
    }
    break;
    case VOICE_RENDERER_NW:
        if ( s_pNwVoiceManager != NULL )
        {
            NwVoice* nwVoice = s_pNwVoiceManager->AllocVoice( prioriry, NwVoiceDisposeCallback, this );
            if ( nwVoice == NULL ) {
                return false;
            }

            m_pNwVoice = nwVoice;
        }
        break;
    default:
        return false;
    }
    Initialize( prioriry );

#endif // NW_SND_CONFIG_ENABLE_VOICE_COMMAND

    return true;
}

void Voice::Free()
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_AxVoiceFlag )
    {
        AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
        VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>();
        //@@@ NW_LOG("__FREE__\n");
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            FreeAllWaveBuffer();
            return;
        }
        command->id = VOICE_COMMAND_FREE_VOICE;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);
    }
    else
    {
        NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
        VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>();
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            FreeAllWaveBuffer();
            return;
        }
        command->id = VOICE_COMMAND_FREE_VOICE;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);
    }

    FreeAllWaveBuffer();

    if ( m_VoiceId != VirtualVoiceManager::INVALID_VOICE_ID )
    {
        VirtualVoiceManager::GetInstance().FreeVirtualVoice( m_VoiceId );
        m_VoiceId = VirtualVoiceManager::INVALID_VOICE_ID;
    }
    m_PlayPosition = 0;
#else
    m_State = VOICE_STATE_STOP;

    if ( m_pAxVoice != NULL )
    {
        m_pAxVoice->Free();
        m_pAxVoice = NULL;
    }
    if ( m_pNwVoice != NULL )
    {
        m_pNwVoice->Free();
        m_pNwVoice = NULL;
    }
#endif
}

void Voice::SetPriority(u32 priority)
{
    m_Priority = priority;

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_AxVoiceFlag )
    {
        AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
        VoiceCommandPriority* command = cmdmgr.AllocCommand<VoiceCommandPriority>();
        //@@@ NW_LOG("__PRIO__\n");
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_SET_PRIORITY;
        command->voiceId = m_VoiceId;
        command->priority = priority;
        cmdmgr.PushCommand(command);
    }
    else
    {
        NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
        VoiceCommandPriority* command = cmdmgr.AllocCommand<VoiceCommandPriority>();
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_SET_PRIORITY;
        command->voiceId = m_VoiceId;
        command->priority = priority;
        cmdmgr.PushCommand(command);
    }
#else
    if ( m_pAxVoice != NULL )
    {
        m_pAxVoice->SetPriority( priority );
    }
    if ( m_pNwVoice != NULL )
    {
        m_pNwVoice->SetPriority( priority );
    }
#endif
}

void Voice::SetState( VoiceState state )
{
    m_State = state;

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( state == VOICE_STATE_PLAY )
    {
        if ( m_AxVoiceFlag )
        {
            AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
            VoiceCommandPlay* command = cmdmgr.AllocCommand<VoiceCommandPlay>();
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                return;
            }
            command->id = VOICE_COMMAND_PLAY;
            command->voiceId = m_VoiceId;
            command->sampleFormat = m_SampleFormat;
            command->sampleRate = m_SampleRate;
            command->adpcmParam = m_AdpcmParam;
            cmdmgr.PushCommand(command);
        }
        else
        {
            NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
            VoiceCommandPlay* command = cmdmgr.AllocCommand<VoiceCommandPlay>();
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                return;
            }
            command->id = VOICE_COMMAND_PLAY;
            command->voiceId = m_VoiceId;
            command->sampleFormat = m_SampleFormat;
            command->sampleRate = m_SampleRate;
            command->adpcmParam = m_AdpcmParam;
            cmdmgr.PushCommand(command);
        }
    }
    else if ( state == VOICE_STATE_PAUSE )
    {
        if ( m_AxVoiceFlag )
        {
            AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
            VoiceCommandPause* command = cmdmgr.AllocCommand<VoiceCommandPause>();
            //@@@ NW_LOG("__PAUSE__\n");
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                return;
            }
            command->id = VOICE_COMMAND_PAUSE;
            command->voiceId = m_VoiceId;
            cmdmgr.PushCommand(command);
        }
        else
        {
            NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
            VoiceCommandPause* command = cmdmgr.AllocCommand<VoiceCommandPause>();
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                return;
            }
            command->id = VOICE_COMMAND_PAUSE;
            command->voiceId = m_VoiceId;
            cmdmgr.PushCommand(command);
        }
    }
    else if ( state == VOICE_STATE_STOP )
    {
        if ( m_AxVoiceFlag )
        {
            AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
            VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>();
            //@@@ NW_LOG("__FREE__\n");
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                FreeAllWaveBuffer();
                return;
            }
            command->id = VOICE_COMMAND_STOP;
            command->voiceId = m_VoiceId;
            cmdmgr.PushCommand(command);
        }
        else
        {
            NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
            VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>();
            if ( command == NULL )
            {
                NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
                FreeAllWaveBuffer();
                return;
            }
            command->id = VOICE_COMMAND_STOP;
            command->voiceId = m_VoiceId;
            cmdmgr.PushCommand(command);
        }
        FreeAllWaveBuffer();
    }
#else
    if ( m_pAxVoice != NULL )
    {
        m_pAxVoice->SetState( state );

        if ( state == VOICE_STATE_PLAY )
        {
            m_pAxVoice->SetSampleRate( m_SampleRate );
            m_pAxVoice->SetSampleFormat( m_SampleFormat );
            m_pAxVoice->SetAdpcmParam( m_AdpcmParam );
        }
        else if ( state == VOICE_STATE_STOP )
        {
            m_pAxVoice->FreeAllWaveBuffer();
        }
    }
    if ( m_pNwVoice != NULL )
    {
        m_pNwVoice->SetState( state );
        if ( state == VOICE_STATE_PLAY )
        {
            m_pNwVoice->SetSampleRate( m_SampleRate );
            m_pNwVoice->SetSampleFormat( m_SampleFormat );
            m_pNwVoice->SetAdpcmParam( m_AdpcmParam );
        }
        else if ( state == VOICE_STATE_STOP )
        {
            m_pNwVoice->FreeAllWaveBuffer();
        }
    }

#endif
}

void Voice::AppendWaveBuffer( WaveBuffer* waveBuffer )
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_AxVoiceFlag )
    {
        AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
        VoiceCommandAppendWaveBuffer* command = cmdmgr.AllocCommand<VoiceCommandAppendWaveBuffer>();
        //@@@ NW_LOG("__APPEND__\n");
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_APPEND_WAVE_BUFFER;
        command->voiceId = m_VoiceId;
        command->tag = waveBuffer;
        command->bufferAddress = waveBuffer->bufferAddress;
        command->sampleLength = waveBuffer->sampleLength;
        command->sampleOffset = waveBuffer->sampleOffset;
        if ( waveBuffer->pAdpcmContext != NULL )
        {
            command->adpcmContext = *(waveBuffer->pAdpcmContext);
            command->adpcmContextEnable = true;
        }
        else
        {
            command->adpcmContextEnable = false;
        }
        command->loopFlag = waveBuffer->loopFlag;
        cmdmgr.PushCommand(command);
    }
    else
    {
        NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
        VoiceCommandAppendWaveBuffer* command = cmdmgr.AllocCommand<VoiceCommandAppendWaveBuffer>();
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_APPEND_WAVE_BUFFER;
        command->voiceId = m_VoiceId;
        command->tag = waveBuffer;
        command->bufferAddress = waveBuffer->bufferAddress;
        command->sampleLength = waveBuffer->sampleLength;
        command->sampleOffset = waveBuffer->sampleOffset;
        if ( waveBuffer->pAdpcmContext != NULL )
        {
            command->adpcmContext = *(waveBuffer->pAdpcmContext);
            command->adpcmContextEnable = true;
        }
        else
        {
            command->adpcmContextEnable = false;
        }
        command->loopFlag = waveBuffer->loopFlag;
        cmdmgr.PushCommand(command);
    }

    waveBuffer->next = NULL;
    waveBuffer->status = WaveBuffer::STATUS_WAIT;

    if ( m_WaveBufferListEnd == NULL )
    {
        m_WaveBufferListBegin = waveBuffer;
    }
    else
    {
        m_WaveBufferListEnd->next = waveBuffer;
    }
    m_WaveBufferListEnd = waveBuffer;
#else
    if ( m_pAxVoice != NULL )
    {
        m_pAxVoice->AppendWaveBuffer(waveBuffer);
    }
    else if ( m_pNwVoice != NULL )
    {
        m_pNwVoice->AppendWaveBuffer(waveBuffer);
    }
#endif
}

void Voice::FreeAllWaveBuffer()
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    WaveBuffer* waveBuffer = m_WaveBufferListBegin;
    while( waveBuffer != NULL ) {
        waveBuffer->status = WaveBuffer::STATUS_DONE;
        waveBuffer = waveBuffer->next;
    }
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;
#endif
}

void Voice::UpdateParam()
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    if ( m_AxVoiceFlag )
    {
        AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
        VoiceCommandParam* command = cmdmgr.AllocCommand<VoiceCommandParam>();
        //@@@ NW_LOG("__PARAM__\n");
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_UPDATE_PARAM;
        command->voiceId = m_VoiceId;
        command->voiceParam = m_VoiceParam;
        cmdmgr.PushCommand(command);
    }
    else
    {
        NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
        VoiceCommandParam* command = cmdmgr.AllocCommand<VoiceCommandParam>();
        if ( command == NULL )
        {
            NW_WARNING( false, "[%s:%d] cannot alloc command", __FUNCTION__, __LINE__ );
            return;
        }
        command->id = VOICE_COMMAND_UPDATE_PARAM;
        command->voiceId = m_VoiceId;
        command->voiceParam = m_VoiceParam;
        cmdmgr.PushCommand(command);
    }
#else
    if ( m_pAxVoice != NULL )
    {
        m_pAxVoice->SetVoiceParam( m_VoiceParam );
    }
    else if ( m_pNwVoice != NULL )
    {
        m_pNwVoice->SetVoiceParam( m_VoiceParam );
    }
#endif
}

u32 Voice::GetPlayPosition() const
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    return m_PlayPosition;
#else
    if ( m_pAxVoice != NULL )
    {
        return m_pAxVoice->GetPlayPosition();
    }
    if ( m_pNwVoice != NULL )
    {
        return m_pNwVoice->GetPlayPosition();
    }
    return 0;
#endif
}

void Voice::SetRemoteMix( const RemoteOutputMix& mix )
{
    m_VoiceParam.m_RemoteMix = mix;
}

void Voice::SetMonoFilter( bool enable, u16 cutoff )
{
    m_VoiceParam.m_MonoFilterFlag = enable;

    if ( enable )
    {
        // 呼び出し側の MultiVoice::CalcLpf で 16000 未満であることが保証されている
        NW_ASSERT( cutoff < 16000 );

        m_VoiceParam.m_MonoFilterCutoff = cutoff;
    }
}

void Voice::SetBiquadFilter( bool enable, const BiquadFilterCoefficients* coef )
{
    m_VoiceParam.m_BiquadFilterFlag = enable;

    if ( enable )
    {
        // 呼び出し側の MultiVoice::CalcBiquadFilter で非 NULL であることが保証されている
        NW_ASSERT_NOT_NULL( coef );

        m_VoiceParam.m_BiquadFilterCoefficients = *coef;
    }
}

void Voice::SetRemoteFilter( bool enable, u8 filter )
{
    m_VoiceParam.m_RemoteFilterFlag = enable;

    if ( enable )
    {
        m_VoiceParam.m_RemoteFilter = filter;
    }
}

void Voice::detail_SetNwVoiceManager( NwVoiceManager* renderer )
{
    s_pNwVoiceManager = renderer;
}

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND

void Voice::AxVoiceDisposeCallback( AxVoice* axVoice, void* arg )
{
    Voice* voice = reinterpret_cast<Voice*>(arg);
    NW_NULL_ASSERT( voice );
    NW_ASSERT( voice->m_pAxVoice == axVoice );
    voice->m_pAxVoice = NULL;
}

void Voice::NwVoiceDisposeCallback( NwVoice* nwVoice, void* arg )
{
    Voice* voice = reinterpret_cast<Voice*>(arg);
    NW_NULL_ASSERT( voice );
    NW_ASSERT( voice->m_pNwVoice == nwVoice );
    voice->m_pNwVoice = NULL;
}

#else

void Voice::UpdateVoiceStatus()
{
    if ( m_VoiceId == VirtualVoiceManager::INVALID_VOICE_ID ) return;

    if ( ! m_VoiceInfoEnableFlag )
    {
        if ( m_AxVoiceFlag )
        {
            AxVoiceCommand& cmdmgr = AxVoiceCommand::GetInstance();
            if ( ! cmdmgr.IsFinishCommand( m_CommandTag ) ) return;
        }
        else
        {
            NwVoiceCommand& cmdmgr = NwVoiceCommand::GetInstance();
            if ( ! cmdmgr.IsFinishCommand( m_CommandTag ) ) return;
        }

        m_VoiceInfoEnableFlag = true;
    }

    const VoiceInfo* voiceInfo = VirtualVoiceManager::GetInstance().GetVoiceInfo( m_VoiceId );
    if ( voiceInfo == NULL ) return;

    if ( voiceInfo->userId != this ) return;

    m_State = voiceInfo->voiceState;

    if ( voiceInfo->voiceState == VOICE_STATE_PLAY )
    {
        if ( voiceInfo->waveBufferStatus == WaveBuffer::STATUS_FREE )
        {
            FreeAllWaveBuffer();
        }
        else
        {
            WaveBuffer* waveBuffer = m_WaveBufferListBegin;
            if (waveBuffer != NULL)
            {
                while( waveBuffer != NULL && waveBuffer != voiceInfo->waveBufferTag )
                {
                    waveBuffer->status = WaveBuffer::STATUS_DONE;
                    waveBuffer = waveBuffer->next;
                }

                if(waveBuffer != NULL)
                {
                    waveBuffer->status = voiceInfo->waveBufferStatus;
                    m_WaveBufferListBegin = waveBuffer;
                }
                else
                {
                    FreeAllWaveBuffer();
                }
            }
        }
    }

    m_PlayPosition = voiceInfo->playPosition;
}

#endif

//===========================================================================
//
// VirtualVoiceManager
//
//===========================================================================

VirtualVoiceManager& VirtualVoiceManager::GetInstance()
{
    static VirtualVoiceManager instance;
    return instance;
}

void VirtualVoiceManager::Initialize()
{
    m_VoiceInfoTableRead = 0;

    for(u32 i = 0 ; i < VIRTUAL_VOICE_ELEMENT_COUNT; i++ )
    {
        sVirtualVoiceAllocationTable[i] = 0xffffffff;
    }
    for(u32 i = 0 ; i < VIRTUAL_VOICE_COUNT; i++ )
    {
        m_AxVoiceTable[i] = NULL;
        m_NwVoiceTable[i] = NULL;

        m_VoiceInfoTable[0][i].voiceState = VOICE_STATE_STOP;
        m_VoiceInfoTable[1][i].voiceState = VOICE_STATE_STOP;
    }
}


u32 VirtualVoiceManager::AllocVirtualVoice()
{
    for(u32 i = 0 ; i < VIRTUAL_VOICE_ELEMENT_COUNT; i++ )
    {
        u32 x = sVirtualVoiceAllocationTable[i];
        if ( x == 0 ) continue;

        u32 shift;
        if ( x & 0x000000ff ) {
            shift = 0;
        }
        else if ( x & 0x0000ff00 ) {
            shift = 8;
        }
        else if ( x & 0x00ff0000 ) {
            shift = 16;
        }
        else /*if ( x & 0xff000000 )*/ {
            shift = 24;
        }
        x >>= shift;

        while( (x & 1) == 0 ) {
            shift++;
            x >>= 1;
        }

        sVirtualVoiceAllocationTable[i] &= ~(1<<shift);
        u32 voiceId = i * 32 + shift;
        return voiceId;
    }
    return INVALID_VOICE_ID;
}

void VirtualVoiceManager::FreeVirtualVoice(u32 voiceId)
{
    u32 i = (voiceId >> 5); // i = voiceId / 32;
    u32 shift = (voiceId & 0x1f);
    sVirtualVoiceAllocationTable[i] |= (1<<shift);
}

void VirtualVoiceManager::UpdateVoiceInfo()
{
    u32 writeIndex = m_VoiceInfoTableRead ? 0 : 1;

    for(u32 i = 0 ; i < VIRTUAL_VOICE_COUNT; i++ )
    {
        VoiceInfo* voiceInfo = &m_VoiceInfoTable[writeIndex][i];

        if (m_AxVoiceTable[i] != NULL )
        {
            m_AxVoiceTable[i]->UpdateVoiceInfo( voiceInfo );
        }
        else if (m_NwVoiceTable[i] != NULL )
        {
            m_NwVoiceTable[i]->UpdateVoiceInfo( voiceInfo );
        }
        else
        {
            voiceInfo->voiceState = VOICE_STATE_STOP;
            voiceInfo->waveBufferStatus = WaveBuffer::STATUS_FREE;
            voiceInfo->userId = NULL;
        }
    }

    m_VoiceInfoTableRead = writeIndex;
}

}}}
