﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_TaskThread.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_DriverCommand.h>

namespace nw {
namespace snd {
namespace internal {

/*---------------------------------------------------------------------------*
  Name:         TaskThread

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
#if defined(NW_PLATFORM_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
TaskThread::TaskThread()
: m_IsFinished( false ),
  m_IsCreated( false ),
  m_Thread(this)
{
}
#if defined(NW_PLATFORM_WIN32)
#pragma warning(pop) // disable:4355
#endif

TaskThread::~TaskThread()
{
    if ( m_IsCreated )
    {
        Destroy();
    }
}

bool TaskThread::Create( s32 priority, void* stackBase, u32 stackSize, u32 affinityMask )
{
    // 多重初期化チェック
    if ( m_IsCreated )
    {
        Destroy();
    }

    m_IsFinished = false;

    nw::ut::Thread::CreateArg arg;
    arg.priority = priority;
#if defined(NW_PLATFORM_CAFE)
    arg.stackBase = stackBase;
    arg.stackSize = stackSize;
    arg.attribute = affinityMask;
    arg.nameString = "nw::snd::TaskThread";
#else
    (void)stackBase; (void)stackSize; (void)affinityMask;
#endif

    bool result = m_Thread.Create( arg );

    if ( ! result )
    {
        NW_LOG("TaskThread::Create Failed: OSCreateThread is Failed\n");
        return false;
    }
    m_Thread.Resume();

    m_IsCreated = true;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Destroy

  Description:  タスクスレッドを終了する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void TaskThread::Destroy()
{
    if ( ! m_IsCreated )
    {
        return;
    }

    // サウンドスレッド終了メッセージ送信
    m_IsFinished = true;
    TaskManager::GetInstance().CancelWaitTask();

    // スレッドの終了を待つ
    m_Thread.Join();

    m_IsCreated = false;
}

/*---------------------------------------------------------------------------*
  Name:         ThreadHandlerProc

  Description:  タスクスレッドプロシージャ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void TaskThread::ThreadHandlerProc()
{
    while ( ! m_IsFinished )
    {
        TaskManager::GetInstance().WaitTask();
        if ( m_IsFinished )
        {
            break;
        }

        {
            // NOTE: タスクスレッドのロック。
            //
            // SoundSystem::(Try)LockDataLoadThread 関数で外部からロックされることによって
            // いったんスレッドが停止します。
            //
            // タスクスレッドはストリームデータのロードや、
            // プレイヤーヒープへのデータロードを行いますが、
            // 「スリープ中はファイルアクセス禁止」の制約に沿うため、
            // このロックが追加されました。
            ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
            TaskManager::GetInstance().ExecuteTask();
        }

        DriverCommand::GetInstanceForTaskThread().RecvCommandReply();
    }
}

TaskThread& TaskThread::GetInstance()
{
    static TaskThread instance;
    return instance;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

