﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamSoundFileReader.h>
#include <nw/snd/snd_StreamSoundPrefetchFileReader.h>
#include <nw/snd/snd_WaveFile.h>                // EncodeMethod

namespace nw {
namespace snd {
namespace internal {

namespace {

const u32 SIGNATURE_FILE_STP               = NW_UT_MAKE_SIGWORD( 'F', 'S', 'T', 'P' );
const u32 SIGNATURE_INFO_BLOCK_STP         = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
const u32 SIGNATURE_PREFETCHDATA_BLOCK_STP = NW_UT_MAKE_SIGWORD( 'P', 'D', 'A', 'T' );

const int SUPPORTED_FILE_VERSION_STP    = NW_UT_MAKE_VERSION( 0, 1, 0, 0 );   // ライブラリがサポートする最低バージョン
const int CURRENT_FILE_VERSION_STP      = NW_UT_MAKE_VERSION( 0, 2, 1, 0 );   // ライブラリがサポートする最新バージョン

} // anonymous namespace

StreamSoundPrefetchFileReader::StreamSoundPrefetchFileReader()
    : m_pHeader( NULL ),
    m_pInfoBlockBody( NULL ),
    m_pPrefetchDataBlockBody( NULL )
{
}

void StreamSoundPrefetchFileReader::Initialize( const void* streamSoundPrefetchFile )
{
    NW_ASSERT_NOT_NULL( streamSoundPrefetchFile );

    if ( ! IsValidFileHeader( streamSoundPrefetchFile ) ) return;

    m_pHeader =
        reinterpret_cast<const StreamSoundPrefetchFile::FileHeader*>( streamSoundPrefetchFile );
    const StreamSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NW_ASSERT( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_STP );
    if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_STP )
    {
        return;
    }
    const StreamSoundPrefetchFile::PrefetchDataBlock* dataBlock = m_pHeader->GetPrefetchDataBlock();
    NW_ASSERT( dataBlock->header.kind == SIGNATURE_PREFETCHDATA_BLOCK_STP );
    if ( dataBlock->header.kind != SIGNATURE_PREFETCHDATA_BLOCK_STP )
    {
        return;
    }

    m_pInfoBlockBody         = &infoBlock->body;
    m_pPrefetchDataBlockBody = &dataBlock->body;
    NW_ASSERT( m_pInfoBlockBody->GetStreamSoundInfo()->oneBlockBytes % 32 == 0 );
}

void StreamSoundPrefetchFileReader::Finalize()
{
    m_pHeader = NULL;
    m_pInfoBlockBody = NULL;
    m_pPrefetchDataBlockBody = NULL;
}

bool StreamSoundPrefetchFileReader::ReadStreamSoundInfo(
        StreamSoundFile::StreamSoundInfo* strmInfo ) const
{
    NW_ASSERT_NOT_NULL( m_pInfoBlockBody );
    const StreamSoundFile::StreamSoundInfo* info =
        m_pInfoBlockBody->GetStreamSoundInfo();

    NW_ASSERT_ALIGN32( (u32)(info->oneBlockBytes) );
    NW_ASSERT_ALIGN32( (u32)(info->lastBlockPaddedBytes) );

    // struct copy
    *strmInfo = *info;

    return true;
}

bool StreamSoundPrefetchFileReader::ReadDspAdpcmChannelInfo(
        DspAdpcmParam* pParam,
        DspAdpcmLoopParam* pLoopParam,
        int channelIndex ) const
{
    NW_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NW_ASSERT_NOT_NULL( pParam );
    NW_ASSERT_NOT_NULL( pLoopParam );

    const StreamSoundFile::DspAdpcmChannelInfo* src =
        m_pInfoBlockBody->
        GetChannelInfoTable()->
        GetChannelInfo( channelIndex )->
        GetDspAdpcmChannelInfo();
    if ( src == NULL ) return false;

    *pParam = src->param;
    *pLoopParam = src->loopParam;
    return true;
}

bool StreamSoundPrefetchFileReader::ReadPrefetchDataInfo(
PrefetchDataInfo* pDataInfo, int prefetchIndex ) const
{
    NW_ASSERT_NOT_NULL( m_pPrefetchDataBlockBody );
    NW_ASSERT_NOT_NULL( pDataInfo );

    const StreamSoundPrefetchFile::PrefetchData* data = m_pPrefetchDataBlockBody->GetPrefetchData( prefetchIndex );
    NW_ASSERT_NOT_NULL( data );

    pDataInfo->startFrame   = data->startFrame;
    pDataInfo->prefetchSize = data->prefetchSize;

    const StreamSoundPrefetchFile::PrefetchSample* sample = data->GetPrefetchSample();
    pDataInfo->dataAddress  = sample->GetSampleAddress();

    return true;
}

bool StreamSoundPrefetchFileReader::IsValidFileHeader( const void* streamSoundPrefetchFile ) const
{
    NW_ASSERT_NOT_NULL( streamSoundPrefetchFile );

#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( streamSoundPrefetchFile );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( streamSoundPrefetchFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG(
            header.signature == SIGNATURE_FILE_STP,
            "invalid file signature. stream data is not available." );
    if ( header.signature != SIGNATURE_FILE_STP ) return false;

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_STP <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_STP) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfstp file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_STP, CURRENT_FILE_VERSION_STP, header.version
    );
    return isSupportedVersion;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

