﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamSound.h>
#include <nw/snd/snd_StreamSoundHandle.h>
#include <nw/snd/snd_DriverCommand.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         StreamSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
StreamSound::StreamSound(
    StreamSoundInstanceManager& manager
)
: m_Manager( manager )
, m_InitializeFlag(false)
, m_pCacheBuffer(NULL)
, m_CacheSize(0)
{
}


void StreamSound::Initialize()
{
    BasicSound::Initialize();

    m_pTempSpecialHandle = NULL;

    for ( int i=0; i<STRM_TRACK_NUM; i++ )
    {
        m_TrackVolume[ i ].InitValue( 0.0f );
        m_TrackVolume[ i ].SetTarget( 1.0f, 1 );
    }

    for ( int i=0; i<WAVE_CHANNEL_MAX; i++ )
    {
        m_AvailableTrackBitFlag[ i ] = 0x0;
    }

    m_InitializeFlag = true;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSound::Finalize()
{
    if ( ! m_InitializeFlag ) return;
    m_InitializeFlag = false;

    BasicSound::Finalize();
    m_Manager.Free( this );
}

void StreamSound::Setup(const driver::StreamSoundPlayer::SetupArg& arg)
{
    NW_NULL_ASSERT( arg.pBufferPool );

    m_AllocTrackFlag = arg.allocTrackFlag;
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundSetup* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundSetup>();
        command->id = DRIVER_COMMAND_STRM_SETUP;
        command->player = &m_PlayerInstance;
        command->arg = arg;  // 構造体コピー
        cmdmgr.PushCommand(command);
    }

    for (u32 trackNo = 0; trackNo < internal::STRM_TRACK_NUM; trackNo++)
    {
        const u8 trackChannelCount = arg.trackInfos.track[trackNo].channelCount;
        for (u8 channelNo = 0; channelNo < trackChannelCount; channelNo++ )
        {
            m_AvailableTrackBitFlag[channelNo] += 0x1 << trackNo;
        }
    }
}

void StreamSound::Prepare(const driver::StreamSoundPlayer::PrepareBaseArg& arg)
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundPrepare>();
        command->id = DRIVER_COMMAND_STRM_PREPARE;
        command->player = &m_PlayerInstance;
        command->arg.baseArg = arg;     // 構造体コピー
        command->arg.cacheBuffer = GetCacheBuffer();
        command->arg.cacheSize = GetCacheSize();
        cmdmgr.PushCommand(command);
    }
}

void StreamSound::PreparePrefetch(const void* strmPrefetchFile, const driver::StreamSoundPlayer::PrepareBaseArg& arg)
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundPreparePrefetch* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundPreparePrefetch>();
        command->id = DRIVER_COMMAND_STRM_PREPARE_PREFETCH;
        command->player = &m_PlayerInstance;
        command->arg.strmPrefetchFile = strmPrefetchFile;
        command->arg.baseArg = arg;
        cmdmgr.PushCommand(command);
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdateMoveValue

  Description:  サウンドのアップデート

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSound::UpdateMoveValue()
{
    BasicSound::UpdateMoveValue();

    u16 bitFlag = m_AllocTrackFlag;
    for ( int trackNo=0; trackNo<STRM_TRACK_NUM; trackNo++,bitFlag>>=1 )
    {
        if ( bitFlag & 0x01 )
        {
            m_TrackVolume[ trackNo ].Update();
        }
    }
}

void StreamSound::OnUpdateParam()
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    u16 bitFlag = m_AllocTrackFlag;
    for ( int trackNo=0; trackNo<STRM_TRACK_NUM; trackNo++, bitFlag>>=1 )
    {
        if ( bitFlag & 0x01 )
        {
            DriverCommandStreamSoundTrackParam* command =
                cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
            command->id = DRIVER_COMMAND_STRM_TRACK_VOLUME;
            command->player = &m_PlayerInstance;
            command->trackBitFlag = static_cast<u32>( 1 << trackNo );
            command->value = m_TrackVolume[ trackNo ].GetValue();
            cmdmgr.PushCommand(command);
        }
    }
}

void StreamSound::SetTrackVolume( u32 trackBitFlag, f32 volume, int frames )
{
    if ( volume < 0.0f ) volume = 0.0f;

    u16 bitFlag = ( m_AllocTrackFlag & trackBitFlag );
    for( int trackNo = 0;
         trackNo < STRM_TRACK_NUM && trackBitFlag != 0;
         trackNo++, bitFlag >>=1
    )
    {
        if ( bitFlag & 0x01 )
        {
            m_TrackVolume[ trackNo ].SetTarget( volume, frames );
        }
    }
}

void StreamSound::SetTrackInitialVolume( u32 trackBitFlag, u32 volume )
{
    u16 bitFlag = ( m_AllocTrackFlag & trackBitFlag );
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackInitialVolume* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackInitialVolume>();
    command->id = DRIVER_COMMAND_STRM_TRACK_INITIAL_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = bitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackOutputLine( u32 trackBitFlag, u32 lineFlag )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_OUTPUTLINE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->u32Value = lineFlag;
    cmdmgr.PushCommand(command);
}

void StreamSound::ResetTrackOutputLine( u32 trackBitFlag )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_OUTPUTLINE_RESET;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    cmdmgr.PushCommand(command);
}



void StreamSound::SetTrackMainOutVolume( u32 trackBitFlag, f32 volume )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackChannelMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam )
{
    NW_ASSERT_MAXLT( srcChNo, WAVE_CHANNEL_MAX );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackMixParameter>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_MIXPARAMETER;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    for ( int channel = 0; channel < CHANNEL_INDEX_NUM; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackPan( u32 trackBitFlag, f32 pan )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_PAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackSurroundPan( u32 trackBitFlag, f32 span )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_SPAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = span;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackMainSend( u32 trackBitFlag, f32 send )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_MAINSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackFxSend( u32 trackBitFlag, AuxBus bus, f32 send )
{
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_TV_FXSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->u32Value = bus;
    cmdmgr.PushCommand(command);
}



void StreamSound::SetTrackDrcOutVolume( u32 drcIndex, u32 trackBitFlag, f32 volume )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackDrcChannelMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam )
{
    NW_ASSERT_MAXLT( srcChNo, WAVE_CHANNEL_MAX );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackMixParameter>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_MIXPARAMETER;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    for ( int channel = 0; channel < CHANNEL_INDEX_NUM; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_PAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_SPAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = span;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_MAINSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
    command->id = DRIVER_COMMAND_STRM_TRACK_DRC_FXSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->u32Value = bus;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}


/*---------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSound::OnUpdatePlayerPriority()
{
    m_Manager.UpdatePriority( this, CalcCurrentPlayerPriority() );
}

bool StreamSound::IsAttachedTempSpecialHandle()
{
    return m_pTempSpecialHandle != NULL;
}

void StreamSound::DetachTempSpecialHandle()
{
    m_pTempSpecialHandle->DetachSound();
}

bool StreamSound::ReadStreamDataInfo( StreamDataInfo* info ) const
{
    if ( ! IsPlayerAvailable() )
    {
        NW_WARNING(false, "ReadStreamDataInfo failed. Please use this API after playback preparing finished.");
        return false;
    }
    return m_PlayerInstance.ReadStreamDataInfo( info );
}

long StreamSound::GetPlayLoopCount() const
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetPlayLoopCount();
}

long StreamSound::GetPlaySamplePosition(bool isOriginalSamplePosition) const
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetPlaySamplePosition(isOriginalSamplePosition);
}

f32 StreamSound::GetFilledBufferPercentage() const
{
    if ( ! IsPlayerAvailable() ) return 0.0f;
    return m_PlayerInstance.GetFilledBufferPercentage();
}

bool StreamSound::IsPrepared() const
{
    if ( ! IsPlayerAvailable() ) return false;
    return m_PlayerInstance.IsPrepared();
}

bool StreamSound::IsSuspendByLoadingDelay() const
{
    if ( ! IsPlayerAvailable() ) return false;
    return m_PlayerInstance.IsSuspendByLoadingDelay();
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

