﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamBufferPool.h>
#include <nw/ut.h>
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

void StreamBufferPool::Initialize( void* pBuffer, size_t size, int blockCount )
{
    if ( blockCount == 0 )
    {
        return;
    }

    NW_WARNING(
        snd::internal::Util::IsValidMemoryForDsp( pBuffer, size ),
        "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
        pBuffer,
        ut::AddOffsetToPtr( pBuffer, size ),
        size );

    // ストリームバッファのアドレス (全チャンネル分)
    m_pBuffer = pBuffer;

    // 全チャンネル分のストリームバッファサイズ
    m_BufferSize = size;

    // 何チャンネル分か？ (bcsar のストリームチャンネル数)
    m_BlockCount = blockCount;

    // 1 チャンネルあたりのバッファサイズ
    // (各バッファ頭が 32 バイト整列するように 32 バイトの倍数にする)
    m_BlockSize = ut::RoundDown( size / blockCount, 32 );

    m_AllocCount = 0;
    ut::internal::Memset( m_AllocFlags, 0, sizeof(m_AllocFlags) );

    NW_ASSERTMSG( m_BlockCount <= BLOCK_MAX, "Too large stream buffer size." );
}

void StreamBufferPool::Finalize()
{
    m_pBuffer = NULL;
    m_BufferSize = 0;
    m_BlockSize = 0;
    m_BlockCount = 0;
}

void* StreamBufferPool::Alloc()
{
    if ( m_AllocCount >= m_BlockCount ) return NULL;

    const int availableByte = ut::RoundUp( m_BlockCount, BIT_PER_BYTE ) / BIT_PER_BYTE;

    for( int byteIndex = 0 ; byteIndex < availableByte ; byteIndex++ )
    {
        const u8 byte = m_AllocFlags[ byteIndex ];
        if ( byte == 0xff ) continue;

        u8 mask = ( 1 << 0 );
        for( int bitIndex = 0 ; bitIndex < BIT_PER_BYTE ; bitIndex++, mask<<=1 )
        {
            if ( ( byte & mask ) == 0 )
            {
                m_AllocFlags[ byteIndex ] |= mask;
                m_AllocCount++;

                const int totalIndex = byteIndex * BIT_PER_BYTE + bitIndex;
                void* buffer = ut::AddOffsetToPtr( m_pBuffer, m_BlockSize * totalIndex );
                // NW_LOG("[StreamBufferPool::%s] m_BlockSize(%p) return(%p)\n",
                //         __FUNCTION__, m_BlockSize, buffer);
                return buffer;
            }
        }
    }

    return NULL;
}

void StreamBufferPool::Free( void* pPtr )
{
    PtrDiff offset = ut::GetOffsetFromPtr( m_pBuffer, pPtr );

    const unsigned long totalIndex = static_cast<unsigned long>( offset / m_BlockSize );
    NW_ASSERT( totalIndex < BLOCK_MAX );
    const unsigned long byteIndex = ( totalIndex / BIT_PER_BYTE );
    const unsigned long bitIndex  = ( totalIndex % BIT_PER_BYTE );

    const int mask = ( 1 << bitIndex );
    NW_ASSERT( ( m_AllocFlags[ byteIndex ] & mask ) != 0 );
    m_AllocFlags[ byteIndex ] &= ~mask;
    m_AllocCount--;
    NW_ASSERT( m_AllocCount >= 0 );
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

