﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundThread.h>

#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_ChannelManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_Voice.h>
#include <nw/snd/snd_VoiceCommand.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         SoundThread

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
#if defined(NW_PLATFORM_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
SoundThread::SoundThread()
:
#ifndef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    m_AxUserCallback(NULL),
#endif
    m_CreateFlag( false )
  , m_PauseFlag( false )
  , m_IsEnableGetTick( false )
  , m_pNwVoiceRenderer( NULL )
    , m_AudioFrameCounter(0)
    , m_AxCallbackCounter(0)
  , m_CurrentProfileBuffer(-1)
  , m_pLastProfile(NULL)
// #ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
  , m_Thread(this)
// #endif
{
#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    m_BlockingQueue.Initialize( m_MsgBuffer, THREAD_MESSAGE_BUFSIZE );
#endif
}
#if defined(NW_PLATFORM_WIN32)
#pragma warning(pop) // disable:4355
#endif

/*---------------------------------------------------------------------------*
  Name:         GetInstance

  Description:  シングルトンのインスタンスを取得します

  Arguments:    無し

  Returns:      インスタンスへの参照
 *---------------------------------------------------------------------------*/
SoundThread& SoundThread::GetInstance()
{
    static SoundThread instance;
    return instance;
}

bool SoundThread::PrepareForCreate( bool enableGetTick )
{
    NW_ASSERTMSG( HardwareManager::GetInstance().IsInitialized(),
            "note initialized nw::snd::internal::HardwareManager.\n" );
    if ( m_CreateFlag )
    {
        return true;
    }
    m_CreateFlag = true;
    m_IsEnableGetTick = enableGetTick;
    return true;
}

bool SoundThread::CreateSoundThread(
        s32 priority,
        void* stackBase,
        u32 stackSize,
        u32 affinityMask,
        bool enableGetTick )
{
    m_LastProfileAvailable = false;

    bool prepareResult = PrepareForCreate( enableGetTick );
    if ( prepareResult == false )
    {
        return false;
    }

    m_SoundThreadAffinityMask = affinityMask;

    nw::ut::Thread::CreateArg arg;
    // TODO: priority値変換
    arg.priority = priority;
#if defined(NW_PLATFORM_CAFE)
    arg.stackBase = stackBase;
    arg.stackSize = stackSize;
    arg.attribute = affinityMask;
    arg.nameString = "nw::snd::SoundThread";
#else
    (void)stackBase; (void)stackSize;
#endif

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    NW_LOG("NW: SoundThread is run on core %d\n", OSGetCoreId() );

#ifdef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    AXPB_ERROR_CODE error = AXRegisterAppFrameCallback( HwCallbackFunc );
    if (error != AXPB_ERROR_NONE)
    {
        return false;
    }
#else
    m_AxUserCallback = AXRegisterFrameCallback(HwCallbackFunc);
#endif

    AXInitProfile( m_ProfileUpdateBuffer[0], MAX_PROFILE_COUNT );
    m_CurrentProfileBuffer = 0;
    m_pLastProfile = NULL;
    return true;
#else
    m_AxCallbackCounter = 0;
    m_AudioFrameCounter = 0;

    bool result = m_Thread.Create( arg );

    if ( result )
    {
        m_Thread.Resume();
    }

    return result;
#endif
}

void SoundThread::Initialize()
{
}

void SoundThread::Destroy()
{
    if ( ! m_CreateFlag ) return;

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    // サウンドスレッド終了メッセージ送信
    m_BlockingQueue.Jam( reinterpret_cast<ut::MessageQueue::MessageType>( MESSAGE_SHUTDOWN ), true );

    // スレッドの終了を待つ
    m_Thread.Join();
#endif

    m_CreateFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドスレッドを終了する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundThread::Finalize()
{
    Destroy();

    // プレイヤーを全て停止する
    for ( PlayerCallbackList::Iterator itr = m_PlayerCallbackList.GetBeginIter();
            itr != m_PlayerCallbackList.GetEndIter();
        )
    {
        PlayerCallbackList::Iterator curItr = itr++;
        curItr->OnShutdownSoundThread();
    }
}

/*---------------------------------------------------------------------------*
  Name:         HwCallbackFunc

  Description:  アラーム周期で呼びだされるコールバック

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundThread::HwCallbackFunc()
{
    SoundThread* soundThread = &GetInstance();
    soundThread->HwCallbackProc();
}

/*---------------------------------------------------------------------------*
  Name:         HwCallbackProc

  Description:  アラーム周期で呼びだされるコールバック

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundThread::HwCallbackProc()
{
    m_AxCallbackCounter++;
    m_AxCallbackCounter &= 0x0fffffff;

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    AxVoiceCommand::GetInstance().ProcessCommand();
    AxVoice::detail_UpdatePriorityList();
#endif

    // AX ボイス API を使ってパラメータを AX 層に反映
    //   AxVoice::UpdateState
    AxVoice::detail_UpdateAllVoices();

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    // サウンドスレッド起動メッセージ送信
    if ( ! m_PauseFlag )
    {
#if !defined(NW_PLATFORM_ANDROID) && !defined(NW_PLATFORM_IOS)
        bool result = m_BlockingQueue.Send(
            reinterpret_cast<ut::MessageQueue::MessageType>( MESSAGE_HW_CALLBACK | m_AxCallbackCounter ), false );

        if ( result == false )
        {
            Util::WarningLogger::GetInstance().Log(
                Util::WarningLogger::LOG_ID_SOUNDTHREAD_FAILED_WAKEUP );
        }
#else
        // 高負荷時に MessageQueue によるスレッドの起床がうまく動かないため、
        // サウンドスレッドの処理を直接呼び出す
        FrameProcess();
#endif
    }
#endif

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    VirtualVoiceManager::GetInstance().UpdateVoiceInfo();
#endif

#ifndef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    if (m_AxUserCallback)
    {
        m_AxUserCallback();
    }
#endif
}

void SoundThread::ForceWakeup()
{
#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    m_BlockingQueue.Send(
        reinterpret_cast<ut::MessageQueue::MessageType>( MESSAGE_FORCE_WAKEUP ), false );
#endif
}

void SoundThread::RegisterSoundFrameCallback( SoundFrameCallback* callback )
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_SoundFrameCallbackList.PushBack( callback );
}

void SoundThread::UnregisterSoundFrameCallback( SoundFrameCallback* callback )
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_SoundFrameCallbackList.Erase( callback );
}

void SoundThread::RegisterPlayerCallback( PlayerCallback* callback )
{
    m_PlayerCallbackList.PushBack( callback );
}

void SoundThread::UnregisterPlayerCallback( PlayerCallback* callback )
{
    m_PlayerCallbackList.Erase( callback );
}

/*---------------------------------------------------------------------------*
  Name:         FrameProcess

  Description:  サウンドフレーム処理

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundThread::FrameProcess()
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    nw::ut::Tick beginTick = nw::ut::Tick::GetSystemCurrent();

    const AXPROFILE* profileBuffer = m_ProfileUpdateBuffer[m_CurrentProfileBuffer];
    m_CurrentProfileBuffer++;
    if ( m_CurrentProfileBuffer >= 3 ) m_CurrentProfileBuffer = 0;
    u32 profileCount = AXGetSwapProfile( m_ProfileUpdateBuffer[m_CurrentProfileBuffer], MAX_PROFILE_COUNT);

    // サウンドフレーム開始コールバック
    {
        for ( SoundFrameCallbackList::Iterator itr = m_SoundFrameCallbackList.GetBeginIter();
              itr != m_SoundFrameCallbackList.GetEndIter();
        )
        {
            SoundFrameCallbackList::Iterator curItr = itr++;
            curItr->OnBeginSoundFrame();
        }
    }

    u32 nwVoiceCount = 0;

    // サウンドスレッドのメイン処理
    {
        // SDK処理結果をボイスに反映
        //   MultiVoice::UpdateVoiceStatus
        //    波形停止チェック
        //    ボイスドロップチェック
        MultiVoiceManager::GetInstance().UpdateAllVoiceStatus();

        // コマンド処理
        {
            // NW_PROFILE("SoundThread::ProcessCommand");
            while( DriverCommand::GetInstanceForTaskThread().ProcessCommand() ) {}
            while( DriverCommand::GetInstance().ProcessCommand() ) {}
        }

        // プレイヤー更新
        {
            // NW_PROFILE("SoundThread::UpdatePlayer");
            for ( PlayerCallbackList::Iterator itr = m_PlayerCallbackList.GetBeginIter();
                  itr != m_PlayerCallbackList.GetEndIter();
            )
            {
                PlayerCallbackList::Iterator curItr = itr++;
                curItr->OnUpdateFrameSoundThread();
            }
        }

        // チャンネルパラメータの計算
        //   Channel::Update
        //     パラメータ計算
        //     エンベロープ・LFO
        //     MultiVoiceへのパラメータ反映
        ChannelManager::GetInstance().UpdateAllChannel();

        // ボイスパラメータの計算
        //   MultiVoice::Calc
        //     パラメータ計算
        //     Voiceへのパラメータ反映
        //   MultiVoice::Update
        //     一部パラメータ計算
        //     ボイスステータス更新
        //     Voiceへのパラメータ反映
        MultiVoiceManager::GetInstance().UpdateAllVoices();

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
        if ( m_pNwVoiceRenderer != NULL )
        {
            // Nwボイスのパラメータ計算
            //   NwVoice::UpdateState
            //     ボイスステータス更新
            m_pNwVoiceRenderer->UpdateAllVoices();

            // Nwボイスの波形合成
            nwVoiceCount = m_pNwVoiceRenderer->Synthesize();
        }

        // AXボイスのプライオリティソート
        AxVoice::detail_UpdatePriorityList();
#endif

        // HardwareManager 更新
        HardwareManager::GetInstance().Update();

        // 乱数列更新
        (void)Util::CalcRandom();
    }

    // サウンドフレーム終了コールバック
    {
        for ( SoundFrameCallbackList::Iterator itr = m_SoundFrameCallbackList.GetBeginIter();
              itr != m_SoundFrameCallbackList.GetEndIter();
        )
        {
            SoundFrameCallbackList::Iterator curItr = itr++;
            curItr->OnEndSoundFrame();
        }
    }

    // ユーザーコールバック呼び出し
    if ( m_UserCallback )
    {
        m_UserCallback( m_UserCallbackArg );
    }

    nw::ut::Tick endTick = nw::ut::Tick::GetSystemCurrent();

    // プロファイル処理
    if ( m_LastProfileAvailable )
    {
        if ( m_pLastProfile != NULL )
        {
            RecordProfile( *m_pLastProfile, m_LastProfileBegin, m_LastProfileEnd, nwVoiceCount );
            m_pLastProfile = NULL;
        }
        if ( profileCount > 1 )
        {
            for( u32 i=0; i<profileCount-1; i++)
            {
                const AXPROFILE& src = profileBuffer[i];

                RecordProfile( src, m_LastProfileBegin, m_LastProfileEnd, nwVoiceCount );
            }
        }
    }

    if ( profileCount > 0 )
    {
        m_pLastProfile = &profileBuffer[profileCount-1];
    }
    m_LastProfileBegin = beginTick;
    m_LastProfileEnd = endTick;
    m_LastProfileAvailable = true;
}

void SoundThread::RecordProfile( const AXPROFILE& src, nw::ut::Tick beginTick,  nw::ut::Tick endTick, u32 nwVoiceCount )
{
    if ( m_ProfileReaderList.IsEmpty() ) return;

    SoundProfile dst;

    dst.nwFrameProcess.begin = beginTick;
    dst.nwFrameProcess.end   = endTick;
    dst.nwVoiceParamUpdate.begin = nw::ut::Tick(src.userCallbackStart);
    dst.nwVoiceParamUpdate.end   = nw::ut::Tick(src.userCallbackEnd);
    dst.voiceRendering.begin = nw::ut::Tick(src.axPPCStart);
    dst.voiceRendering.end = nw::ut::Tick(src.axPPCEnd);
    dst.syncVoiceParam.begin = nw::ut::Tick(src.axFrameStart);
    dst.syncVoiceParam.end   = nw::ut::Tick(src.axPPCStart);
    dst.outputFormatProcess.begin = nw::ut::Tick(src.userFinalMixEnd);
    dst.outputFormatProcess.end   = nw::ut::Tick(src.axFrameEnd);
    dst.auxProcess.begin = nw::ut::Tick(src.userAuxProcStart);
    dst.auxProcess.end   = nw::ut::Tick(src.userAuxProcEnd);
    dst.mainMixProcess.begin = nw::ut::Tick(src.axPPCPostStart);
    dst.mainMixProcess.end   = nw::ut::Tick(src.axPPCPostEnd);
    dst.finalMixProcess.begin = nw::ut::Tick(src.userFinalMixStart);
    dst.finalMixProcess.end   = nw::ut::Tick(src.userFinalMixEnd);
    dst.axFrameProcess.begin = nw::ut::Tick(src.axFrameStart);
    dst.axFrameProcess.end   = nw::ut::Tick(src.axFrameEnd);
    dst.dspFrameProcess.begin = nw::ut::Tick(src.axDSPStart);
    dst.dspFrameProcess.end   = nw::ut::Tick(src.axDSPEnd);

    // ボイス数
    dst.dspVoiceCount = src.axNumDspVoices;
    dst.ppcVoiceCount = src.axNumVoices - src.axNumDspVoices;
    dst.nwVoiceCount = nwVoiceCount;
    dst.totalVoiceCount = dst.dspVoiceCount + dst.ppcVoiceCount + dst.nwVoiceCount;

    // フレームティック
    dst.axFrameProcessTick = ( src.userCallbackEnd - src.axFrameStart ) + ( src.axFrameEnd - src.axPPCPostStart );

    dst.nwFrameProcessTick = ( dst.nwFrameProcess.end - dst.nwFrameProcess.begin );

    if ( (s64)dst.nwFrameProcess.begin <= (s64)dst.axFrameProcess.begin && (s64)dst.axFrameProcess.begin <= (s64)dst.nwFrameProcess.end )
    {
        dst.nwFrameProcessTick -= ( src.userCallbackEnd - src.axFrameStart );
    }

    nw::ut::Tick axPPCPostStart = nw::ut::Tick(src.axPPCPostStart);

    if ( (s64)dst.nwFrameProcess.begin <= axPPCPostStart && axPPCPostStart <= (s64)dst.nwFrameProcess.end )
    {
        dst.nwFrameProcessTick -= ( src.axFrameEnd - src.axPPCPostStart );
    }

    // 互換性維持のための不要コード
    dst.axNumVoices = src.axNumVoices;
    dst.dspCycles = 0;

    // ProfileReaderへの反映
    for (ProfileReaderList::Iterator it = m_ProfileReaderList.begin(); it != m_ProfileReaderList.end(); ++it)
    {
        it->Record(dst);
    }
}

/*---------------------------------------------------------------------------*
  Name:         SoundThreadProc

  Description:  サウンドスレッドプロシージャ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundThread::ThreadHandlerProc()
{
    NW_LOG("NW: SoundThread is run on core %d\n", OSGetCoreId() );

#ifdef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    AXPB_ERROR_CODE error = AXRegisterAppFrameCallback( HwCallbackFunc );
    NW_ASSERTMSG(error == AXPB_ERROR_NONE, "AXRegisterAppFrameCallback is failed(%d)\n",error);
    if (error != AXPB_ERROR_NONE)
    {
        return;
    }
#else
    m_AxUserCallback = AXRegisterFrameCallback(HwCallbackFunc);
#endif

    AXInitProfile( m_ProfileUpdateBuffer[0], MAX_PROFILE_COUNT );
    m_CurrentProfileBuffer = 0;
    m_pLastProfile = NULL;

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    for ( ;; )
    {
        ut::MessageQueue::MessageType msgBuf;
        m_BlockingQueue.Recv(&msgBuf, true);
        Message msg = static_cast<Message>( reinterpret_cast<u32>(msgBuf) & 0xf0000000 );
        u32 msgData = ( reinterpret_cast<u32>(msgBuf) & 0x0fffffff );

        if ( msg == MESSAGE_HW_CALLBACK )
        {
            m_AudioFrameCounter = msgData;
            FrameProcess();
        }
        else if ( msg == MESSAGE_FORCE_WAKEUP )
        {
            // コマンド処理だけ行う
            while(DriverCommand::GetInstance().ProcessCommand()){}
            while(DriverCommand::GetInstanceForTaskThread().ProcessCommand()){}
        }
        else if ( msg == MESSAGE_SHUTDOWN )
        {
            break;
        }
    }
#endif

#ifdef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    error = AXDeregisterAppFrameCallback( HwCallbackFunc );
    NW_ASSERTMSG(error == AXPB_ERROR_NONE, "AXDeegisterAppFrameCallback is failed(%d)\n",error);
#else
    AXRegisterFrameCallback(m_AxUserCallback);
#endif
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

