﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Sound3DManager.h>

#include <nw/snd/snd_Sound3DEngine.h>
#include <nw/snd/snd_SoundArchive.h>

namespace nw {
namespace snd {

namespace {

Sound3DEngine sSound3DEngine;

}

/* ------------------------------------------------------------------------
        member function
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         Sound3DManager

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Sound3DManager::Sound3DManager()
: m_pSound3DEngine( &sSound3DEngine ),
  m_MaxPriorityReduction( 32 ),
  m_PanRange( 0.9f ),
  m_SonicVelocity( 0.0f ),
  m_BiquadFilterType( BIQUAD_FILTER_TYPE_INHERIT ),
  m_FreeMemSizeAfterCheking( 0 ),
  m_IsInitialized( false )
{
}

/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  3Dサウンドに必要なメモリサイズを取得する

  Arguments:    arc - サウンドアーカイブ

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
size_t Sound3DManager::GetRequiredMemSize( const SoundArchive* arc )
{
    NW_NULL_ASSERT( arc );

    s32 numSounds = 0;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        numSounds += soundArchivePlayerInfo.sequenceSoundMax;
        numSounds += soundArchivePlayerInfo.streamSoundMax;
        numSounds += soundArchivePlayerInfo.waveSoundMax;
    }

    return static_cast<size_t>( numSounds ) * sizeof( Sound3DParam );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  3Dサウンドを使用するメモリの初期化を行う

  Arguments:    arc - サウンドアーカイブ
                buffer - バッファのアドレス
                size - バッファサイズ

  Returns:      成功したら true
 *---------------------------------------------------------------------------*/
bool Sound3DManager::Initialize(
    const SoundArchive* arc,
    void* buffer,
    size_t size
)
{
    if ( m_IsInitialized == true )
    {
        return false;
    }
    NW_UNUSED_VARIABLE( arc );

    NW_NULL_ASSERT( arc );
    NW_NULL_ASSERT( buffer );
    size_t memSize = GetRequiredMemSize( arc );
    NW_ASSERT( size >= memSize );

    // ActorParams Create
    m_ParamPool.Create( buffer, size );
    m_FreeMemSizeAfterCheking = size - memSize;
    m_IsInitialized = true;

    return true;
}

bool Sound3DManager::InitializeWithMoreSoundArchive( const SoundArchive* arc )
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    NW_UNUSED_VARIABLE( arc );
    NW_NULL_ASSERT( arc );
    size_t memSize = GetRequiredMemSize( arc );

    m_FreeMemSizeAfterCheking -= memSize;
    NW_ASSERT( m_FreeMemSizeAfterCheking >= 0 );
    return true;
}

bool Sound3DManager::Finalize()
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    // 初期値に戻す
    m_pSound3DEngine = &sSound3DEngine;
    m_MaxPriorityReduction = 32;
    m_PanRange = 0.9f;
    m_SonicVelocity = 0.0f;
    m_BiquadFilterType = BIQUAD_FILTER_TYPE_INHERIT;

    // リスナーを外す
    while ( !m_ListenerList.IsEmpty() )
    {
        Sound3DListener& listener = m_ListenerList.GetBack();
        m_ListenerList.Erase( &listener );
    }

    // パラメータプールを破棄する
    m_ParamPool.Destroy();
    m_FreeMemSizeAfterCheking = 0;

    m_IsInitialized = false;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetEngine

  Description:  3D エンジンを登録する

  Arguments:    engine - 3D エンジン

  Returns:      なし
 *---------------------------------------------------------------------------*/
void Sound3DManager::SetEngine( Sound3DEngine* engine )
{
    m_pSound3DEngine = engine;
}

/*---------------------------------------------------------------------------*
  Name:         detail_UpdateAmbientParam

  Description:  3Dサウンドのパラメータ更新

  Arguments:    arg   - AmbientArgのポインタ
                soundId - サウンドＩＤ
                param - 3Dサウンドの計算結果を格納するパラメータ構造体

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DManager::detail_UpdateAmbientParam(
    const void* arg,
    u32 soundId,
    SoundAmbientParam* param
)
{
    const Sound3DParam* actorParam = static_cast<const Sound3DParam*>( arg );

    if ( m_pSound3DEngine != NULL )
    {
        m_pSound3DEngine->detail_UpdateAmbientParam(
            this,
            actorParam,
            soundId,
            param
        );
    }
}

/*---------------------------------------------------------------------------*
  Name:         detail_GetAmbientPriority

  Description:  アンビエントプライオリティの計算

  Arguments:    arg   - AmbientArgのポインタ
                soundId - サウンドＩＤ

  Returns:      None.
  *---------------------------------------------------------------------------*/
int Sound3DManager::detail_GetAmbientPriority(
    const void* arg,
    u32 soundId
)
{
    const Sound3DParam* actorParam = static_cast<const Sound3DParam*>( arg );

    int priority = 0;

    if ( m_pSound3DEngine != NULL )
    {
        priority = m_pSound3DEngine->GetAmbientPriority(
            this,
            actorParam,
            soundId
        );
    }

    return priority;
}

/*---------------------------------------------------------------------------*
  Name:         AllocAmbientArg

  Description:  AmbientArgを確保します

  Arguments:    argSize - AmbientArgのサイズ

  Returns:      AmbientArgへのポインタ
 *---------------------------------------------------------------------------*/
void* Sound3DManager::detail_AllocAmbientArg( size_t argSize )
{
    if ( argSize != sizeof( Sound3DParam ) ) return NULL;
    return m_ParamPool.Alloc();
}

/*---------------------------------------------------------------------------*
  Name:         FreeAmbientArg

  Description:  AmbientArgを解放します

  Arguments:    arg - 解放するAmbientArg
                sound - サウンドのインスタンス

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DManager::detail_FreeAmbientArg( void* arg, const internal::BasicSound* sound )
{
    (void)sound;

    Sound3DParam* actorParam = static_cast<Sound3DParam*>( arg );
    m_ParamPool.Free( actorParam );
}

/*---------------------------------------------------------------------------*
  Name:         SetBiquadFilterType

  Description:  Biquad フィルタの種類を指定します

  Arguments:    type - Biquad フィルタの種類

  Returns:      なし
 *---------------------------------------------------------------------------*/
void Sound3DManager::SetBiquadFilterType( int type )
{
    NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_MIN, BIQUAD_FILTER_TYPE_MAX );
    m_BiquadFilterType = type;
}

Sound3DParam::Sound3DParam()
: ctrlFlag( 0 ),
  actorUserParam( 0 ),
  soundUserParam( 0 ),
  decayRatio( 0.5f ),
  decayCurve( SoundArchive::Sound3DInfo::DECAY_CURVE_LOG ),
  dopplerFactor( 0 )
{
}

} // namespace nw::snd
} // namespace nw

