﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_RemoteSpeakerManager.h>
#include <nw/assert.h>

namespace nw {
namespace snd {
namespace internal {

/*---------------------------------------------------------------------------*
  Name:         GetInstance

  Description:  シングルトンのインスタンスを取得する

  Arguments:    なし

  Returns:      インスタンス
 *---------------------------------------------------------------------------*/
RemoteSpeakerManager& RemoteSpeakerManager::GetInstance()
{
    static RemoteSpeakerManager instance;
    return instance;
}

/*---------------------------------------------------------------------------*
  Name:         RemoteSpeakerManager

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
RemoteSpeakerManager::RemoteSpeakerManager()
#if defined( NW_PLATFORM_CAFE )
: m_Initialized( false )
#endif // defined( NW_PLATFORM_CAFE )
{
#if defined( NW_PLATFORM_CAFE )
    for ( int i=0; i<NW_SND_WPAD_MAX_CONTROLLERS; i++ )
    {
        m_Speaker[ i ].SetChannelIndex( i );
    }
#endif // defined( NW_PLATFORM_CAFE )
}

/*---------------------------------------------------------------------------*
  Name:         GetRemoteSpeaker

  Description:  リモコンスピーカーのインスタンスを取得する

  Arguments:    channelIndex - リモコンチャンネル番号

  Returns:      インスタンス
 *---------------------------------------------------------------------------*/
RemoteSpeaker& RemoteSpeakerManager::GetRemoteSpeaker( int channelIndex )
{
    NW_MINMAXLT_ASSERT( channelIndex, 0, NW_SND_WPAD_MAX_CONTROLLERS );

    return m_Speaker[ channelIndex ];
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  リモコンのスピーカーを使用するための初期化を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeakerManager::Initialize()
{
#if defined( NW_PLATFORM_CAFE )
    if ( m_Initialized ) return;

    // アラームセットアップ
    OSCreateAlarm( &m_RemoteSpeakerAlarm );
    OSSetPeriodicAlarm(
        &m_RemoteSpeakerAlarm,
        OSGetTime(),
        WPAD_STRM_INTERVAL,
        RemoteSpeakerAlarmProc
    );

    m_Initialized = true;
#endif // defined( NW_PLATFORM_CAFE )
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  リモコンのスピーカーを使用のシャットダウンを行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeakerManager::Finalize()
{
#if defined( NW_PLATFORM_CAFE )
    if ( ! m_Initialized ) return;

    // アラーム停止
    OSCancelAlarm( &m_RemoteSpeakerAlarm );

    m_Initialized = false;
#endif // defined( NW_PLATFORM_CAFE )
}

#if defined( NW_PLATFORM_CAFE )
/*---------------------------------------------------------------------------*
  Name:         RemoteSpeakerAlarmProc

  Description:  リモコンへデータを送る処理を行うプロシージャ
                アラーム周期で呼び出される

  Arguments:    alarm -
                context -

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeakerManager::RemoteSpeakerAlarmProc( OSAlarm *alarm, OSContext *context )
{
    (void)alarm;
    (void)context;

    RemoteSpeakerManager& manager = GetInstance();
    NW_ASSERT( &manager.m_RemoteSpeakerAlarm == alarm );

    if ( AXRmtGetSamplesLeft() < SAMPLES_PER_AUDIO_PACKET ) return;

    for( int channelIndex = 0; channelIndex < WPAD_MAX_CONTROLLERS; channelIndex++ )
    {
        if ( manager.m_Speaker[ channelIndex ].IsAvailable() )
        {
            // AXからリモコン用のサンプルデータ取得
            s16 remoteSampleBuffer[ SAMPLES_PER_AUDIO_PACKET ];

            s32 sample = AXRmtGetSamples(
                channelIndex,
                remoteSampleBuffer,
                SAMPLES_PER_AUDIO_PACKET
            );
            NW_WARNING( sample == SAMPLES_PER_AUDIO_PACKET, "wrong remote sample size" );

            // スピーカークラスにサンプルデータを送る
            manager.m_Speaker[ channelIndex ].UpdateStreamData( remoteSampleBuffer );
        }

        // スピーカークラスのフレーム処理
        manager.m_Speaker[ channelIndex ].Update();
    }

    // AXのリモコン用バッファを先を進める
    AXRmtAdvancePtr( SAMPLES_PER_AUDIO_PACKET );
}
#endif // defined( NW_PLATFORM_CAFE )

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

