﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_PlayerHeapDataManager.h>

namespace nw {
namespace snd {
namespace internal {

PlayerHeapDataManager::PlayerHeapDataManager()
: m_IsInitialized( false ),
  m_IsFinalized ( true )
{
}

PlayerHeapDataManager::~PlayerHeapDataManager()
{
    Finalize();
}

void PlayerHeapDataManager::Initialize( const SoundArchive* arc )
{
    if ( m_IsInitialized )
    {
        return;
    }
    m_IsInitialized = true;
    m_IsFinalized = false;

    for ( int i = 0; i < FILE_ADDRESS_COUNT; i++ )
    {
        m_FileAddress[i].fileId = SoundArchive::INVALID_ID;
        m_FileAddress[i].address = NULL;
    }
    SetSoundArchive( arc );
}

void PlayerHeapDataManager::Finalize()
{
    if ( m_IsFinalized )
    {
        return;
    }
    m_IsInitialized = false;
    m_IsFinalized = true;
    SetSoundArchive( NULL );
}

const void* PlayerHeapDataManager::SetFileAddress(
        SoundArchive::FileId fileId, const void* address )
{
    return SetFileAddressToTable( fileId, address );
}
const void* PlayerHeapDataManager::GetFileAddress( SoundArchive::FileId fileId ) const
{
    // ファイルの検索順は、SoundDataManager → PlayerHeapDataManager。
    // サウンドアーカイブ内からの検索は、SoundDataManager で行っているので、
    // ここでは、PlayerHeapDataManager 内の検索だけでよい。
    return GetFileAddressFromTable( fileId );
}

void PlayerHeapDataManager::InvalidateData( const void* /*start*/, const void* /*end*/ )
{
    // ファイルアドレステーブルから破棄
    for ( int i = 0; i < FILE_ADDRESS_COUNT; i++ )
    {
        m_FileAddress[i].fileId = SoundArchive::INVALID_ID;
        m_FileAddress[i].address = NULL;
    }
}

const void* PlayerHeapDataManager::SetFileAddressToTable(
        SoundArchive::FileId fileId, const void* address )
{
    // 同じ fileId がある場合
    for ( int i = 0; i< FILE_ADDRESS_COUNT; i++ )
    {
        if ( m_FileAddress[i].fileId == fileId )
        {
            const void* prev = m_FileAddress[i].address;
            m_FileAddress[i].address = address;
            return prev;
        }
    }

    // ない場合
    for ( int i = 0; i< FILE_ADDRESS_COUNT; i++ )
    {
        if ( m_FileAddress[i].fileId == SoundArchive::INVALID_ID )
        {
            m_FileAddress[i].fileId = fileId;
            m_FileAddress[i].address = address;
            return NULL;
        }
    }

    // ここにくることは無いはず
    NW_ASSERTMSG( false, "invalid PlayerHeap" );
    return NULL;
}

const void* PlayerHeapDataManager::GetFileAddressFromTable( SoundArchive::FileId fileId ) const
{
    for ( int i = 0; i< FILE_ADDRESS_COUNT; i++ )
    {
        if ( m_FileAddress[i].fileId == fileId )
        {
            return m_FileAddress[i].address;
        }
    }
    return NULL;
}

const void* PlayerHeapDataManager::GetFileAddressImpl( SoundArchive::FileId fileId ) const
{
    return GetFileAddressFromTable( fileId );
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

