﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_MemorySoundArchive.h>
#include <nw/snd/snd_GroupFileReader.h>

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         MemorySoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
MemorySoundArchive::MemorySoundArchive()
: m_pData( NULL )
{
}

/*---------------------------------------------------------------------------*
  Name:         ~MemorySoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
MemorySoundArchive::~MemorySoundArchive()
{
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  サウンドアーカイブを初期化する

  Arguments:    soundArchiveData - サウンドアーカイブデータ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool MemorySoundArchive::Initialize( const void* soundArchiveData )
{
    NW_NULL_ASSERT( soundArchiveData );
    NW_ALIGN4_ASSERT( soundArchiveData );

    m_FileReader.Initialize( soundArchiveData );
    SoundArchive::Initialize( &m_FileReader );

    const unsigned long infoBlockOffset = m_FileReader.GetInfoBlockOffset();

    m_FileReader.SetInfoBlock(
        ut::AddOffsetToPtr( soundArchiveData, infoBlockOffset )
    );

    const unsigned long stringBlockOffset = m_FileReader.GetStringBlockOffset();
    const unsigned long stringBlockSize = m_FileReader.GetStringBlockSize();

    // 文字列ブロックへのリファレンスは必ず含まれるが、
    // ブロック自体が含まれない場合、stringBlockOffset/stringBlockSize の両方共、
    // 0xffffffff が入っている。
    if ( stringBlockOffset == 0xffffffff || stringBlockSize == 0xffffffff )
    {
        // なにもしない
    }
    else
    {
        m_FileReader.SetStringBlock(
                ut::AddOffsetToPtr( soundArchiveData, stringBlockOffset ));
    }

    m_pData = soundArchiveData;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドアーカイブを閉じる

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void MemorySoundArchive::Finalize()
{
    m_pData = NULL;
    m_FileReader.Finalize();
    SoundArchive::Finalize();
}

const void* MemorySoundArchive::detail_GetFileAddress( FileId fileId ) const
{
    FileInfo fileInfo;
    bool result = detail_ReadFileInfo( fileId, &fileInfo );

    if ( result && fileInfo.offsetFromFileBlockHead != 0xffffffff )
    {
        return ut::AddOffsetToPtr( m_pData,
                fileInfo.offsetFromFileBlockHead +
                m_FileReader.GetFileBlockOffset() );
    }
    else
    {
        // 「自動(共有)」の bxwsd/bxbnk は detail_ReadFileInfo で見つからない可能性
        // があるので、別途検索する (時間はかかるが仕方ない)
        u32 groupCount = GetGroupCount();
        for ( u32 i = 0; i < groupCount; i++ )
        {
            GroupInfo info;
            if ( ReadGroupInfo( GetGroupIdFromIndex( i ), &info ) )
            {
                FileInfo groupFileInfo;
                if ( detail_ReadFileInfo( info.fileId, &groupFileInfo ) )
                {
                    const void* groupFile = ut::AddOffsetToPtr(
                            m_pData,
                            groupFileInfo.offsetFromFileBlockHead +
                            m_FileReader.GetFileBlockOffset() );

                    internal::GroupFileReader reader(groupFile);
                    u32 groupItemCount = reader.GetGroupItemCount();
                    for ( u32 j = 0; j < groupItemCount; j++ )
                    {
                        internal::GroupItemLocationInfo locationInfo;
                        if ( reader.ReadGroupItemLocationInfo( &locationInfo, j ) )
                        {
                            if ( locationInfo.fileId == fileId &&
                                 locationInfo.address != NULL )
                            {
                                return locationInfo.address;
                            }
                        }
                    }
                }
            }
        }
        return NULL;
    }

    // 外部参照ファイルは対応しない
}

ut::FileStream* MemorySoundArchive::OpenStream( void* buffer, int size, u32 begin, u32 length ) const
{
    if ( m_pData == NULL ) return NULL;
    if ( size < sizeof( MemoryFileStream ) ) return NULL;
    MemoryFileStream* stream = new( buffer ) MemoryFileStream(
        ut::AddOffsetToPtr( m_pData, begin ), length );
    return stream;
}

ut::FileStream* MemorySoundArchive::OpenExtStream(
        void* buffer,
        int size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize ) const
{
    (void)buffer;
    (void)size;
    (void)extFilePath;
    (void)cacheBuffer;
    (void)cacheSize;

    NW_WARNING( false, "Cannot OpenExtStream for MemorySoundArchive\n");

    return NULL;
}

size_t MemorySoundArchive::detail_GetRequiredStreamBufferSize() const
{
    return sizeof( MemoryFileStream );
}

MemorySoundArchive::MemoryFileStream::MemoryFileStream( const void* buffer, u32 size )
: m_pBuffer( buffer ),
  m_Size( size ),
  m_Position( 0 )
{
}

void MemorySoundArchive::MemoryFileStream::Close()
{
    m_pBuffer = NULL;
    m_Size = 0;
    m_Position = 0;
}

s32 MemorySoundArchive::MemoryFileStream::Read( void* buf, u32 length )
{
    u32 readLen = ut::Min( length, m_Size - m_Position );
    std::memcpy( buf, ut::AddOffsetToPtr( m_pBuffer, m_Position ), readLen );
    return static_cast<s32>(readLen);
}

bool MemorySoundArchive::MemoryFileStream::Seek( s32 offset, u32 origin )
{
    switch( origin ) {
    case ut::FILE_STREAM_SEEK_BEGIN:
        m_Position = static_cast<u32>(offset);
        break;

    case ut::FILE_STREAM_SEEK_CURRENT:
        m_Position += offset;
        break;

    case ut::FILE_STREAM_SEEK_END:
        m_Position = m_Size - offset;
        break;

    default:
        NW_ASSERTMSG( false, "Unsupported Seek origin" );
        return false;
    }

    return true;
}

} // namespace nw::snd
} // namespace nw

