﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/snd/snd_HardwareManager.h>

#if defined( NW_PLATFORM_CAFE )
#include <cafe/axfx.h>
#elif defined( NW_PLATFORM_WIN32 )
#include <winext/cafe/axfx.h>
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
#include <winext/cafe/axfx.h>
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
#include <winext/cafe/axfx.h>
#else
#error
#endif


#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_Config.h>

#include <nw/assert.h>
#include <cstring>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {

namespace {

//
// AuxDeviceSet
//

enum AuxDeviceSet
{
    AUX_DEVICE_MAIN_AUX_A,
    AUX_DEVICE_MAIN_AUX_B,
    AUX_DEVICE_MAIN_AUX_C,
    AUX_DEVICE_DRC_AUX_A,
    AUX_DEVICE_DRC_AUX_B,
    AUX_DEVICE_DRC_AUX_C,
    AUX_DEVICE_DRC0_AUX_A = AUX_DEVICE_DRC_AUX_A,
    AUX_DEVICE_DRC0_AUX_B = AUX_DEVICE_DRC_AUX_B,
    AUX_DEVICE_DRC0_AUX_C = AUX_DEVICE_DRC_AUX_C,
    AUX_DEVICE_DRC1_AUX_A,
    AUX_DEVICE_DRC1_AUX_B,
    AUX_DEVICE_DRC1_AUX_C,
    AUX_DEVICE_COUNT,
    AUX_DEVICE_INVALID = -1
};

AuxDeviceSet GetAuxDeviceSet( AuxBus bus, OutputDevice device )
{
    NW_ASSERT_MINMAXLT(bus, 0, AUX_BUS_NUM);
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    AuxDeviceSet result = AUX_DEVICE_INVALID;
    switch ( device )
    {
    case OUTPUT_DEVICE_MAIN:
        result = static_cast<AuxDeviceSet>(AUX_DEVICE_MAIN_AUX_A + bus);
        break;
    case OUTPUT_DEVICE_DRC:
        result = static_cast<AuxDeviceSet>(AUX_DEVICE_DRC_AUX_A + bus);
        break;
    default:
        break;
    }
    return result;
}

AuxBus GetAuxBusFromAuxDeviceSet( AuxDeviceSet set )
{
    AuxBus bus = static_cast<AuxBus>(set % AUX_BUS_NUM);
    return bus;
}

OutputDevice GetDeviceFromAuxDeviceSet( AuxDeviceSet set )
{
    OutputDevice device = static_cast<OutputDevice>(set / AUX_BUS_NUM);
    return device;
}


//
// AxDeviceAndId
//
struct AxDeviceAndId
{
    AXPBDeviceType device;
    u32 id;
};

bool GetAxDeviceAndId( OutputDevice device, AxDeviceAndId& dev_id )
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    switch ( device )
    {
    case OUTPUT_DEVICE_MAIN:
        dev_id.device = AX_DEVICE_TV;
        dev_id.id = 0;
        break;
    case OUTPUT_DEVICE_DRC:
        dev_id.device = AX_DEVICE_DRC;
        dev_id.id = 0;
        break;
    default:
        NW_ASSERT(false);
        return false;
    }
    return true;
}


//
// GetAxAuxId
//
u32 GetAxAuxId( AuxBus bus )
{
    return static_cast<u32>(bus);
}

} // namespace

namespace internal {
namespace driver {

const BiquadFilterLpf     HardwareManager::BIQUAD_FILTER_LPF;
const BiquadFilterHpf     HardwareManager::BIQUAD_FILTER_HPF;
const BiquadFilterBpf512  HardwareManager::BIQUAD_FILTER_BPF_512;
const BiquadFilterBpf1024 HardwareManager::BIQUAD_FILTER_BPF_1024;
const BiquadFilterBpf2048 HardwareManager::BIQUAD_FILTER_BPF_2048;

/*---------------------------------------------------------------------------*
    Name:           HardwareManager

    Description:    コンストラクタ

    Arguments:      なし

    Returns:        なし
 *---------------------------------------------------------------------------*/
HardwareManager::HardwareManager()
: m_IsInitialized( false ),
  m_SrcType( SRC_TYPE_4TAP )
{
    m_MasterVolume.InitValue( 1.0f );
    m_VolumeForReset.InitValue( 1.0f );
    for ( int i = 0; i < OUTPUT_DEVICE_COUNT; i++ )
    {
        // 不正な値にしておく。Initialize で STEREO にセットされる
        m_OutputMode[i] = OUTPUT_MODE_NUM;
        m_EndUserOutputMode[i] = OUTPUT_MODE_NUM;
    }

    for ( int device = 0; device < OUTPUT_DEVICE_COUNT; device++ )
    {
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            m_AuxFadeVolume[ device ][ i ].InitValue( 1.0f );
            m_AuxUserVolume[ device ][ i ].InitValue( 1.0f );
            m_AuxCallback[ device ][ i ] = NULL;
            m_AuxCallbackContext[ device ][ i ] = NULL;
            // m_EffectProcessTick[ i ] = nn::os::Tick( 0 );
        }
    }
}

void HardwareManager::Initialize()
{
    if ( m_IsInitialized ) return;

    // AUX コールバック設定、既に登録されている AUX コールバックを保存する
    for ( int device = 0; device < AX_MAX_NUM_DEVICES; device++ )
    {
        for ( int busId = 0; busId < AUX_BUS_NUM; busId++ )
        {
            AXGetAuxCallback(
                static_cast<AXPBDeviceType>(device),
                0,
                GetAxAuxId(static_cast<AuxBus>(busId)),
                reinterpret_cast<AXUserAuxCallback*>(&m_AuxCallback[device][busId]),
                &m_AuxCallbackContext[device][busId]);
            AXRegisterAuxCallback( static_cast<AXPBDeviceType>(device), 0, GetAxAuxId(static_cast<AuxBus>(busId)), NULL, NULL);
        }
    }

    // biquadフィルタテーブル初期化
    for ( int i = BIQUAD_FILTER_TYPE_DATA_MIN; i < BIQUAD_FILTER_TYPE_MAX + 1; i++ )
    {
        m_BiquadFilterCallbackTable[i] = NULL;
    }
    SetBiquadFilterCallback( BIQUAD_FILTER_TYPE_LPF,     &BIQUAD_FILTER_LPF );
    SetBiquadFilterCallback( BIQUAD_FILTER_TYPE_HPF,     &BIQUAD_FILTER_HPF );
    SetBiquadFilterCallback( BIQUAD_FILTER_TYPE_BPF512,  &BIQUAD_FILTER_BPF_512 );
    SetBiquadFilterCallback( BIQUAD_FILTER_TYPE_BPF1024, &BIQUAD_FILTER_BPF_1024 );
    SetBiquadFilterCallback( BIQUAD_FILTER_TYPE_BPF2048, &BIQUAD_FILTER_BPF_2048 );

    // マスターボリューム設定
#if 0 // TODO
    nn::snd::SetMasterVolume( 1.0f );
#endif

    // DeviceFinalMixCallbackコールバック設定
    if ( ::AXGetDeviceFinalMixCallback(AX_DEVICE_TV, &m_LastMainFinalMixCallback) == AXPB_ERROR_NONE )
    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_TV, MainFinalMixCallbackFunc) == AXPB_ERROR_NONE;
        NW_WARNING(isSuccess, "unexpected result.");
    }

    if ( ::AXGetDeviceFinalMixCallback(AX_DEVICE_DRC, &m_LastDrcFinalMixCallback) == AXPB_ERROR_NONE )
    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_DRC, DrcFinalMixCallbackFunc) == AXPB_ERROR_NONE;
        NW_WARNING(isSuccess, "unexpected result.");
    }

#if defined( NW_PLATFORM_CAFE )
    // サウンド出力モード設定 (本体設定を参照する)
    // TV
    {
        u32 mode;
        AXPB_ERROR_CODE err = AXGetDeviceMode(AX_DEVICE_TV, &mode);
        NW_ASSERTMSG(err == AXPB_ERROR_NONE,
                "AXGetDeviceMode error(%s)\n", Util::GetAxErrorCodeString(err));
        OutputMode nwMode = OUTPUT_MODE_STEREO;
        switch (mode)
        {
        case AX_MODE_STEREO:
            nwMode = OUTPUT_MODE_STEREO;
            break;
        case AX_MODE_MONO:
            nwMode = OUTPUT_MODE_MONO;
            break;
        case AX_MODE_6CHAN:
            nwMode = OUTPUT_MODE_SURROUND;
            break;
        default:
            NW_ASSERT(mode == AX_MODE_STEREO
                   || mode == AX_MODE_MONO
                   || mode == AX_MODE_6CHAN);
            break;
        }
        m_EndUserOutputMode[OUTPUT_DEVICE_MAIN] = nwMode;
        SetOutputMode( nwMode, OUTPUT_DEVICE_MAIN );
    }

    // DRC
    {
        u32 mode;
        AXPB_ERROR_CODE err = AXGetDeviceMode(AX_DEVICE_DRC, &mode);
        NW_ASSERTMSG(err == AXPB_ERROR_NONE,
                "AXGetDeviceMode error(%s)\n", Util::GetAxErrorCodeString(err));
        OutputMode nwMode = OUTPUT_MODE_STEREO;
        switch (mode)
        {
        case AX_MODE_STEREO:
            nwMode = OUTPUT_MODE_STEREO;
            break;
        case AX_MODE_MONO:
            nwMode = OUTPUT_MODE_MONO;
            break;
        case AX_MODE_SURROUND:
            nwMode = OUTPUT_MODE_SURROUND;
            break;
        default:
            NW_ASSERT(mode == AX_MODE_STEREO
                   || mode == AX_MODE_MONO
                   || mode == AX_MODE_SURROUND);
            break;
        }
        m_EndUserOutputMode[OUTPUT_DEVICE_DRC] = nwMode;
        SetOutputMode( nwMode, OUTPUT_DEVICE_DRC );
    }
#else
    m_EndUserOutputMode[OUTPUT_DEVICE_MAIN] = OUTPUT_MODE_STEREO;
    m_EndUserOutputMode[OUTPUT_DEVICE_DRC] = OUTPUT_MODE_SURROUND;
    SetOutputMode( OUTPUT_MODE_STEREO, OUTPUT_DEVICE_MAIN );
    SetOutputMode( OUTPUT_MODE_SURROUND, OUTPUT_DEVICE_DRC );
#endif

    // 出力先デバイス
    for ( int i = 0; i < OUTPUT_LINE_INDEX_MAX; i++ )
    {
        m_OutputDeviceFlag[i] = ( i < OUTPUT_LINE_INDEX_RESERVED_MAX ) ? 1 << i : 0;
    }

    m_IsInitialized = true;
}

void HardwareManager::UpdateEndUserOutputMode()
{
#if defined( NW_PLATFORM_CAFE )
    // サウンド出力モード設定 (本体設定を参照する)
    // TV
    {
        u32 mode;
        AXPB_ERROR_CODE err = AXGetDeviceMode(AX_DEVICE_TV, &mode);
        NW_ASSERTMSG(err == AXPB_ERROR_NONE,
                "AXGetDeviceMode error(%s)\n", Util::GetAxErrorCodeString(err));
        OutputMode nwMode = OUTPUT_MODE_STEREO;
        switch (mode)
        {
        case AX_MODE_STEREO:
            nwMode = OUTPUT_MODE_STEREO;
            break;
        case AX_MODE_MONO:
            nwMode = OUTPUT_MODE_MONO;
            break;
        case AX_MODE_6CHAN:
            nwMode = OUTPUT_MODE_SURROUND;
            break;
        default:
            NW_ASSERT(mode == AX_MODE_STEREO
                   || mode == AX_MODE_MONO
                   || mode == AX_MODE_6CHAN);
            break;
        }
        m_EndUserOutputMode[OUTPUT_DEVICE_MAIN] = nwMode;
    }

    // DRC
    {
        u32 mode;
        AXPB_ERROR_CODE err = AXGetDeviceMode(AX_DEVICE_DRC, &mode);
        NW_ASSERTMSG(err == AXPB_ERROR_NONE,
                "AXGetDeviceMode error(%s)\n", Util::GetAxErrorCodeString(err));
        OutputMode nwMode = OUTPUT_MODE_STEREO;
        switch (mode)
        {
        case AX_MODE_STEREO:
            nwMode = OUTPUT_MODE_STEREO;
            break;
        case AX_MODE_MONO:
            nwMode = OUTPUT_MODE_MONO;
            break;
        case AX_MODE_SURROUND:
            nwMode = OUTPUT_MODE_SURROUND;
            break;
        default:
            NW_ASSERT(mode == AX_MODE_STEREO
                   || mode == AX_MODE_MONO
                   || mode == AX_MODE_SURROUND);
            break;
        }
        m_EndUserOutputMode[OUTPUT_DEVICE_DRC] = nwMode;
    }
#else
    m_EndUserOutputMode[OUTPUT_DEVICE_MAIN] = OUTPUT_MODE_STEREO;
    m_EndUserOutputMode[OUTPUT_DEVICE_DRC] = OUTPUT_MODE_SURROUND;
#endif
}

void HardwareManager::Finalize()
{
    if ( !m_IsInitialized ) return;

    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_TV, m_LastMainFinalMixCallback) == AXPB_ERROR_NONE;
        NW_WARNING(isSuccess, "unexpected result.");
        m_LastMainFinalMixCallback = NULL;
    }

    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_DRC, m_LastDrcFinalMixCallback) == AXPB_ERROR_NONE;
        NW_WARNING(isSuccess, "unexpected result.");
        m_LastDrcFinalMixCallback = NULL;
    }

    // エフェクトクリア
    for ( int busId = 0; busId < AUX_BUS_NUM; busId++ )
    {
        for ( int device = 0; device < OUTPUT_DEVICE_COUNT; device++ )
        {
            FinalizeEffect( static_cast<AuxBus>( busId ), static_cast<OutputDevice>(device) );
        }

        for ( int device = 0; device < AX_MAX_NUM_DEVICES; device++ )
        {
            AXRegisterAuxCallback(
                static_cast<AXPBDeviceType>(device),
                0,
                GetAxAuxId(static_cast<AuxBus>(busId)),
                static_cast<AXUserAuxCallback>(m_AuxCallback[device][busId]),
                &m_AuxCallbackContext[device][busId]);
            m_AuxCallback[ device ][ busId ] = NULL;
            m_AuxCallbackContext[ device ][ busId ] = NULL;
        }
    }

    m_IsInitialized = false;
}

void HardwareManager::Update()
{
    // ClearEffect のフェード処理。フェードが完了したらエフェクトをクリア
    for ( int device = 0; device < OUTPUT_DEVICE_COUNT; device++ )
    {
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            bool updateFlag = false;
            if ( ! m_AuxUserVolume[ device ][ i ].IsFinished() )
            {
                m_AuxUserVolume[ device ][ i ].Update();
                updateFlag = true;
            }
            if ( ! m_AuxFadeVolume[ device ][ i ].IsFinished() )
            {
                m_AuxFadeVolume[ device ][ i ].Update();
                if ( m_AuxFadeVolume[ device ][ i ].IsFinished() )
                {
                    FinalizeEffect( static_cast<AuxBus>( i ), static_cast<OutputDevice>(device) );
                }
                updateFlag = true;
            }

            if ( updateFlag )
            {
                f32 returnVolumeF32 = 1.0f;
                returnVolumeF32 *= ut::Clamp( m_AuxUserVolume[device][i].GetValue(), 0.0f, 1.0f );
                returnVolumeF32 *= ut::Clamp( m_AuxFadeVolume[device][i].GetValue(), 0.0f, 1.0f );
                u16 returnVolume = static_cast<u16>(returnVolumeF32 * AUX_RETURN_VOLUME_MAX);

                AxDeviceAndId devId;
                bool result = GetAxDeviceAndId(static_cast<OutputDevice>(device), devId);
                if (result)
                {
                    AXPB_ERROR_CODE error = AXSetAuxReturnVolume(
                            devId.device, devId.id, GetAxAuxId(static_cast<AuxBus>(i)), returnVolume);
                    NW_ASSERTMSG(error >= 0,
                            "[%s] AXSetAuxReturnVolume(device[%d], id[%d], bus[%d]) is failed(%s)",
                            __FUNCTION__,
                            devId.device, devId.id, GetAxAuxId(static_cast<AuxBus>(i)), Util::GetAxErrorCodeString(error));
                }
            }
        }
    }

    // 音量のアップデート
    if ( ! m_MasterVolume.IsFinished() ) {
        m_MasterVolume.Update();
        MultiVoiceManager::GetInstance().UpdateAllVoicesSync( MultiVoice::UPDATE_VE );
    }

    if ( ! m_VolumeForReset.IsFinished() ) {
        m_VolumeForReset.Update();
        MultiVoiceManager::GetInstance().UpdateAllVoicesSync( MultiVoice::UPDATE_VE );
    }
}
    // RVL は計96ch中、マージン16だった


/* ========================================================================
        リセット前準備
   ======================================================================== */

void HardwareManager::PrepareReset()
{
    // if ( m_OldAidCallback != NULL ) return;

    m_VolumeForReset.SetTarget( 0.0f, 3 ); // ３オーディオフレームかけてフェードアウト
    // m_ResetReadyCounter = -1;
    // m_OldAidCallback = ::AI_RegisterDMACallback( AiDmaCallbackFunc );
}

bool HardwareManager::IsResetReady() const
{
    return true; // m_ResetReadyCounter == 0 ? true : false;
}


void HardwareManager::SetOutputMode( OutputMode mode, OutputDevice device )
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    if ( m_OutputMode[device] == mode )
    {
        return;
    }

    m_OutputMode[device] = mode;

#if defined(NW_PLATFORM_CAFE)
    if ( device == OUTPUT_DEVICE_DRC )
    {
        /*AXDRCVSModeType*/ int type = -1;

        switch ( mode )
        {
        case OUTPUT_MODE_MONO:
        case OUTPUT_MODE_STEREO:
            type = AX_DRC_VS_OFF;
            break;
        case OUTPUT_MODE_SURROUND:
            type = AX_DRC_VS_ON_FRONT_BYPASS;
            break;
        default:
            NW_ASSERT( false );
            break;
        }
        NW_ASSERTMSG(type >= 0 && type <= AX_DRC_VS_ON_FRONT_BYPASS,
                "[%s] AXDRCVSModeType is invalid(%d)\n", __FUNCTION__, type);

        AXPB_ERROR_CODE result = AXSetDRCVSMode(static_cast<AXDRCVSModeType>(type));
        NW_ASSERTMSG(result == AXPB_ERROR_NONE, "[%s] AXSetDRCVSMode(%d) failed => [%d]\n",
                __FUNCTION__, type, result);
    }
#endif

    // コマンドを投げる
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
        command->id = DRIVER_COMMAND_ALLVOICES_SYNC;
        command->syncFlag = MultiVoice::UPDATE_MIX;
        cmdmgr.PushCommand(command);
    }

    // エフェクトの出力モード変更処理
    for ( int device = 0; device < OUTPUT_DEVICE_COUNT; device++ )
    {
        for ( int bus = 0; bus < AUX_BUS_NUM; bus++ )
        {
            FxList& list = GetEffectList( static_cast<AuxBus>( bus ), static_cast<OutputDevice>(device) );
            for ( FxList::Iterator itr = list.GetBeginIter();
                    itr != list.GetEndIter();
                    ++itr )
            {
                itr->OnChangeOutputMode();
            }
        }
    }
}

f32 HardwareManager::GetOutputVolume() const
{
    f32 volume = m_MasterVolume.GetValue();
    volume *= m_VolumeForReset.GetValue();
    return volume;
}

void HardwareManager::SetOutputDeviceFlag( u32 outputLineIndex, u8 outputDeviceFlag )
{
    if ( outputLineIndex >= OUTPUT_LINE_INDEX_RESERVED_MAX && outputLineIndex < OUTPUT_LINE_INDEX_MAX )
    {
        if ( m_OutputDeviceFlag[outputLineIndex] == outputDeviceFlag )
        {
            return;
        }
        m_OutputDeviceFlag[outputLineIndex] = outputDeviceFlag;

        // コマンドを投げる
        {
            DriverCommand& cmdmgr = DriverCommand::GetInstance();
            DriverCommandAllVoicesSync* command =
                cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
            command->id = DRIVER_COMMAND_ALLVOICES_SYNC;
            command->syncFlag = MultiVoice::UPDATE_MIX;
            cmdmgr.PushCommand(command);
        }
    }
}

void HardwareManager::SetMasterVolume( float volume, int fadeTimes )
{
    if ( volume < 0.0f ) volume = 0.0f;
    m_MasterVolume.SetTarget(
        volume,
        ( fadeTimes + HardwareManager::SOUND_FRAME_INTERVAL_MSEC - 1 )
        / SOUND_FRAME_INTERVAL_MSEC
    );

    if ( fadeTimes == 0 )
    {
        // コマンドを投げる
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
        command->id = DRIVER_COMMAND_ALLVOICES_SYNC;
        command->syncFlag = MultiVoice::UPDATE_VE;
        cmdmgr.PushCommand(command);
    }
}

/* ========================================================================
        SRC タイプ
   ======================================================================== */

/*---------------------------------------------------------------------------*
    Name:           SetSrcType

    Description:    SRC タイプを変更する

    Arguments:      type: SRC タイプ

    Returns:        なし
 *---------------------------------------------------------------------------*/
void HardwareManager::SetSrcType( SrcType type )
{
    if ( m_SrcType == type ) return;

    m_SrcType = type;

    // コマンドを投げる
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
        command->id = DRIVER_COMMAND_ALLVOICES_SYNC;
        command->syncFlag = MultiVoice::UPDATE_SRC;
        cmdmgr.PushCommand(command);
    }
}

/* ========================================================================
        エフェクト
   ======================================================================== */

bool HardwareManager::AppendEffect( AuxBus bus, FxBase* pFx, OutputDevice device )
{
    NW_NULL_ASSERT( pFx );
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    if ( ! m_AuxFadeVolume[ device ][ bus ].IsFinished() )
    {
        FinalizeEffect( bus, device );
    }

    m_AuxFadeVolume[ device ][ bus ].SetTarget( 1.0f, 0 );

    AxDeviceAndId devId;
    {
        bool result = GetAxDeviceAndId( device, devId );
        if ( result == false )
        {
            return false;
        }
        AXPB_ERROR_CODE error = AXSetAuxReturnVolume(
                devId.device, devId.id, GetAxAuxId(bus), AUX_RETURN_VOLUME_MAX );
        NW_ASSERTMSG( error >= 0,
                "[%s] AXSetAuxReturnVolume(device[%d], id[%d], bus[%d]) is failed(%s)",
                __FUNCTION__,
                devId.device, devId.id, GetAxAuxId(bus), Util::GetAxErrorCodeString(error) );
        if ( error < 0 )
        {
            return false;
        }
    }

    if ( GetEffectList( bus, device ).IsEmpty() )
    {
        AuxDeviceSet set = GetAuxDeviceSet(bus, device);

        AXPB_ERROR_CODE error = AXRegisterAuxCallback(
                devId.device, devId.id, GetAxAuxId(bus),
                AuxCallbackFunc, reinterpret_cast<void*>(set) );
        NW_ASSERTMSG( error >= 0,
                "[%s] AXRegisterAuxCallback(device[%d], id[%d], bus[%d] cb[%p]) is failed(%s)",
                __FUNCTION__,
                devId.device, devId.id, GetAxAuxId(bus),
                AuxCallbackFunc, Util::GetAxErrorCodeString(error) );
        if ( error < 0 )
        {
            return false;
        }
    }

    GetEffectList( bus, device ).PushBack( pFx );
    return true;
}

void HardwareManager::ClearEffect( AuxBus bus, int fadeTimes, OutputDevice device )
{
    NW_ASSERT_MINMAXLT(bus, 0, AUX_BUS_NUM);
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    if ( fadeTimes == 0 )
    {
        FinalizeEffect( bus, device );

        if ( m_AuxFadeVolume[ device ][ bus ].IsFinished() )
        {
            m_AuxFadeVolume[ device ][ bus ].SetTarget( 0.0f, 0 );
        }
    }
    else
    {
        m_AuxFadeVolume[ device ][ bus ].SetTarget(
            0.0f,
            ( fadeTimes + SOUND_FRAME_INTERVAL_MSEC - 1 ) / SOUND_FRAME_INTERVAL_MSEC
        );
    }
}

void HardwareManager::FinalizeEffect( AuxBus bus, OutputDevice device )
{
    NW_ASSERT_MINMAXLT(bus, 0, AUX_BUS_NUM);
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);

    FxList& list = GetEffectList( bus, device );
    if ( list.IsEmpty() )
    {
        return;
    }

    AxDeviceAndId devId;
    {
        bool result = GetAxDeviceAndId(device, devId);
        if ( result == false )
        {
            return;
        }
        AXPB_ERROR_CODE error = AXRegisterAuxCallback(
                devId.device, devId.id, GetAxAuxId(bus), NULL, NULL);
        NW_ASSERTMSG( error >= 0,
                "[%s] AXRegisterAuxCallback(device[%d], id[%d], bus[%d]) is failed(%s)",
                __FUNCTION__,
                devId.device, devId.id, GetAxAuxId(bus), Util::GetAxErrorCodeString(error) );
        if ( error < 0 )
        {
            return;
        }
    }

    m_EffectProcessTick[ device ][ bus ] = nw::ut::Tick(0);

    for ( FxList::Iterator itr = list.GetBeginIter();
          itr != list.GetEndIter();
          ++itr )
    {
        itr->Finalize();
    }
    list.Clear();
}

void HardwareManager::AuxCallbackFunc( s32** data, void* context, AXAUXCBSTRUCT* info )
{
    nw::ut::Tick tick = nw::ut::Tick::GetSystemCurrent();

    AuxDeviceSet set = static_cast<AuxDeviceSet>( reinterpret_cast<u32>(context) );

    AuxBus bus = GetAuxBusFromAuxDeviceSet(set);
    OutputDevice device = GetDeviceFromAuxDeviceSet(set);
    NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
    NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
    NW_ASSERT_NOT_NULL(info);

    void* buffer[6] = {NULL};    // 最大 6ch (TODO:あとで s32* にしたい)
    for ( u32 ch = 0; ch < info->numChs; ch++ )
    {
        buffer[ch] = reinterpret_cast<void*>(data[ch]);
    }

    const u32 BUFFER_SIZE = info->numChs * info->numSamples * sizeof(s32);
    const OutputMode OUTPUT_MODE = GetInstance().GetOutputMode(device);
    const f32 FX_SAMPLE_RATE_F32 = static_cast<f32>(FX_SAMPLE_RATE);

    for ( FxList::Iterator itr = GetInstance().GetEffectList( bus, device ).GetBeginIter();
          itr != GetInstance().GetEffectList( bus, device ).GetEndIter();
          (void)++itr )
    {
        itr->UpdateBuffer(
                info->numChs,
                buffer,
                BUFFER_SIZE,
                FX_SAMPLE_FORMAT,
                FX_SAMPLE_RATE_F32,
                OUTPUT_MODE );
    }

    // 処理時間計測
    GetInstance().m_EffectProcessTick[ device ][ bus ] = nw::ut::Tick::GetSystemCurrent() -  tick;
}

void HardwareManager::SetBiquadFilterCallback( int type, const BiquadFilterCallback* cb )
{
    NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_DATA_MIN, BIQUAD_FILTER_TYPE_MAX );

    if ( type <= BIQUAD_FILTER_TYPE_NONE /* 0 */ )
    {
        // 0 番は Biquad フィルタ OFF なので、なにもしない
        return;
    }

    m_BiquadFilterCallbackTable[ type ] = cb;
}

void HardwareManager::FlushDataCache( void* address, u32 length )
{
#if defined( NW_PLATFORM_CAFE )
    DCFlushRange(address,length);
#else
    (void)address;
    (void)length;
#endif
}

void HardwareManager::MainFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info )
{
    HardwareManager::GetInstance().OnMainFinalMixCallback( info );
}

void HardwareManager::OnMainFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info )
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    NW_NULL_ASSERT(info);
    NW_WARNING(info->numDevices == 1, "unexpected value.");

    if ( m_LastMainFinalMixCallback != NULL )
    {
        m_LastMainFinalMixCallback( info );
    }

    FinalMixData data;
    if ( info->numChnsIn > 0 )
    {
        data.frontLeft   = info->numChnsIn > 0 ? info->data[0] : NULL;
        data.frontRight  = info->numChnsIn > 1 ? info->data[1] : NULL;
        data.rearLeft    = info->numChnsIn > 2 ? info->data[2] : NULL;
        data.rearRight   = info->numChnsIn > 3 ? info->data[3] : NULL;
        data.frontCenter = info->numChnsIn > 4 ? info->data[4] : NULL;
        data.lfe         = info->numChnsIn > 5 ? info->data[5] : NULL;
    }
    data.sampleCount   = info->numSamples;
    data.channelCount  = info->numChnsIn;

    for ( FinalMixCallbackList::iterator itr = m_FinalMixCallbackList.begin();
          itr != m_FinalMixCallbackList.end();
          ++itr
        )
    {
        itr->OnFinalMix(OUTPUT_DEVICE_MAIN, &data);
    }

    for ( ReadOnlyFinalMixCallbackList::iterator itr = m_ReadOnlyFinalMixCallbackList.begin();
          itr != m_ReadOnlyFinalMixCallbackList.end();
          ++itr
        )
    {
        itr->OnFinalMix(OUTPUT_DEVICE_MAIN, &data);
    }
}

void HardwareManager::DrcFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info )
{
    HardwareManager::GetInstance().OnDrcFinalMixCallback( info );
}

void HardwareManager::OnDrcFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info )
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    const u32 NUM_DEVICES = 2;

    NW_NULL_ASSERT(info);
#ifdef NW_PLATFORM_CAFE
    NW_WARNING(info->numDevices == NUM_DEVICES, "unexpected value.");
#endif

    if ( m_LastDrcFinalMixCallback != NULL )
    {
        m_LastDrcFinalMixCallback( info );
    }

    FinalMixData data[NUM_DEVICES];
    const u16 numChns = info->numChnsIn;
    if ( info->numChnsIn > 0 )
    {
        for (s32 deviceIdx = 0; deviceIdx < info->numDevices; ++deviceIdx)
        {
            data[deviceIdx].frontLeft    = numChns > 0 ? info->data[0 + (numChns * deviceIdx)] : NULL;
            data[deviceIdx].frontRight   = numChns > 1 ? info->data[1 + (numChns * deviceIdx)] : NULL;
            data[deviceIdx].rearLeft     = numChns > 2 ? info->data[2 + (numChns * deviceIdx)] : NULL;
            data[deviceIdx].rearRight    = numChns > 3 ? info->data[3 + (numChns * deviceIdx)] : NULL;
            data[deviceIdx].sampleCount  = info->numSamples;
            data[deviceIdx].channelCount = info->numChnsIn;

            OutputDevice outputDevice = static_cast<OutputDevice>(static_cast<s32>(OUTPUT_DEVICE_DRC0) + deviceIdx);

            for ( FinalMixCallbackList::iterator itr = m_FinalMixCallbackList.begin();
                  itr != m_FinalMixCallbackList.end();
                  ++itr
                )
            {
                itr->OnFinalMix(outputDevice, &data[deviceIdx]);
            }

            for ( ReadOnlyFinalMixCallbackList::iterator itr = m_ReadOnlyFinalMixCallbackList.begin();
                  itr != m_ReadOnlyFinalMixCallbackList.end();
                  ++itr
                )
            {
                itr->OnFinalMix(outputDevice, &data[deviceIdx]);
            }
        }
    }
}

void HardwareManager::AppendFinalMixCallback( FinalMixCallback* userCallback )
{
    NW_NULL_ASSERT( userCallback );
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_FinalMixCallbackList.PushBack(userCallback);
}

void HardwareManager::PrependFinalMixCallback( FinalMixCallback* userCallback )
{
    NW_NULL_ASSERT( userCallback );
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_FinalMixCallbackList.PushFront(userCallback);
}

void HardwareManager::EraseFinalMixCallback( FinalMixCallback* userCallback )
{
    NW_NULL_ASSERT( userCallback );
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_FinalMixCallbackList.Erase(userCallback);
}

void HardwareManager::AppendReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback )
{
    NW_NULL_ASSERT( userCallback );
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_ReadOnlyFinalMixCallbackList.PushBack(userCallback);
}

void HardwareManager::EraseReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback )
{
    NW_NULL_ASSERT( userCallback );
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
    m_ReadOnlyFinalMixCallbackList.Erase(userCallback);
}

} // namespace nw::snd::internal:driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
