﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_GroupFileReader.h>

namespace nw {
namespace snd {
namespace internal {

namespace {

const u32 SIGNATURE_INFO_BLOCK_GROUP  = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
const u32 SIGNATURE_FILE_BLOCK_GROUP  = NW_UT_MAKE_SIGWORD( 'F', 'I', 'L', 'E' );
const u32 SIGNATURE_INFOEX_BLOCK_GROUP  = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'X' );

const u32 SUPPORTED_FILE_VERSION_GROUP = 0x00010000;
const u32 CURRENT_FILE_VERSION_GROUP   = 0x00010000;

bool IsValidFileHeaderGroup( const void* groupFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader* header =
        reinterpret_cast<const ut::BinaryFileHeader*>( groupFile );
#else
    const BinaryFileHeader* header =
        reinterpret_cast<const BinaryFileHeader*>( groupFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header->signature == GroupFileReader::SIGNATURE_FILE,
            "invalid file signature. group file is not available." );
    if ( header->signature != GroupFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_GROUP <= header->version) &&
                                   (header->version <= CURRENT_FILE_VERSION_GROUP) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfgrp file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_GROUP, CURRENT_FILE_VERSION_GROUP, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace

GroupFileReader::GroupFileReader( const void* groupFile )
: m_pInfoBlockBody( NULL ),
  m_pFileBlockBody( NULL ),
  m_pInfoExBlockBody( NULL )
{
    NW_NULL_ASSERT( groupFile );

    if ( ! IsValidFileHeaderGroup( groupFile ) ) return;

    const GroupFile::FileHeader* header =
        static_cast<const GroupFile::FileHeader*>( groupFile );

    const GroupFile::InfoBlock* infoBlock = header->GetInfoBlock();
    const GroupFile::FileBlock* fileBlock = header->GetFileBlock();
    const GroupFile::InfoExBlock* infoExBlock = header->GetInfoExBlock();

    if ( infoBlock == NULL ) return;
    if ( fileBlock == NULL ) return;

    NW_ASSERT( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_GROUP );
    NW_ASSERT( fileBlock->header.kind == SIGNATURE_FILE_BLOCK_GROUP );

    if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_GROUP )
    {
        return;
    }
    if ( fileBlock->header.kind != SIGNATURE_FILE_BLOCK_GROUP )
    {
        return;
    }

    // バイナリバージョンが 1.0.0.0 のときは InfoExBlock が含まれない
    if ( infoExBlock != NULL )
    {
        NW_ASSERT( infoExBlock->header.kind == SIGNATURE_INFOEX_BLOCK_GROUP );
        if ( infoExBlock->header.kind != SIGNATURE_INFOEX_BLOCK_GROUP )
        {
            return;
        }
        m_pInfoExBlockBody = &(infoExBlock->body);
    }

    m_pInfoBlockBody = &(infoBlock->body);
    m_pFileBlockBody = &(fileBlock->body);
}

bool GroupFileReader::ReadGroupItemLocationInfo(
        GroupItemLocationInfo* out, u32 index ) const
{
    if ( m_pInfoBlockBody == NULL )
    {
        return false;
    }

    const GroupFile::GroupItemInfo* groupItemInfo = m_pInfoBlockBody->GetGroupItemInfo( index );
    if ( groupItemInfo == NULL )
    {
        return false;
    }

    out->fileId = groupItemInfo->fileId;
    out->address = groupItemInfo->GetFileLocation( m_pFileBlockBody );
    return true;
}

u32 GroupFileReader::GetGroupItemExCount() const
{
    if ( m_pInfoExBlockBody == NULL )
    {
        return 0;
    }

    return m_pInfoExBlockBody->GetGroupItemInfoExCount();
}

bool GroupFileReader::ReadGroupItemInfoEx( GroupFile::GroupItemInfoEx* out, u32 index ) const
{
    if ( m_pInfoExBlockBody == NULL )
    {
        return false;
    }

    const GroupFile::GroupItemInfoEx* groupItemInfoEx =
        m_pInfoExBlockBody->GetGroupItemInfoEx( index );
    if ( groupItemInfoEx == NULL )
    {
        return false;
    }

    *out = *groupItemInfoEx; // 構造体のコピー
    return true;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

