﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FxReverbStd.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {

/* ========================================================================
        constant definition
   ======================================================================== */

const f32 FxReverbStd::PRE_DELAY_TIME_MAX_MIN = AXFX_REVSTD_EXP_MIN_PREDELAYTIMEMAX; // 0.0 sec
const f32 FxReverbStd::PRE_DELAY_TIME_MIN = AXFX_REVSTD_EXP_MIN_PREDELAYTIME; // 0.0 sec
const f32 FxReverbStd::FUSED_TIME_MIN = AXFX_REVSTD_EXP_MIN_FUSEDTIME; // 0.0 sec
const f32 FxReverbStd::COLORATION_MIN = AXFX_REVSTD_EXP_MIN_COLORATION; // 0.0
const f32 FxReverbStd::COLORATION_MAX = AXFX_REVSTD_EXP_MAX_COLORATION; // 1.0
const f32 FxReverbStd::DAMPING_MIN = AXFX_REVSTD_EXP_MIN_DAMPING; // 0.0
const f32 FxReverbStd::DAMPING_MAX = AXFX_REVSTD_EXP_MAX_DAMPING; // 1.0
const f32 FxReverbStd::EARLY_GAIN_MIN = AXFX_REVSTD_EXP_MIN_EARLYGAIN; // 0.0
const f32 FxReverbStd::EARLY_GAIN_MAX = AXFX_REVSTD_EXP_MAX_EARLYGAIN; // 1.0
const f32 FxReverbStd::FUSED_GAIN_MIN = AXFX_REVSTD_EXP_MIN_FUSEDGAIN; // 0.0
const f32 FxReverbStd::FUSED_GAIN_MAX = AXFX_REVSTD_EXP_MAX_FUSEDGAIN; // 1.0
const f32 FxReverbStd::OUT_GAIN_MIN = AXFX_REVSTD_EXP_MIN_OUTGAIN; // 0.0
const f32 FxReverbStd::OUT_GAIN_MAX = AXFX_REVSTD_EXP_MAX_OUTGAIN; // 1.0

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         FxReverbStd

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
FxReverbStd::FxReverbStd()
: m_IsActive( false )
{
    // パラメータ初期化
    const ReverbStdParam param;
    SetParam( param );
}

/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  エフェクトを使用するのに必用となるメモリサイズを取得する

  Arguments:    None.

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
u32 FxReverbStd::GetRequiredMemSize()
{
    u32 requiredSize = static_cast<u32>( AXFXReverbStdExpGetMemSize( &m_AxfxParam ) );
    size_t size = ut::RoundUp(
        sizeof( MEMiHeapHead )
        + sizeof( MEMiFrmHeapHead )
        + requiredSize
        + 32,
        32
    );

    return static_cast<u32>( size );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファのアドレス
                size - バッファのサイズ

  Returns:      割り当てに成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxReverbStd::AssignWorkBuffer( void* buffer, u32 size )
{
    return m_Impl.CreateHeap( buffer, size );
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを解放する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxReverbStd::ReleaseWorkBuffer()
{
    m_Impl.DestroyHeap();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  エフェクトの開始処理を行う

  Arguments:    None.

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxReverbStd::Initialize()
{
    if ( GetRequiredMemSize() > static_cast<u32>( m_Impl.GetHeapTotalSize()) ) return false;

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    int result = AXFXReverbStdExpInit( &m_AxfxParam );
    u32 allocatedSize = m_Impl.RestoreAlloc( alloc, free );

    // AXで実際にAllocateされたメモリと同じかどうか確認
    u32 requiredMemSize = GetRequiredMemSize();
    NW_WARNING(
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ) == requiredMemSize,
        "differ between allocated buffer size(%d) and required mem size(%d).",
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ),
        requiredMemSize
    );

    m_IsActive = true;
    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  エフェクトの終了処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxReverbStd::Finalize()
{
    if ( ! m_IsActive ) return;

    m_IsActive = false;

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFXReverbStdExpShutdown( &m_AxfxParam );
    m_Impl.RestoreAlloc( alloc, free );
}

/*---------------------------------------------------------------------------*
  Name:         SetParam

  Description:  エフェクトのパラメータを変更する

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool FxReverbStd::SetParam( const ReverbStdParam& param )
{
    NW_ASSERT( param.preDelayTime >= PRE_DELAY_TIME_MIN );
    NW_ASSERT( param.preDelayTimeMax >= PRE_DELAY_TIME_MAX_MIN );
    NW_ASSERT( param.fusedTime >= FUSED_TIME_MIN );
    NW_FMINMAX_ASSERT( param.coloration,   COLORATION_MIN,     COLORATION_MAX );
    NW_FMINMAX_ASSERT( param.damping,      DAMPING_MIN,        DAMPING_MAX );
    NW_FMINMAX_ASSERT( param.earlyGain,    EARLY_GAIN_MIN,     EARLY_GAIN_MAX );
    NW_FMINMAX_ASSERT( param.fusedGain,    FUSED_GAIN_MIN,     FUSED_GAIN_MAX );
    NW_FMINMAX_ASSERT( param.outGain,      OUT_GAIN_MIN,       OUT_GAIN_MAX );

    m_Param = param; // struct copy;

    f32 preDelayTimeMax = ut::Max( param.preDelayTimeMax, PRE_DELAY_TIME_MAX_MIN );
    bool needReAlloc = ( preDelayTimeMax != m_AxfxParam.preDelayTimeMax );

    m_AxfxParam.earlyMode  = static_cast<u32>( param.earlyMode );
    m_AxfxParam.preDelayTimeMax = preDelayTimeMax;
    m_AxfxParam.preDelayTime = ut::Clamp( param.preDelayTime, PRE_DELAY_TIME_MIN, param.preDelayTimeMax );
    m_AxfxParam.fusedMode  = static_cast<u32>( param.fusedMode );
    m_AxfxParam.fusedTime  = ut::Max( param.fusedTime, FUSED_TIME_MIN );
    m_AxfxParam.coloration = ut::Clamp( param.coloration,   COLORATION_MIN,     COLORATION_MAX );
    m_AxfxParam.damping    = ut::Clamp( param.damping,      DAMPING_MIN,        DAMPING_MAX );
    m_AxfxParam.earlyGain  = ut::Clamp( param.earlyGain,    EARLY_GAIN_MIN,     EARLY_GAIN_MAX );
    m_AxfxParam.fusedGain  = ut::Clamp( param.fusedGain,    FUSED_GAIN_MIN,     FUSED_GAIN_MAX );
    m_AxfxParam.outGain    = ut::Clamp( param.outGain,      OUT_GAIN_MIN,       OUT_GAIN_MAX );

    m_AxfxParam.busIn      = NULL;
    m_AxfxParam.busOut     = NULL;
    m_AxfxParam.sendGain   = 0.0f;

    if ( ! m_IsActive ) return true;

    if ( GetRequiredMemSize() > static_cast<u32>(m_Impl.GetHeapTotalSize()) ) return false;

    int result;
    if ( needReAlloc )
    {
        AXFXAlloc alloc;
        AXFXFree free;
        m_Impl.HookAlloc( &alloc, &free );
        result = AXFXReverbStdExpSettings( &m_AxfxParam );
        m_Impl.RestoreAlloc( alloc, free );
    }
    else
    {
        result = AXFXReverbStdExpSettingsUpdate( &m_AxfxParam );
    }

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         GetParam

  Description:  エフェクトのパラメータを取得する

  Arguments:    None.

  Returns:      エフェクトパラメータ構造体
 *---------------------------------------------------------------------------*/
const FxReverbStd::ReverbStdParam& FxReverbStd::GetParam() const
{
    return m_Param;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  エフェクトコールバック

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxReverbStd::UpdateBuffer(
    int numChannels,
    void* buffer[],
    unsigned long bufferSize,
    SampleFormat format,
    f32 sampleRate,
    OutputMode mode
)
{
    if ( ! m_IsActive ) return;

    // DPL2モードの時はバスの解釈が違うので処理しない
    if ( mode == OUTPUT_MODE_DPL2 ) return;

    (void)bufferSize;
    (void)numChannels;
    (void)format;
    (void)sampleRate;

    NW_ASSERT( numChannels >= 3 );
    NW_ASSERT( format == SAMPLE_FORMAT_PCM_S32 );

    AXFX_BUFFERUPDATE axfxbuf =
    {
        static_cast<s32*>( buffer[0] ),
        static_cast<s32*>( buffer[1] ),
        static_cast<s32*>( buffer[2] )
    };
    AXFXReverbStdExpCallback( &axfxbuf, &m_AxfxParam );
}

} // namespace nw::snd
} // namespace nw
