﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FxChorus.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {

/* ========================================================================
        constant definition
   ======================================================================== */

const f32 FxChorus::DELAY_TIME_MIN = AXFX_CHORUS_EXP_MIN_DELAYTIME; // 0.1f ms
const f32 FxChorus::DELAY_TIME_MAX = AXFX_CHORUS_EXP_MAX_DELAYTIME; // 50.0f ms
const f32 FxChorus::DEPTH_MIN      = AXFX_CHORUS_EXP_MIN_DEAPTH; // 0.0f
const f32 FxChorus::DEPTH_MAX      = AXFX_CHORUS_EXP_MAX_DEAPTH; // 1.0f
const f32 FxChorus::RATE_MIN       = AXFX_CHORUS_EXP_MIN_RATE; // 0.1f
const f32 FxChorus::RATE_MAX       = AXFX_CHORUS_EXP_MAX_RATE; // 2.0f
const f32 FxChorus::FEEDBACK_MIN   = AXFX_CHORUS_EXP_MIN_FEEDBACK; // 0.0f
const f32 FxChorus::FEEDBACK_MAX   = AXFX_CHORUS_EXP_MAX_FEEDBACK; // 0.99f ( feedback < 1.0f )
const f32 FxChorus::OUT_GAIN_MIN   = AXFX_CHORUS_EXP_MIN_OUTGAIN; // 0.0f
const f32 FxChorus::OUT_GAIN_MAX   = AXFX_CHORUS_EXP_MAX_OUTGAIN; // 1.0f

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         FxChorus

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
FxChorus::FxChorus()
: m_IsActive( false )
{
    // パラメータ初期化
    const ChorusParam param;
    SetParam( param );
}
/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  エフェクトを使用するのに必用となるメモリサイズを取得する

  Arguments:    None.

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
u32 FxChorus::GetRequiredMemSize()
{
    u32 requiredSize =  static_cast<u32>( AXFXChorusExpGetMemSize( &m_AxfxParam ) );
    size_t size = ut::RoundUp(
        sizeof( MEMiHeapHead )
        + sizeof( MEMiFrmHeapHead )
        + requiredSize
        + 32,
        32
    );

    return static_cast<u32>( size );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファのアドレス
                size - バッファのサイズ

  Returns:      割り当てに成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxChorus::AssignWorkBuffer( void* buffer, u32 size )
{
    return m_Impl.CreateHeap( buffer, size );
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを解放する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxChorus::ReleaseWorkBuffer()
{
    m_Impl.DestroyHeap();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  エフェクトの開始処理を行う

  Arguments:    None.

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxChorus::Initialize()
{
    if ( static_cast<s32>(GetRequiredMemSize()) > m_Impl.GetHeapTotalSize() )
    {
        return false;
    }

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    int result = AXFXChorusExpInit( &m_AxfxParam );
    u32 allocatedSize = m_Impl.RestoreAlloc( alloc, free );

    // AXで実際にAllocateされたメモリと同じかどうか確認
    u32 requiredMemSize = GetRequiredMemSize();
    NW_WARNING(
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ) == requiredMemSize,
        "differ between allocated buffer size(%d) and required mem size(%d).",
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ),
        requiredMemSize
    );

    m_IsActive = true;
    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  エフェクトの終了処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxChorus::Finalize()
{
    if ( ! m_IsActive ) return;

    m_IsActive = false;

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFXChorusExpShutdown( &m_AxfxParam );
    m_Impl.RestoreAlloc( alloc, free );
}

/*---------------------------------------------------------------------------*
  Name:         SetParam

  Description:  エフェクトのパラメータを変更する

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool FxChorus::SetParam( const ChorusParam& param )
{
    NW_FMINMAX_ASSERT( param.delayTime, DELAY_TIME_MIN, DELAY_TIME_MAX );
    NW_FMINMAX_ASSERT( param.depth,     DEPTH_MIN,      DEPTH_MAX      );
    NW_FMINMAX_ASSERT( param.rate,      RATE_MIN,       RATE_MAX       );
    NW_FMINMAX_ASSERT( param.feedback,  FEEDBACK_MIN,   FEEDBACK_MAX   );
    NW_FMINMAX_ASSERT( param.outGain,   OUT_GAIN_MIN,   OUT_GAIN_MAX   );

    m_Param = param; // struct copy

    m_AxfxParam.delayTime = ut::Clamp( param.delayTime, DELAY_TIME_MIN, DELAY_TIME_MAX );
    m_AxfxParam.depth     = ut::Clamp( param.depth,     DEPTH_MIN,      DEPTH_MAX      );
    m_AxfxParam.rate      = ut::Clamp( param.rate,      RATE_MIN,       RATE_MAX       );
    m_AxfxParam.feedback  = ut::Clamp( param.feedback,  FEEDBACK_MIN,   FEEDBACK_MAX   );
    m_AxfxParam.outGain   = ut::Clamp( param.outGain,   OUT_GAIN_MIN,   OUT_GAIN_MAX   );

    m_AxfxParam.busIn     = NULL;
    m_AxfxParam.busOut    = NULL;
    m_AxfxParam.sendGain  = 0.0f;

    if ( ! m_IsActive ) return true;

    if ( static_cast<s32>(GetRequiredMemSize()) > m_Impl.GetHeapTotalSize() )
    {
        return false;
    }

    int result = AXFXChorusExpSettingsUpdate( &m_AxfxParam );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         GetParam

  Description:  現在のエフェクトのパラメータを取得する

  Arguments:    None.

  Returns:      エフェクトパラメータ
 *---------------------------------------------------------------------------*/
const FxChorus::ChorusParam& FxChorus::GetParam() const
{
    return m_Param;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  エフェクトコールバック

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxChorus::UpdateBuffer(
    int numChannels,
    void* buffer[],
    unsigned long bufferSize,
    SampleFormat format,
    f32 sampleRate,
    OutputMode mode
)
{
    if ( ! m_IsActive ) return;

    // DPL2モードの時はバスの解釈が違うので処理しない
    if ( mode == OUTPUT_MODE_DPL2 ) return;

    (void)bufferSize;
    (void)numChannels;
    (void)format;
    (void)sampleRate;

    NW_ASSERT( numChannels >= 3 );
    NW_ASSERT( format == SAMPLE_FORMAT_PCM_S32 );

    AXFX_BUFFERUPDATE axfxbuf =
    {
        static_cast<s32*>( buffer[0] ),
        static_cast<s32*>( buffer[1] ),
        static_cast<s32*>( buffer[2] )
    };
    AXFXChorusExpCallback( &axfxbuf, &m_AxfxParam );
}

} // namespace nw::snd
} // namespace nw
