﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Fx2Pitchshift.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

#if defined(NW_SND_CONFIG_ENABLE_SOUND2)

namespace nw {
namespace snd {

/* ========================================================================
        constant definition
   ======================================================================== */

const f32 Fx2Pitchshift::PITCH_SHIFT_MIN = -24.0f;
const f32 Fx2Pitchshift::PITCH_SHIFT_MAX =  24.0f;
const f32 Fx2Pitchshift::OUT_GAIN_MIN    =   0.0f;
const f32 Fx2Pitchshift::OUT_GAIN_MAX    =   2.0f;
const f32 Fx2Pitchshift::DRY_GAIN_MIN    =   0.0f;
const f32 Fx2Pitchshift::DRY_GAIN_MAX    =   2.0f;

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         Fx2Pitchshift

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Fx2Pitchshift::Fx2Pitchshift()
: m_IsActive( false ),
  m_ChannelMode( CHANNEL_MODE_6CH ),
  m_SampleRate( SAMPLE_RATE_32000 )
{
    // パラメータ初期化
    const PitchshiftParam param;
    SetParam( param );
}
/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  エフェクトを使用するのに必用となるメモリサイズを取得する

  Arguments:    None.

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
u32 Fx2Pitchshift::GetRequiredMemSize()
{
    AXFX_PITCHSHIFT_MODE mode = AXFX_PITCHSHIFT_MODE_6CH;

    switch ( m_ChannelMode )
    {
    case CHANNEL_MODE_2CH:
        mode = AXFX_PITCHSHIFT_MODE_2CH;
        break;
    case CHANNEL_MODE_4CH:
        mode = AXFX_PITCHSHIFT_MODE_4CH;
        break;
    case CHANNEL_MODE_6CH:
        mode = AXFX_PITCHSHIFT_MODE_6CH;
        break;
    default:
        mode = AXFX_PITCHSHIFT_MODE_6CH;
    }

    AXFX_SAMPLE_RATE rate = GetFxSampleRate( m_SampleRate );
    s32 requiredSize = AXFX2PitchshiftGetMemSize( mode, rate );
    NW_ASSERT( requiredSize >= 0 );
    size_t size = ut::RoundUp(
        sizeof( MEMiHeapHead )
        + sizeof( MEMiFrmHeapHead )
        + requiredSize
        + 32,
        32
    );

    return static_cast<u32>( size );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファのアドレス
                size - バッファのサイズ

  Returns:      割り当てに成功したらtrue
 *---------------------------------------------------------------------------*/
bool Fx2Pitchshift::AssignWorkBuffer( void* buffer, u32 size )
{
    return m_Impl.CreateHeap( buffer, size );
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを解放する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Fx2Pitchshift::ReleaseWorkBuffer()
{
    m_Impl.DestroyHeap();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  エフェクトの開始処理を行う

  Arguments:    None.

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool Fx2Pitchshift::Initialize()
{
    if ( static_cast<s32>(GetRequiredMemSize()) > m_Impl.GetHeapTotalSize() )
    {
        return false;
    }

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFX_PITCHSHIFT_MODE mode;
    switch ( m_ChannelMode )
    {
    case CHANNEL_MODE_2CH:
        mode = AXFX_PITCHSHIFT_MODE_2CH;
        break;
    case CHANNEL_MODE_4CH:
        mode = AXFX_PITCHSHIFT_MODE_4CH;
        break;
    case CHANNEL_MODE_6CH:
        mode = AXFX_PITCHSHIFT_MODE_6CH;
        break;
    default:
        mode = AXFX_PITCHSHIFT_MODE_6CH;
    }

    AXFX_SAMPLE_RATE rate = GetFxSampleRate( m_SampleRate );
    int result = AXFX2PitchshiftInit( &m_AxfxParam, mode, rate );
    u32 allocatedSize = m_Impl.RestoreAlloc( alloc, free );
    m_IsActive = true;

    // AXで実際にAllocateされたメモリと同じかどうか確認
    u32 requiredMemSize = GetRequiredMemSize();
    NW_WARNING(
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ) == requiredMemSize,
        "differ between allocated buffer size(%d) and required mem size(%d).",
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ),
        requiredMemSize
    );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  エフェクトの終了処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Fx2Pitchshift::Finalize()
{
    if ( ! m_IsActive ) return;

    m_IsActive = false;

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFX2PitchshiftShutdown( &m_AxfxParam );
    m_Impl.RestoreAlloc( alloc, free );
}

/*---------------------------------------------------------------------------*
  Name:         SetParam

  Description:  エフェクトのパラメータを変更する

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool Fx2Pitchshift::SetParam( const PitchshiftParam& param )
{
    NW_FMINMAX_ASSERT( param.pitchShift, PITCH_SHIFT_MIN, PITCH_SHIFT_MAX );
    NW_FMINMAX_ASSERT( param.outGain,    OUT_GAIN_MIN,    OUT_GAIN_MAX );
    NW_FMINMAX_ASSERT( param.dryGain,    DRY_GAIN_MIN,    DRY_GAIN_MAX    );

    m_Param = param; // struct copy

    m_AxfxParam.pitch_shift = ut::Clamp( param.pitchShift, PITCH_SHIFT_MIN, PITCH_SHIFT_MAX );
    m_AxfxParam.out_gain    = ut::Clamp( param.outGain, OUT_GAIN_MIN, OUT_GAIN_MAX );
    m_AxfxParam.dry_gain    = ut::Clamp( param.dryGain,    DRY_GAIN_MIN,    DRY_GAIN_MAX    );

    if ( ! m_IsActive ) return true;

    if ( static_cast<s32>(GetRequiredMemSize()) > m_Impl.GetHeapTotalSize() )
    {
        return false;
    }

    int result = AXFX2PitchshiftSettingsUpdate( &m_AxfxParam );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  エフェクトコールバック

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Fx2Pitchshift::UpdateBuffer(
    int numChannels,
    void* buffer[],
    unsigned long bufferSize,
    SampleFormat format,
    f32 sampleRate,
    OutputMode mode
)
{
    if ( ! m_IsActive ) return;

    // DPL2モードの時はバスの解釈が違うので処理しない
    if ( mode == OUTPUT_MODE_DPL2 ) return;

    (void)bufferSize;
    (void)numChannels;
    (void)format;
    (void)sampleRate;

    // 6ch, 48kHz 対応の修正が必要
    NW_ASSERT( numChannels >= 3 );
    NW_ASSERT( format == SAMPLE_FORMAT_PCM_S32 );

    AXFX_6CH_BUFFERUPDATE axfxbuf =
    {
        static_cast<s32*>( buffer[0] ),
        static_cast<s32*>( buffer[1] ),
        static_cast<s32*>( buffer[2] ),
        static_cast<s32*>( buffer[3] ),
        static_cast<s32*>( buffer[4] ),
        static_cast<s32*>( buffer[5] )
    };
    AXAUXCBSTRUCT auxCbStruct;

    auxCbStruct.numChs = numChannels;
    auxCbStruct.numSamples = bufferSize / ( numChannels * sizeof(s32) );
    AXFX2PitchshiftCallback( &axfxbuf, &m_AxfxParam, &auxCbStruct );
}

/*---------------------------------------------------------------------------*
  Name:         IsValidChannelNum

  Description:  デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。

  Arguments:    device チェック対象となるデバイスです。

  Returns:      チャンネル数が妥当であれば true を、妥当でなければ false を返します。
 *---------------------------------------------------------------------------*/
bool Fx2Pitchshift::IsValidChannelNum( OutputDevice device )
{
    if ( device == OUTPUT_DEVICE_DRC0 || device == OUTPUT_DEVICE_DRC1 )
    {
        if ( m_ChannelMode == CHANNEL_MODE_6CH )
        {
            NW_ASSERTMSG( false, "CHANNEL_MODE_6CH (OUTPUT_DEVICE_DRC) is inValid. Please use lower channel mode." );
            return false;
        }
    }

    return true;
}

} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_SOUND2
