﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FsSoundArchive.h>
#include <nw/snd/snd_Util.h>
#include <nw/ut.h>

#include <nw/config.h>
#if defined(NW_PLATFORM_CAFE)
#elif defined(NW_PLATFORM_WIN32)
    #include <Windows.h>
    #include "Shlwapi.h"
#endif

#if defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
#include <unistd.h>
#endif

// #define NW_SND_DEBUG_PRINT_ENABLE

namespace nw {
namespace snd {

#if defined(NW_PLATFORM_ANDROID)
const char* const FsSoundArchive::ANDROID_ASSETS_PATH_PREFIX = nw::snd::internal::FsFileStream::ANDROID_ASSETS_PATH_PREFIX;
#endif

namespace
{
const int FS_CMD_BLOCK_COUNT = 2;
}
#if defined(NW_PLATFORM_CAFE)
size_t FsSoundArchive::GetRequiredMemSize() const
{
    size_t result = sizeof(FSCmdBlock) * FS_CMD_BLOCK_COUNT;
    return result;
}
#endif

/*---------------------------------------------------------------------------*
  Name:         FsSoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
FsSoundArchive::FsSoundArchive()
: m_IsOpened( false ),
  m_FileAccessMode( FILE_ACCESS_ALWAYS ),
  m_FileAccessCount( 0 )
#if defined(NW_PLATFORM_CAFE)
    ,  m_FsPriority( FS_PRIORITY_DEFAULT )
    ,  m_FsStreamPriority( FS_STREAM_PRIORITY_DEFAULT )
#endif
{
}

/*---------------------------------------------------------------------------*
  Name:         ~FsSoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
FsSoundArchive::~FsSoundArchive()
{
    Close();
}

#if defined(NW_PLATFORM_CAFE)
bool FsSoundArchive::Open( FSClient* client, const char* path, void* buffer, size_t bufferSize, u8 fsPriority, u8 fsStreamPriority )
{
    CreateFsCommandBlockBufferPool(buffer, bufferSize);
    internal::FsFileStream::FsCommandBlockPool* fsCmdBlockPool = detail_GetFsCommandBlockPool();
    NW_ASSERT_NOT_NULL(fsCmdBlockPool);

    SetFsClient(client);

    m_FsPriority = fsPriority;
    m_FsStreamPriority = fsStreamPriority;

    m_FileStream.SetPriority(m_FsPriority);

    bool result = m_FileStream.Open( client, path, "r", fsCmdBlockPool );
    if ( result == false )
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    result = LoadFileHeader();
    if ( result == false )
    {
        NW_WARNING( false, "Cannot load header\n" );
        return false;
    }

    FileAccessEnd();

    // ロジックのメモ
    //  if ( path.isFullPath )  // フルパスかどうかは path[0] が "/" かどうかで判断
    //  {
    //      SetExternalFileRoot( path.dir );
    //  }
    //  else
    //  {
    //      SetExternalFileRoot( CurrentDir + path.dir );
    //  }

    bool isFullPath = false;
    if ( path[0] == '/' || path[0] == '\\' )
    {
        isFullPath = true;
    }

    char pathDir[FILE_PATH_MAX];
    {
        pathDir[0] = '\0';
        int pathLen = std::strlen( path );
        NW_ASSERT( pathLen < FILE_PATH_MAX );
        if ( pathLen >= FILE_PATH_MAX )
        {
            NW_WARNING( false, "pathLen(%d) >= FILE_PATH_MAX(%d)", pathLen, FILE_PATH_MAX );
            return false;
        }

        for ( int i = std::strlen( path ) - 1; i >= 0; i-- )
        {
            const char ch = path[ i ] ;
            if ( ch == '/' || ch == '\\' )
            {
                nw::ut::strncpy( pathDir, FILE_PATH_MAX, path, i );
                pathDir[i] = '\0';
                break;
            }
        }
    }

    FSCmdBlock* block = fsCmdBlockPool->Alloc();
    if ( isFullPath )
    {
        SetExternalFileRoot( pathDir );
    }
    else
    {
        // フルパスでカレントディレクトリを取得
        char currentDir[FILE_PATH_MAX];
        FSStatus status = FSGetCwd( client, block, currentDir, FILE_PATH_MAX, internal::Util::GetRetFlag() );
        if ( status != FS_STATUS_OK )
        {
#if !defined(NW_RELEASE)
            NW_WARNING( status == FS_STATUS_OK, "CurrentDir cannot get(%s)\n", internal::Util::FSStatusToString(status) );
#endif
            fsCmdBlockPool->Free(block);
            return false;
        }

        // currentDir <= currentDir + pathDir
        nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
        SetExternalFileRoot( currentDir );
    }

    // サウンドアーカイブのパスをフルパスで保存
    {
        FSStatus status = FSGetCwd( client, block, m_SoundArchiveFullPath, FILE_PATH_MAX,
                internal::Util::GetRetFlag());
        if ( status != FS_STATUS_OK )
        {
#if !defined(NW_RELEASE)
            NW_WARNING( status == FS_STATUS_OK, "CurrentDir cannot get(%s)\n", internal::Util::FSStatusToString(status) );
#endif
            fsCmdBlockPool->Free(block);
            return false;
        }
        nw::ut::strncat( m_SoundArchiveFullPath, FILE_PATH_MAX, path, std::strlen(path));
    }

    fsCmdBlockPool->Free(block);
    return true;
}
#elif defined(NW_USE_NINTENDO_SDK)
bool FsSoundArchive::Open( const char* path )
{
    NW_ASSERT_NOT_NULL(path);

    bool result = m_FileStream.Open( path, internal::fnd::File::ACCESS_MODE_READ );
    if ( result == false )
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    result = LoadFileHeader();
    if ( result == false )
    {
        NW_WARNING( false, "Cannot load header\n" );
        return false;
    }

    FileAccessEnd();

    char pathDir[FILE_PATH_MAX];
    {
        pathDir[0] = '\0';
        int pathLen = std::strlen( path );
        NW_ASSERT( pathLen < FILE_PATH_MAX );
        if ( pathLen >= FILE_PATH_MAX )
        {
            NW_WARNING( false, "pathLen(%d) >= FILE_PATH_MAX(%d)", pathLen, FILE_PATH_MAX );
            return false;
        }

        for ( int i = std::strlen( path ) - 1; i >= 0; i-- )
        {
            const char ch = path[ i ] ;
            if ( ch == '/' || ch == '\\' )
            {
                nw::ut::strncpy( pathDir, FILE_PATH_MAX, path, i );
                pathDir[i] = '\0';
                break;
            }
        }
    }

    SetExternalFileRoot( pathDir );

    // サウンドアーカイブのパスを保存
    {
        nw::ut::snprintf(m_SoundArchiveFullPath, FILE_PATH_MAX, "%s/%s", path);
    }

    return true;
}
#elif defined(NW_PLATFORM_ANDROID)
bool FsSoundArchive::Open(const char* path)
{
    NW_ASSERT_NOT_NULL(path);

    bool result = m_FileStream.Open(path, internal::fnd::File::ACCESS_MODE_READ);
    if (result == false)
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    result = LoadFileHeader();
    if (result == false)
    {
        NW_WARNING(false, "Cannot load header\n");
        return false;
    }

    FileAccessEnd();

    char pathDir[FILE_PATH_MAX];
    {
        pathDir[0] = '\0';
        int pathLen = std::strlen(path);
        NW_ASSERT(pathLen < FILE_PATH_MAX);
        if (pathLen >= FILE_PATH_MAX)
        {
            NW_WARNING(false, "pathLen(%d) >= FILE_PATH_MAX(%d)", pathLen, FILE_PATH_MAX);
            return false;
        }

        for (int i = std::strlen(path) - 1; i >= 0; i--)
        {
            const char ch = path[i];
            if (ch == '/' || ch == '\\')
            {
                nw::ut::strncpy(pathDir, FILE_PATH_MAX, path, i);
                pathDir[i] = '\0';
                break;
            }
        }
    }

    // 【/】から始まる時はフルパス
    //  Assets へアクセスするときもフルパス
    if (pathDir[0] == '/' || strncmp(pathDir, ANDROID_ASSETS_PATH_PREFIX, strlen(ANDROID_ASSETS_PATH_PREFIX)) == 0)
    {
        SetExternalFileRoot(pathDir);
    }
    else
    {
        // フルパスでカレントディレクトリを取得
        char currentDir[FILE_PATH_MAX];

        char* result = getcwd(currentDir, FILE_PATH_MAX);
        if(result == NULL)
        {
            NW_LOG("getcwd cannot get path");
            return false;
        }

        // currentDir <= currentDir + pathDir
        nw::ut::strncat(currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir));
        SetExternalFileRoot(currentDir);
    }

    // サウンドアーカイブのパスをフルパスで保存
    {
        if(strncmp(pathDir, ANDROID_ASSETS_PATH_PREFIX, sizeof(ANDROID_ASSETS_PATH_PREFIX)) == 0)
        {
            strcpy(m_SoundArchiveFullPath, path);
        }
        else
        {
            char* result = getcwd(m_SoundArchiveFullPath, FILE_PATH_MAX);

            if(result == NULL)
            {
                NW_LOG("getcwd cannot get path");
                return false;
            }
            nw::ut::snprintf(m_SoundArchiveFullPath, FILE_PATH_MAX, "%s/%s", m_SoundArchiveFullPath, path);
        }

    }

    return true;
}
#elif defined(NW_PLATFORM_IOS)
bool FsSoundArchive::Open(const char* path)
{
    NW_ASSERT_NOT_NULL(path);

    bool result = m_FileStream.Open(path, internal::fnd::File::ACCESS_MODE_READ);
    if (result == false)
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    result = LoadFileHeader();
    if (result == false)
    {
        NW_WARNING(false, "Cannot load header\n");
        return false;
    }

    FileAccessEnd();

    char pathDir[FILE_PATH_MAX];
    {
        pathDir[0] = '\0';
        int pathLen = std::strlen(path);
        NW_ASSERT(pathLen < FILE_PATH_MAX);
        if (pathLen >= FILE_PATH_MAX)
        {
            NW_WARNING(false, "pathLen(%d) >= FILE_PATH_MAX(%d)", pathLen, FILE_PATH_MAX);
            return false;
        }

        for (int i = std::strlen(path) - 1; i >= 0; i--)
        {
            const char ch = path[i];
            if (ch == '/' || ch == '\\')
            {
                nw::ut::strncpy(pathDir, FILE_PATH_MAX, path, i);
                pathDir[i] = '\0';
                break;
            }
        }
    }

    // 【/】から始まる時はフルパス
    if (pathDir[0] == '/')
    {
        SetExternalFileRoot(pathDir);
    }
    else
    {
        // フルパスでカレントディレクトリを取得
        char currentDir[FILE_PATH_MAX];

        char* result = getcwd(currentDir, FILE_PATH_MAX);
        if(result == NULL)
        {
            NW_LOG("getcwd cannot get path");
            return false;
        }

        // currentDir <= currentDir + pathDir
        nw::ut::strncat(currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir));
        SetExternalFileRoot(currentDir);
    }

    // サウンドアーカイブのパスをフルパスで保存
    {
        char* result = getcwd(m_SoundArchiveFullPath, FILE_PATH_MAX);

        if(result == NULL)
        {
            NW_LOG("getcwd cannot get path");
            return false;
        }
        nw::ut::snprintf(m_SoundArchiveFullPath, FILE_PATH_MAX, "%s/%s", m_SoundArchiveFullPath, path);
    }

    return true;
}
#else
bool FsSoundArchive::Open( const char* path )
{
    NW_ASSERT_NOT_NULL(path);

    bool result = m_FileStream.Open( path, internal::fnd::File::ACCESS_MODE_READ );
    if ( result == false )
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    result = LoadFileHeader();
    if ( result == false )
    {
        NW_WARNING( false, "Cannot load header\n" );
        return false;
    }

    FileAccessEnd();

    char pathDir[FILE_PATH_MAX];
    {
        pathDir[0] = '\0';
        int pathLen = std::strlen( path );
        NW_ASSERT( pathLen < FILE_PATH_MAX );
        if ( pathLen >= FILE_PATH_MAX )
        {
            NW_WARNING( false, "pathLen(%d) >= FILE_PATH_MAX(%d)", pathLen, FILE_PATH_MAX );
            return false;
        }

        for ( int i = std::strlen( path ) - 1; i >= 0; i-- )
        {
            const char ch = path[ i ] ;
            if ( ch == '/' || ch == '\\' )
            {
                nw::ut::strncpy( pathDir, FILE_PATH_MAX, path, i );
                pathDir[i] = '\0';
                break;
            }
        }
    }

    if ( ! PathIsRelativeA(pathDir) )
    {
        SetExternalFileRoot( pathDir );
    }
    else
    {
        // フルパスでカレントディレクトリを取得
        char currentDir[FILE_PATH_MAX];
        DWORD result = GetCurrentDirectoryA(FILE_PATH_MAX, currentDir);
        if ( result == 0 || result > FILE_PATH_MAX )
        {
            NW_LOG("CurrentDir cannot get(%d)\n", result );
            return false;
        }

        if(currentDir[result - 1] != '\\')
        {
            if ( result + 1 > FILE_PATH_MAX )
            {
                NW_LOG("current directory is too long.(%s)\n", currentDir );
                return false;
            }

            currentDir[result + 1] = '\0';
            currentDir[result] = '\\';
        }

        // currentDir <= currentDir + pathDir
        nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
        SetExternalFileRoot( currentDir );
    }

    // サウンドアーカイブのパスをフルパスで保存
    {

        DWORD result = GetCurrentDirectoryA(FILE_PATH_MAX, m_SoundArchiveFullPath);
        if ( result == 0 || result > FILE_PATH_MAX )
        {
            NW_LOG("CurrentDir cannot get(%d)\n", result);
            return false;
        }
        // GetCurrentDirectoryA は末尾に "/" が付かないので、下記でつける
        nw::ut::snprintf(m_SoundArchiveFullPath, FILE_PATH_MAX, "%s/%s", m_SoundArchiveFullPath, path);
    }

    return true;
}
#endif

/*---------------------------------------------------------------------------*
  Name:         Close

  Description:  サウンドアーカイブを閉じる

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FsSoundArchive::Close()
{
    FileAccessBegin();

    if ( m_IsOpened )
    {
        m_FileStream.Close();
    #if defined(NW_PLATFORM_CAFE)
        SetFsClient(NULL);
        DestroyFsCommandBlockBufferPool();
    #endif
        m_IsOpened = false;
    }

    Finalize();
}

ut::FileStream*
FsSoundArchive::OpenStream( void* buffer, int size, u32 begin, u32 length ) const
{
    if ( ! m_IsOpened ) return NULL;
    if ( size < sizeof( FsFileStream ) ) return NULL;
    FsFileStream* stream = new( buffer ) FsFileStream( &m_FileStream, begin, length );
    return stream;
}

ut::FileStream*
FsSoundArchive::OpenExtStream(
        void* buffer,
        int size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize ) const
{
    if ( ! m_IsOpened ) return NULL;
    if ( size < sizeof( CachedFsFileStream ) ) return NULL;

#if defined(NW_PLATFORM_CAFE)
    CachedFsFileStream* stream = new( buffer )
        CachedFsFileStream( detail_GetFsClient(), extFilePath, detail_GetFsCommandBlockPool() );
#else
    CachedFsFileStream* stream = new( buffer )
        CachedFsFileStream( extFilePath );
#endif

    if ( stream )
    {
        if ( !stream->IsAvailable() )
        {
            return NULL;
        }

        if ((cacheBuffer != NULL) && (cacheSize > 0))
        {
            stream->SetCacheBuffer(cacheBuffer, cacheSize);
        }
    }
    return stream;
}

size_t FsSoundArchive::detail_GetRequiredStreamBufferSize() const
{
    return sizeof( FsFileStream );
}

/*---------------------------------------------------------------------------*
  Name:         LoadFileHeader

  Description:  サウンドアーカイブファイルのヘッダをロードする

  Arguments:    None.

  Returns:      成功したら true 失敗したら false
 *---------------------------------------------------------------------------*/
bool FsSoundArchive::LoadFileHeader()
{
    NW_ASSERT( m_IsOpened );

    const int ALIGN = 256;

    const unsigned long headerAlignSize = static_cast<unsigned long>(
        ut::RoundUp( sizeof(internal::SoundArchiveFile::FileHeader), ALIGN )
    );
    u8 headerArea[ sizeof(internal::SoundArchiveFile::FileHeader) + ALIGN*2 ];
    void* file = ut::RoundUp( headerArea, ALIGN );

#if defined(NW_PLATFORM_CAFE)
    m_FileStream.SetPriority(m_FsPriority);
#endif
    s32 readSize = m_FileStream.Read( file, static_cast<s32>(headerAlignSize) );
    if ( readSize != static_cast<s32>(headerAlignSize) )
    {
        NW_WARNING(
            false,
            "FsSoundArchive::LoadFileHeader cannot read file.\n address(%08x) readSize(%d) != headerAlignSize(%d)\n",
            file, readSize, headerAlignSize
        );
        return false;
    }

    m_ArchiveReader.Initialize( file );
    Initialize( &m_ArchiveReader );

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         LoadHeader

  Description:  サウンドデータの情報テーブルをロードする

  Arguments:    buffer - ロードアドレス
                size - バッファサイズ

  Returns:      成功したら true 失敗したら false
 *---------------------------------------------------------------------------*/
bool FsSoundArchive::LoadHeader( void* buffer, unsigned long size )
{
    NW_ASSERT( m_IsOpened );

    const s32 infoChunkOffset = m_ArchiveReader.GetInfoBlockOffset();
    const u32 infoChunkSize = m_ArchiveReader.GetInfoBlockSize();

    if ( size < infoChunkSize )
    {
        NW_WARNING(
            size >= infoChunkSize,
            "FsSoundArchive::LoadHeader buffer size is too small.\n"
        );
        return false;
    }

    //-----------------------------------------------------------------------------
    // 情報テーブルのロード

    FileAccessBegin();
    {
        m_FileStream.Seek( infoChunkOffset, nw::ut::FILE_STREAM_SEEK_BEGIN );
        s32 readSize = m_FileStream.Read( buffer, static_cast<s32>(infoChunkSize) );

        if ( readSize != static_cast<s32>(infoChunkSize) )
        {
            NW_WARNING(
                    false,
                    "FsSoundArchive::LoadHeader cannot read file.\n"
                    );
            return false;
        }
    }
    FileAccessEnd();

    m_ArchiveReader.SetInfoBlock( buffer /*, infoChunkSize */ );

#ifdef NW_SND_DEBUG_PRINT_ENABLE
    // デバッグ出力
    {
        // サウンド情報
        NW_LOG("### Sound INFO(%d)\n", m_ArchiveReader.GetSoundCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetSoundCount(); i++ )
        {
            // サウンド共通情報
            SoundArchive::ItemId soundId = GetSoundIdFromIndex( i );
            SoundArchive::SoundType type = m_ArchiveReader.GetSoundType( soundId );
            SoundArchive::SoundInfo info;
            bool ret = m_ArchiveReader.ReadSoundInfo( soundId, &info );
            NW_LOG("[%08X]?(%d) fileId(%d) playerId(0x%08X) plPrio(%d) actorId(%d) type(%d)\n",
                    soundId, ret, info.fileId, info.playerId, info.playerPriority, info.actorPlayerId, type );
            NW_LOG("  *common* volume(%d) panMode(%d) panCurve (%d)\n",
                    info.volume, info.panMode, info.panCurve );

            // ユーザーパラメータ
            {
                char buf[256]; buf[0] = '\0';
                for ( int i = 0; i <= SoundArchive::USER_PARAM_INDEX_MAX; i++ )
                {
                    u32 param;
                    bool result = m_ArchiveReader.ReadSoundUserParam( soundId, i, param );
                    ut::snprintf( buf, 256, "%s [%d](0x%08x)", buf, result, param );
                }
                NW_LOG("  *userparam* %s\n", buf);
            }

            // 3D サウンド情報
            {
                SoundArchive::Sound3DInfo info3d;
                if ( ReadSound3DInfo( soundId, &info3d ) )
                {
                    u32 flag = info3d.flags;
                    NW_LOG("  *3D* ra(%.2f) cv(%d) df(%d) vol(%d) pr(%d) pa(%d) sp(%d) bqf(%d)\n",
                            info3d.decayRatio, info3d.decayCurve, info3d.dopplerFactor,
                            (flag & Sound3DInfo::FLAG_CTRL_VOLUME) > 0,
                            (flag & Sound3DInfo::FLAG_CTRL_PRIORITY) > 0,
                            (flag & Sound3DInfo::FLAG_CTRL_PAN) > 0,
                            (flag & Sound3DInfo::FLAG_CTRL_SPAN) > 0,
                            (flag & Sound3DInfo::FLAG_CTRL_FILTER) > 0 );
                }
                else
                {
                    NW_LOG("  *3D* data not found\n");
                }
            }

            // サウンド別個別情報
            switch ( type )
            {
            case SoundArchive::SOUND_TYPE_SEQ:
            {
                SoundArchive::SequenceSoundInfo seqInfo;
                bool retDetail = m_ArchiveReader.ReadSequenceSoundInfo( soundId, &seqInfo );
                NW_LOG(" *SEQ* ret(%d) ofs(%d) bnk(0x%08X,0x%08X,0x%08X,0x%08X) trk(0x%x) chPrio(%d) rPrioFix(%d)\n",
                        retDetail,
                        seqInfo.startOffset,
                        seqInfo.bankIds[0], seqInfo.bankIds[1],
                        seqInfo.bankIds[2], seqInfo.bankIds[3],
                        seqInfo.allocateTrackFlags,
                        seqInfo.channelPriority, seqInfo.isReleasePriorityFix );
            }
            break;
            case SoundArchive::SOUND_TYPE_STRM:
            {
                SoundArchive::StreamSoundInfo strmInfo;
                bool retDetail = m_ArchiveReader.ReadStreamSoundInfo( soundId, &strmInfo );
                NW_LOG("  *STRM* ret(%d) trk(0x%08x) channel(%d) pitch(%f) mainSend(%d) fxSend(%d,%d,%d)\n",
                        retDetail, strmInfo.allocateTrackFlags, strmInfo.allocateChannelCount,
                        strmInfo.pitch, strmInfo.mainSend, strmInfo.fxSend[AUX_BUS_A], strmInfo.fxSend[AUX_BUS_B], strmInfo.fxSend[AUX_BUS_C] );
                for (u32 i = 0; i < SoundArchive::STRM_TRACK_NUM; i++)
                {
                    NW_LOG("  trk[%d] vol(%3d) pan(%3d) span(%3d) flags(0x%04x) mainSend(%d) fxSend(%d,%d,%d) lpf(%d) bqf(%d,%d) chCount(%d) globalCh(%2d,%2d)\n",
                            i, strmInfo.trackInfo[i].volume, strmInfo.trackInfo[i].pan,
                            strmInfo.trackInfo[i].span, strmInfo.trackInfo[i].flags,
                            strmInfo.trackInfo[i].mainSend,
                            strmInfo.trackInfo[i].fxSend[0],
                            strmInfo.trackInfo[i].fxSend[1],
                            strmInfo.trackInfo[i].fxSend[2],
                            strmInfo.trackInfo[i].lpfFreq,
                            strmInfo.trackInfo[i].biquadType,
                            strmInfo.trackInfo[i].biquadValue,
                            strmInfo.trackInfo[i].channelCount,
                            strmInfo.trackInfo[i].globalChannelIndex[0],
                            strmInfo.trackInfo[i].globalChannelIndex[1]);
                }
            }
            break;
            case SoundArchive::SOUND_TYPE_WAVE:
            {
                SoundArchive::WaveSoundInfo wsdInfo;
                bool retDetail = m_ArchiveReader.ReadWaveSoundInfo( soundId, &wsdInfo );
                NW_LOG("  *WSD* ret(%d) index(%d) trk(0x%x) chPrio(%d) rPrioFix(%d)\n",
                        retDetail,
                        wsdInfo.index, wsdInfo.allocateTrackCount,
                        wsdInfo.channelPriority, wsdInfo.isReleasePriorityFix );
            }
            break;
            case SoundArchive::SOUND_TYPE_INVALID:
            {
                NW_LOG("Invalid SoundType (not STRM/WSD/SEQ)\n");
            }
            break;
            }
        }

        // バンク情報
        NW_LOG("### BANK Info(%d)\n", m_ArchiveReader.GetBankCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetBankCount(); i++ )
        {
            SoundArchive::ItemId bankId = GetBankIdFromIndex( i );
            SoundArchive::BankInfo info;
            bool ret = m_ArchiveReader.ReadBankInfo( bankId, &info );
            NW_LOG("[%08X]?(%d) fileId(%d)\n", bankId, ret, info.fileId );
        }

        // プレイヤー情報
        NW_LOG("### PLAYER Info(%d)\n", m_ArchiveReader.GetPlayerCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetPlayerCount(); i++ )
        {
            SoundArchive::ItemId playerId = GetPlayerIdFromIndex( i );
            SoundArchive::PlayerInfo info;
            bool ret = m_ArchiveReader.ReadPlayerInfo( playerId, &info );
            NW_LOG("[%08X]?(%d) max(%d) heapSize(%d)\n",
                    playerId, ret, info.playableSoundMax, info.playerHeapSize );
        }

        // サウンドグループ情報
        NW_LOG("### SOUND-GROUP Info(%d)\n", m_ArchiveReader.GetSoundGroupCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetSoundGroupCount(); i++ )
        {
            SoundArchive::ItemId soundGroupId = GetSoundGroupIdFromIndex( i );
            SoundArchive::SoundGroupInfo info;
            bool ret = m_ArchiveReader.ReadSoundGroupInfo( soundGroupId, &info );
            NW_LOG("[%08X]?(%d) startId(%08X) end(%08X) fileId:count(%d)",
                    soundGroupId, ret, info.startId, info.endId, info.fileIdTable->count );
            for ( u32 j = 0; j < info.fileIdTable->count; j++ )
            {
                NW_LOG(" [%08X]", info.fileIdTable->item[j]);
            }
            NW_LOG("\n");
        }

        // グループ情報
        NW_LOG("### GROUP Info(%d)\n", m_ArchiveReader.GetGroupCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetGroupCount(); i++ )
        {
            SoundArchive::ItemId groupId = GetGroupIdFromIndex( i );
            SoundArchive::GroupInfo info;
            bool ret = m_ArchiveReader.ReadGroupInfo( groupId, &info );
            NW_LOG("[%08X]?(%d) fileId(%4d) fileSize(%8d:0x%08x)\n",
                    groupId, ret, info.fileId, info.groupFileSize, info.groupFileSize );
        }

        // 波形アーカイブ情報
        NW_LOG("### WAVE-ARCHIVE Info(%d)\n", m_ArchiveReader.GetWaveArchiveCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetWaveArchiveCount(); i++ )
        {
            SoundArchive::ItemId warcId = GetWaveArchiveIdFromIndex( i );
            SoundArchive::WaveArchiveInfo info;
            bool ret = m_ArchiveReader.ReadWaveArchiveInfo( warcId, &info );
            NW_LOG("[%08X]?(%d) fileId(%d)\n", warcId, ret, info.fileId );
        }

        // ファイル情報
        NW_LOG("### FILE Info(%d)\n", m_ArchiveReader.GetFileCount() );
        for ( u32 i = 0; i < m_ArchiveReader.GetFileCount(); i++ )
        {
            SoundArchive::FileInfo info;
            bool ret = m_ArchiveReader.ReadFileInfo( i, &info );
            if ( info.externalFilePath != NULL )
            {
                NW_LOG("[%4d]?(%d) fileSize(%8d) ofs(%8d) path(%s)\n", i, ret,
                        info.fileSize, info.offsetFromFileBlockHead, info.externalFilePath );
            }
            else
            {
                NW_LOG("[%4d]?(%d) fileSize(%8d) ofs(%8d) path((null))\n", i, ret,
                        info.fileSize, info.offsetFromFileBlockHead );
            }
        }

        // サウンドアーカイブプレイヤー情報
        NW_LOG("### SOUND-ARCHIVE-PLAYER Info\n");
        {
            SoundArchive::SoundArchivePlayerInfo info;
            bool ret = m_ArchiveReader.ReadSoundArchivePlayerInfo( &info );
            NW_LOG("sequenceSoundMax (%2d)\n", info.sequenceSoundMax );
            NW_LOG("sequenceTrackMax (%2d)\n", info.sequenceTrackMax );
            NW_LOG("streamSoundMax   (%2d)\n", info.streamSoundMax );
            NW_LOG("streamTrackMax   (%2d)\n", info.streamTrackMax );
            NW_LOG("streamChannelMax (%2d)\n", info.streamChannelMax );
            NW_LOG("waveSoundMax     (%2d)\n", info.waveSoundMax );
            NW_LOG("waveTrackMax     (%2d)\n", info.waveTrackMax );
            NW_LOG("streamBuffetTimes(%2d)\n", info.streamBufferTimes );
        }
    }
#endif /* NW_SND_DEBUG_PRINT_ENABLE */

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         LoadLabelStringData

  Description:  ラベルデータをロードする

  Arguments:    buffer - ロードアドレス
                size - バッファサイズ

  Returns:      成功したら true 失敗したら false
 *---------------------------------------------------------------------------*/
bool FsSoundArchive::LoadLabelStringData( void* buffer, unsigned long size )
{
    NW_ASSERT( m_IsOpened );

    const s32 stringBlockOffset = m_ArchiveReader.GetStringBlockOffset();
    const u32 stringBlockSize = m_ArchiveReader.GetStringBlockSize();

    if ( stringBlockOffset == internal::Util::Reference::INVALID_OFFSET )
    {
        // サウンドアーカイブの文字列ブロックが含まれていない
        return false;
    }

    if ( size < stringBlockSize )
    {
        NW_WARNING(
            size >= stringBlockSize,
            "FsSoundArchive::LoadLabelStringData buffer size is too small."
        );
        return false;
    }

    FileAccessBegin();
    {
        m_FileStream.Seek( stringBlockOffset, nw::ut::FILE_STREAM_SEEK_BEGIN );
        s32 readSize = m_FileStream.Read( buffer, static_cast<s32>(stringBlockSize) );
        if ( readSize != static_cast<s32>(stringBlockSize) )
        {
            NW_WARNING(
                    false,
                    "FsSoundArchive::LoadLabelStringData cannot read file.\n"
                    );
            return false;
        }
    }
    FileAccessEnd();

    m_ArchiveReader.SetStringBlock( buffer/*, stringBlockSize*/ );

#ifdef NW_SND_DEBUG_PRINT_ENABLE
    // デバッグ出力
    {
        NW_LOG("### PATRICIA-TREE Info\n");
        m_ArchiveReader.DumpTree();

        NW_LOG("### LABEL => ID\n");
        int count = m_ArchiveReader.GetStringCount();
        for ( int i = 0; i < count; i++ )
        {
            const char* str = m_ArchiveReader.GetString( i );
            NW_LOG("[%02d] (%-16s) => ItemId(0x%08X)\n", i, str, GetItemId( str ) );
#if 0
            NW_LOG("     as Sound ID:       (0x%08X)\n", GetSoundId( str ) );
            NW_LOG("     as Bank ID:        (0x%08X)\n", GetBankId( str ) );
            NW_LOG("     as Player ID:      (0x%08X)\n", GetPlayerId( str ) );
            NW_LOG("     as SoundGroup ID:  (0x%08X)\n", GetSoundGroupId( str ) );
            NW_LOG("     as Group ID:       (0x%08X)\n", GetGroupId( str ) );
            NW_LOG("     as WaveArchive ID: (0x%08X)\n", GetWaveArchiveId( str ) );
#endif
        }

        NW_LOG("### ID => LABEL\n");
        NW_LOG("[Sound]\n");
        for ( u32 i = 0; i < GetSoundCount(); i++ )
        {
            u32 id = GetSoundIdFromIndex( i );
            NW_LOG("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NW_LOG("[Bank]\n");
        for ( u32 i = 0; i < GetBankCount(); i++ )
        {
            u32 id = GetBankIdFromIndex( i );
            NW_LOG("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NW_LOG("[Player]\n");
        for ( u32 i = 0; i < GetPlayerCount(); i++ )
        {
            u32 id = GetPlayerIdFromIndex( i );
            NW_LOG("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NW_LOG("[SoundGroup]\n");
        for ( u32 i = 0; i < GetSoundGroupCount(); i++ )
        {
            u32 id = GetSoundGroupIdFromIndex( i );
            NW_LOG("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NW_LOG("[Group]\n");
        for ( u32 i = 0; i < GetGroupCount(); i++ )
        {
            u32 id = GetGroupIdFromIndex( i );
            NW_LOG("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NW_LOG("[WaveArchive]\n");
        for ( u32 i = 0; i < GetWaveArchiveCount(); i++ )
        {
            u32 id = GetWaveArchiveIdFromIndex( i );
            const char* label = GetItemLabel( id );
            if ( label != NULL )
            {
                NW_LOG("  [%08X] (%s)\n", id, label );
            }
            else
            {
                NW_LOG("  [%08X] ((anonymous))\n", id );
            }
        }
    }
#endif /* NW_SND_DEBUG_PRINT_ENABLE */

    return true;
}


void FsSoundArchive::FileAccessBegin() const
{
    if ( m_FileAccessMode == FILE_ACCESS_IN_FUNCTION )
    {
        if ( m_FileAccessCount == 0 )
        {
        #if defined(NW_PLATFORM_CAFE)
            m_FileStream.SetPriority(m_FsPriority);
            m_FileStream.Open( detail_GetFsClient(), m_SoundArchiveFullPath, "r", detail_GetFsCommandBlockPool() );
        #else
            m_FileStream.Open( m_SoundArchiveFullPath, internal::fnd::File::ACCESS_MODE_READ );
        #endif
        }
        m_FileAccessCount++;
    }
}

void FsSoundArchive::FileAccessEnd() const
{
    if ( m_FileAccessMode == FILE_ACCESS_IN_FUNCTION )
    {
        if ( m_FileAccessCount == 1 )
        {
            m_FileStream.Close();
        }
        if ( m_FileAccessCount > 0 )
        {
            m_FileAccessCount--;
        }
    }
}


#if defined(NW_PLATFORM_CAFE)
FsSoundArchive::FsFileStream::FsFileStream(
        FSClient* client, const char* path, u32 offset, u32 size,
        internal::FsFileStream::FsCommandBlockPool* pool )
: m_Offset( static_cast<s32>( offset ) ),
  m_Size( size )
{
    Open( client, path, "r", pool );

    NW_ASSERT( m_Size <= internal::FsFileStream::GetSize() );
    if ( size == 0 )
    {
        m_Size = internal::FsFileStream::GetSize();
    }

    internal::FsFileStream::Seek( m_Offset, ut::FILE_STREAM_SEEK_BEGIN );
}
#else
FsSoundArchive::FsFileStream::FsFileStream(
        const char* path, u32 offset, u32 size )
: m_Offset( static_cast<s32>( offset ) ),
  m_Size( size )
{
    Open( path, internal::fnd::File::ACCESS_MODE_READ );

    NW_ASSERT( m_Size <= internal::FsFileStream::GetSize() );
    if ( size == 0 )
    {
        m_Size = internal::FsFileStream::GetSize();
    }

    internal::FsFileStream::Seek( m_Offset, ut::FILE_STREAM_SEEK_BEGIN );
}
#endif

// オープン済みのファイルからストリームを作成して内部で持つ
FsSoundArchive::FsFileStream::FsFileStream(
    const internal::FsFileStream* fileStream, u32 offset, u32 size )
: internal::FsFileStream( fileStream ),
  m_Offset( static_cast<s32>( offset ) ),
  m_Size( size )
{
    NW_ASSERT( m_Size <= internal::FsFileStream::GetSize() );
    if ( size == 0 )
    {
        m_Size = internal::FsFileStream::GetSize();
    }
    internal::FsFileStream::Seek( m_Offset, ut::FILE_STREAM_SEEK_BEGIN );
}

s32 FsSoundArchive::FsFileStream::Read( void* buf, u32 length )
{
    NW_ALIGN32_ASSERT( buf );
    NW_ALIGN32_ASSERT( length );

    u32 curPos = internal::FsFileStream::Tell();
    if ( curPos + length > m_Offset + m_Size ) {
        length = static_cast<u32>( ut::RoundUp( m_Offset + m_Size - curPos, 32 ) );
    }
    return internal::FsFileStream::Read( buf, length );
}

bool FsSoundArchive::FsFileStream::Seek( s32 offset, u32 origin )
{
    switch( origin ) {
    case ut::FILE_STREAM_SEEK_BEGIN:
        offset += m_Offset;
        break;
    case ut::FILE_STREAM_SEEK_CURRENT:
        offset += internal::FsFileStream::Tell();
        break;
    case ut::FILE_STREAM_SEEK_END:
        offset = m_Offset + static_cast<s32>( m_Size ) - offset;
        break;
    default:
        NW_ASSERTMSG( false, "Unsupported Seek origin" );
        return false;
    }

    if ( offset < m_Offset )
    {
        offset = m_Offset;
    }
    else if ( offset > m_Offset + static_cast<s32>( m_Size ) )
    {
        offset = m_Offset + static_cast<s32>( m_Size );
    }

    return internal::FsFileStream::Seek( offset, ut::FILE_STREAM_SEEK_BEGIN );
}




#if defined(NW_PLATFORM_CAFE)
FsSoundArchive::CachedFsFileStream::CachedFsFileStream(
        FSClient* client, const char* path,
        internal::CachedFsFileStream::FsCommandBlockPool* pool )
{
    Open( client, path, "r", pool );
}
#else
FsSoundArchive::CachedFsFileStream::CachedFsFileStream(const char* path)
{
    Open( path, internal::fnd::File::ACCESS_MODE_READ );
}
#endif

} // namespace nw::snd
} // namespace nw

