﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_ExternalSoundPlayer.h>

#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_SoundActor.h>

namespace nw {
namespace snd {
namespace internal {

/*---------------------------------------------------------------------------*
  Name:         ExxternalSoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
ExternalSoundPlayer::ExternalSoundPlayer()
: m_PlayableCount( 1 )
{
}

/*---------------------------------------------------------------------------*
  Name:         ExternalSoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
ExternalSoundPlayer::~ExternalSoundPlayer()
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->DetachExternalSoundPlayer( this );
    }
}

/*---------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void ExternalSoundPlayer::StopAllSound( int fadeFrames )
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->Stop( fadeFrames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void ExternalSoundPlayer::PauseAllSound( bool flag, int fadeFrames )
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames );
    }
}

void ExternalSoundPlayer::Finalize( SoundActor* actor )
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->DetachSoundActor( actor );
        RemoveSound( &*curItr );
    }
}

/*---------------------------------------------------------------------------*
  Name:         AppendSound

  Description:  サウンドを登録する

  Arguments:    sound - サウンド

  Returns:      登録できたらtrue
 *---------------------------------------------------------------------------*/
bool ExternalSoundPlayer::AppendSound( internal::BasicSound* sound )
{
    NW_NULL_ASSERT( sound );

    int allocPriority = sound->CalcCurrentPlayerPriority();

    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 ) return false;
    while ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL ) return false;
        if ( allocPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        dropSound->Finalize();
    }

    m_SoundList.PushBack( sound );

    sound->AttachExternalSoundPlayer( this );

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  同時再生数を設定

  Arguments:    count - 同時再生数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void ExternalSoundPlayer::SetPlayableSoundCount( int count )
{
    m_PlayableCount = count;

    // 新しく設定された同時再生数を越えるサウンドを終了する
    while ( GetPlayingSoundCount() > GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        NW_NULL_ASSERT( dropSound );
        dropSound->Finalize();
    }
}

/*---------------------------------------------------------------------------*
  Name:         RemoveSound

  Description:  サウンドをプレイヤーリストから削除する

  Arguments:    sound - シーケンスサウンド

  Returns:      None.
 *---------------------------------------------------------------------------*/
void ExternalSoundPlayer::RemoveSound( internal::BasicSound* sound )
{
    // 再生リストから削除する
    m_SoundList.Erase( sound );
    sound->DetachExternalSoundPlayer( this );
}

/*---------------------------------------------------------------------------*
  Name:         detail_CanPlaySound

  Description:  指定したプライオリティのサウンドを再生できるかどうかを調べる

  Arguments:    startPriority - プライオリティ

  Returns:      再生可能ならtrue
 *---------------------------------------------------------------------------*/
bool ExternalSoundPlayer::detail_CanPlaySound( int startPriority )
{
    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 )
    {
        return false;
    }
    if ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL )
        {
            return false;
        }
        if ( startPriority < dropSound->CalcCurrentPlayerPriority() )
        {
            return false;
        }
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         GetLowestPrioritySound

  Description:  サウンドリストからもっともプライオリティの低いサウンドを探す

  Arguments:    None.

  Returns:      プライオリティの低いサウンドへのポインタ
 *---------------------------------------------------------------------------*/
internal::BasicSound* ExternalSoundPlayer::GetLowestPrioritySound()
{
    // insert to priority list
    int priority = internal::BasicSound::PRIORITY_MAX + 1;
    internal::BasicSound* sound = NULL;
    SoundList::Iterator itr = m_SoundList.GetBeginIter();
    while ( itr != m_SoundList.GetEndIter() )
    {
        int itrPriority = itr->CalcCurrentPlayerPriority();
        if ( priority > itrPriority )
        {
            sound = &*itr;
            priority = itrPriority;
        }
        (void)++itr;
    }
    return sound;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

