﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Debug.h>

#include <nw/assert.h>
#include <nw/ut/ut_String.h>

#include <cstring>

#if defined( NW_PLATFORM_WIN32 )
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <windows.h>
#elif defined( NW_USE_NINTENDO_SDK )
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <nn/nn_Windows.h>
#endif

namespace nw {
namespace snd {

namespace {

const u32 NOT_ENOUGH_SEQSOUND    = 0x00000001;
const u32 NOT_ENOUGH_STRMSOUND   = 0x00000002;
const u32 NOT_ENOUGH_WAVESOUND   = 0x00000004;
const u32 NOT_ENOUGH_SEQTRACK    = 0x00000008;
const u32 NOT_ENOUGH_STRMCHANNEL = 0x00000010;
const u32 NOT_ENOUGH_INSTANCE    = NOT_ENOUGH_SEQSOUND |
                                   NOT_ENOUGH_STRMSOUND |
                                   NOT_ENOUGH_WAVESOUND |
                                   NOT_ENOUGH_SEQTRACK |
                                   NOT_ENOUGH_STRMCHANNEL;

u32 gWarningFlag = NOT_ENOUGH_INSTANCE;

u32 GetWarningBitFlag( DebugWarningFlag warning )
{
    u32 bitFlag = 0;
    switch ( warning )
    {
    case DEBUG_WARNING_NOT_ENOUGH_INSTANCE:
        bitFlag = NOT_ENOUGH_INSTANCE;
        break;
    case DEBUG_WARNING_NOT_ENOUGH_SEQSOUND:
        bitFlag = NOT_ENOUGH_SEQSOUND;
        break;
    case DEBUG_WARNING_NOT_ENOUGH_STRMSOUND:
        bitFlag = NOT_ENOUGH_STRMSOUND;
        break;
    case DEBUG_WARNING_NOT_ENOUGH_WAVESOUND:
        bitFlag = NOT_ENOUGH_WAVESOUND;
        break;
    case DEBUG_WARNING_NOT_ENOUGH_SEQTRACK:
        bitFlag = NOT_ENOUGH_SEQTRACK;
        break;
    case DEBUG_WARNING_NOT_ENOUGH_STRMCHANNEL:
        bitFlag = NOT_ENOUGH_STRMCHANNEL;
        break;
    }
    return bitFlag;
}

} // anonymous namespace

/*---------------------------------------------------------------------------*
  Name:         Debug_SetWarningFlag

  Description:  警告メッセージの出力を設定する

  Arguments:    warning -   警告メッセージの種類
                enable -    出力するなら true

  Returns:      なし
 *---------------------------------------------------------------------------*/
void Debug_SetWarningFlag( DebugWarningFlag warning, bool enable )
{
    u32 bitFlag = GetWarningBitFlag( warning );

    if ( enable )
    {
        gWarningFlag |= bitFlag;
    }
    else
    {
        gWarningFlag &= ~bitFlag;
    }
}


/* ========================================================================
        非公開
   ======================================================================== */

namespace internal {

DebugLogFunc g_DebugLogHookFunc = NULL;

bool Debug_GetWarningFlag( DebugWarningFlag warning )
{
    u32 bitFlag = GetWarningBitFlag( warning );

    return ( gWarningFlag & bitFlag ) == bitFlag;
}

DebugWarningFlag Debug_GetDebugWarningFlagFromSoundType( DebugSoundType type )
{
    switch ( type )
    {
    case DEBUG_SOUND_TYPE_SEQSOUND:
        return DEBUG_WARNING_NOT_ENOUGH_SEQSOUND;
    case DEBUG_SOUND_TYPE_STRMSOUND:
        return DEBUG_WARNING_NOT_ENOUGH_STRMSOUND;
    case DEBUG_SOUND_TYPE_WAVESOUND:
        return DEBUG_WARNING_NOT_ENOUGH_WAVESOUND;
    default:
        NW_ASSERT( false );
        return DEBUG_WARNING_NOT_ENOUGH_SEQSOUND;
    }
}

const char* Debug_GetSoundTypeString( DebugSoundType type )
{
    switch ( type )
    {
    case DEBUG_SOUND_TYPE_SEQSOUND:
        return "seq";
    case DEBUG_SOUND_TYPE_STRMSOUND:
        return "strm";
    case DEBUG_SOUND_TYPE_WAVESOUND:
        return "wave";
    default:
        NW_ASSERT( false );
        return "";
    }
}

#if !defined( NW_RELEASE )
void Debug_OutputLog( const char* category, const char* message, ... )
{
    static const int TEXT_MAXLEN = 255;
    char text[TEXT_MAXLEN + 1];

    int categoryLength = category == NULL ? 0 : strlen(category);

    if ( categoryLength < 0 ) {
        return;
    }

#if defined( NW_PLATFORM_WIN32 )
    strncpy_s(text, TEXT_MAXLEN+1, category, categoryLength);
#elif defined( NW_USE_NINTENDO_SDK )
    nw::ut::strncpy(text, category, categoryLength);
#elif defined( NW_PLATFORM_CAFE )
    nw::ut::strncpy(text, category, categoryLength);
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    nw::ut::strncpy(text, category, categoryLength);
#else
    #error
#endif

    int length = categoryLength;
    int restLength = TEXT_MAXLEN + 1 - categoryLength;

    if(restLength > 0)
    {
        va_list arglist;
        va_start(arglist, message);

#if defined( NW_PLATFORM_WIN32 )
        length += vsprintf_s(text + length, restLength, message, arglist);
#elif defined( NW_USE_NINTENDO_SDK )
        length += vsnprintf(text + length, restLength, message, arglist);
#elif defined( NW_PLATFORM_CAFE )
        length += vsnprintf(text + length, restLength, message, arglist);
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
        length += vsnprintf(text + length, restLength, message, arglist);
#else
        #error
#endif

        va_end(arglist);
    }

    if ( length < categoryLength ) {
        // format error
        return;
    }

    if ( length > TEXT_MAXLEN ) length = TEXT_MAXLEN;
    text[length] = '\0';

    if ( g_DebugLogHookFunc != NULL )
    {
        g_DebugLogHookFunc( text );
    }
#if 0
    NW_LOG(text);
    NW_LOG("\n");
#endif
}

#endif

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

