﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_CurveLfo.h>
#include <nw/assert.h>
#include <nw/math/math_Triangular.h>
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {
namespace internal {

namespace {

// sine
f32 CurveSine(f32 arg)
{
    return nw::math::SinIdx(static_cast<u32>(arg * 0x100000000));
}

//
// square
//
f32 CurveSquare(f32 arg)
{
    if (arg < 0.5f)
    {
        return 0.0f;
    }
    return 1.0f;
}

//
// saw
//
f32 CurveSaw(f32 arg)
{
    // 定義域 (arg) 0.0f → 1.0f の間を、
    // 値域 0.0f → 1.0f まで変化していく
    return arg;
}

//
// triangle
//
f32 CurveTriangle(f32 arg)
{
    // [0] 0.00 -> 0.25  :  4 * arg;
    // [1] 0.25 -> 0.75  : -4 * arg + 2;
    // [2] 0.75 -> 1.00  :  4 * arg - 3;
    const f32 gradients[5] = { 4.0f, -4.0f, -4.0f, 4.0f, 4.0f };
    const f32 intercepts[5] = { 0.0f, 2.0f, 2.0f, -4.0f, 0.0f };
    u32 tmp = static_cast<u32>(arg * 4);
    return gradients[tmp] * arg + intercepts[tmp];
}

//
// random
//
f32 CurveRandom(f32 /*arg*/)
{
    u16 rand = Util::CalcRandom();
    f32 ret = static_cast<f32>(rand) / 0xffff;
    return ret;
}

CurveLfo::CurveFunc s_CurveFuncTable[CurveLfoParam::CURVE_COUNT] = {NULL};

} // anonymous namespace


// カーブテーブルの定義
void CurveLfo::InitializeCurveTable()
{
    std::memset(s_CurveFuncTable, 0, sizeof(CurveFunc) * CurveLfoParam::CURVE_COUNT);

    s_CurveFuncTable[CurveLfoParam::CURVE_SINE] = CurveSine;
    s_CurveFuncTable[CurveLfoParam::CURVE_TRIANGLE] = CurveTriangle;
    s_CurveFuncTable[CurveLfoParam::CURVE_SAW] = CurveSaw;
    s_CurveFuncTable[CurveLfoParam::CURVE_SQUARE] = CurveSquare;
    s_CurveFuncTable[CurveLfoParam::CURVE_RANDOM] = CurveRandom;
}

CurveLfo::CurveFunc CurveLfo::RegisterUserCurve(CurveFunc func, u32 index)
{
    NW_ASSERT_MINMAX(
            static_cast<s32>(index),
            0,
            CurveLfoParam::CURVE_USER_MAX - CurveLfoParam::CURVE_USER_MIN);

    s32 tableIndex = index + CurveLfoParam::CURVE_USER_MIN;
    CurveFunc tmp = s_CurveFuncTable[tableIndex];
    s_CurveFuncTable[tableIndex] = func;
    return tmp;
}

CurveLfo::CurveFunc CurveLfo::UnregisterUserCurve(u32 index)
{
    NW_ASSERT_MINMAX(
            static_cast<s32>(index),
            0,
            CurveLfoParam::CURVE_USER_MAX - CurveLfoParam::CURVE_USER_MIN);

    s32 tableIndex = index + CurveLfoParam::CURVE_USER_MIN;
    CurveFunc tmp = s_CurveFuncTable[tableIndex];
    s_CurveFuncTable[tableIndex] = NULL;
    return tmp;
}

/* ========================================================================
        public function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         LfoParam::Init

  Description:  LFOパラメータ構造体を初期化します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void CurveLfoParam::Initialize()
{
    depth = 0.0f;
    speed = 6.25f;
    delay = 0;
    range = 1;
    curve = 0;
    phase = 0;
}

/*---------------------------------------------------------------------------*
  Name:         Reset

  Description:  LFOをリセットします

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void CurveLfo::Reset()
{
    m_Counter = 0.0f;
    m_RandomValue = 1.0f;
    m_DelayCounter = 0;
    m_IsStart = false;
    m_IsNext = false;
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  LFOを更新します

  Arguments:    msec - 更新時間[msec]

  Returns:      なし
 *---------------------------------------------------------------------------*/
void CurveLfo::Update( int msec )
{
    if ( m_DelayCounter < m_Param.delay )
    {
        if ( m_DelayCounter + msec <= m_Param.delay )
        {
            m_DelayCounter += msec;
            return;
        }
        else
        {
            msec -= m_Param.delay - m_DelayCounter;
            m_DelayCounter = m_Param.delay;
        }
    }

    if (m_Param.speed > 0.0f)
    {
        if (m_IsStart == false)
        {
            m_Counter = m_Param.phase / 127.0f;
            m_IsStart = true;
        }
        m_Counter += m_Param.speed * msec / 1000.0f;
        if (m_Counter >= 1.0f)
        {
            m_IsNext = true;
        }
        else
        {
            m_IsNext = false;
        }
        m_Counter -= static_cast<int>( m_Counter );
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetValue

  Description:  現在のLFO値を取得します

  Arguments:    なし

  Returns:      現在のLFO値
 *---------------------------------------------------------------------------*/
f32 CurveLfo::GetValue() const
{
    if ( m_Param.depth == 0.0f ) return 0.0f;
    if ( m_DelayCounter < m_Param.delay ) return 0.0f;

    NW_ASSERT_MINMAX(
            static_cast<CurveLfoParam::CurveType>(m_Param.curve),
            CurveLfoParam::CURVE_MIN,
            CurveLfoParam::CURVE_MAX);
    CurveFunc func = s_CurveFuncTable[m_Param.curve];

    f32 value = 0.f;
    if (func != NULL)
    {
        NW_ASSERT_MINMAX(m_Counter, 0.0f, 1.0f);

        // ランダムは、１周期中ずっと同じ値を取るため、特別対応とした
        if (m_Param.curve == CurveLfoParam::CURVE_RANDOM)
        {
            if (m_IsNext)
            {
                value = func(m_Counter);
                m_RandomValue = value;
            }
            else
            {
                value = m_RandomValue;
            }
        }
        else
        {
            value = func(m_Counter);
        }
    }
    else
    {
        value = 1.0f;
    }
    value *= m_Param.depth;
    value *= m_Param.range;
    return value;
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

