﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_SoundPlayer.h>
#include <nw/snd/snd_ExternalSoundPlayer.h>
#include <nw/snd/snd_SoundActor.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_SoundHandle.h>
#include <cstring>  // std::memcpy


namespace nw {
namespace snd {
namespace internal {

u32 BasicSound::s_LastInstanceId = 0;

/*---------------------------------------------------------------------------*
  Name:         BasicSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
BasicSound::BasicSound()
: m_State(STATE_CONSTRUCTED),
  m_pUserParam(NULL),
  m_UserParamSize(0),
  m_SoundStopCallback(NULL)
{
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  初期化

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Initialize()
{
    // パラメータの初期化
    {
        m_InstanceId = s_LastInstanceId++;

        m_pPlayerHeap = NULL;
        m_pGeneralHandle = NULL;
        m_pTempGeneralHandle = NULL;
        m_pSoundPlayer = NULL;
        m_pSoundActor = NULL;
        m_pExtSoundPlayer = NULL;
        m_Id = INVALID_ID;

        m_AmbientInfo.paramUpdateCallback = NULL;
        m_AmbientInfo.argUpdateCallback = NULL;
        m_AmbientInfo.argAllocatorCallback = NULL;
        m_AmbientInfo.arg = NULL;
        m_AmbientInfo.argSize = 0;

        m_PlayerState         = PLAYER_STATE_INIT;
        m_PauseState          = PAUSE_STATE_NORMAL;
        m_MuteState           = MUTE_STATE_NORMAL;
        m_UnPauseFlag         = false;
        m_StartFlag           = false;
        m_StartedFlag         = false;
        m_AutoStopFlag        = false;
        m_FadeOutFlag         = false;
        m_PlayerAvailableFlag = false;

        m_AutoStopCounter     = 0;
        m_UpdateCounter       = 0;
        m_PlayingCounter      = 0;

        m_FadeVolume.InitValue( 1.0f );
        m_PauseFadeVolume.InitValue( 1.0f );
        m_MuteFadeVolume.InitValue( 1.0f );

        m_InitVolume        = 1.0f;
        m_Pitch             = 1.0f;
        m_LpfFreq           = 0.0f;
        m_BiquadFilterType  = BIQUAD_FILTER_TYPE_INHERIT;
        m_BiquadFilterValue = 0.0f;

        m_OutputLineFlag = ( m_pSoundPlayer != NULL ) ?
            m_pSoundPlayer->GetDefaultOutputLine() :
            OUTPUT_LINE_MAIN;

        m_CommonParam.Initialize();
        for ( int i = 0; i < OUTPUT_DEVICE_COUNT; i++ )
        {
            m_OutputParam[i].Initialize();
        }
        for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
        {
            m_RemoteParam[i].Initialize();
        }

        // AmbientParam
        m_AmbientParam.Initialize();
        m_ActorParam.Reset();
        if (m_UserParamSize > 0)
        {
            std::memset(m_pUserParam, 0x0, m_UserParamSize);
        }
    }

    // プレイヤーの初期化
    {
        driver::BasicSoundPlayer* basicPlayer = GetBasicSoundPlayerHandle();
        DriverCommand& cmdmgr = DriverCommand::GetInstance();

        DriverCommandPlayerInit* command =
            cmdmgr.AllocCommand<DriverCommandPlayerInit>();
        command->id = DRIVER_COMMAND_PLAYER_INIT;
        command->player = basicPlayer;
        command->availableFlagPtr = &m_PlayerAvailableFlag;
        cmdmgr.PushCommand(command);
        basicPlayer->InitializeEvent();
    }
    m_State = STATE_INITIALIZED;
}

void BasicSound::SetPriority( int priority, int ambientPriority )
{
    NW_MINMAX_ASSERT( priority, PRIORITY_MIN, PRIORITY_MAX );

    m_Priority = static_cast<u8>(priority);
    m_AmbientParam.priority = ambientPriority;
}
void BasicSound::GetPriority( int* priority, int* ambientPriority ) const
{
    if ( priority != NULL )
    {
        *priority = m_Priority;
    }
    if ( ambientPriority != NULL )
    {
        *ambientPriority = m_AmbientParam.priority;
    }
}

void BasicSound::ClearIsFinalizedForCannotAllocatedResourceFlag()
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandPlayerClearResourceFlag* command =
        cmdmgr.AllocCommand<DriverCommandPlayerClearResourceFlag>();
    command->id = DRIVER_COMMAND_PLAYER_CLEAR_RESOURCEFLAG;
    command->player = GetBasicSoundPlayerHandle();
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Finalize()
{
    if (m_State < STATE_INITIALIZED)
    {
        return;
    }
    if (m_State >= STATE_FINALIZED)
    {
        return;
    }

    // IDを無効化
    SetId( INVALID_ID );

    // ハンドルとの切断
    if ( IsAttachedGeneralHandle() )
    {
        DetachGeneralHandle();
    }
    if ( IsAttachedTempGeneralHandle() )
    {
        DetachTempGeneralHandle();
    }
    if ( IsAttachedTempSpecialHandle() )
    {
        DetachTempSpecialHandle();
    }

    // プレイヤーとの切断
    if ( m_pSoundPlayer != NULL ) m_pSoundPlayer->detail_RemoveSound( this );
    if ( m_pExtSoundPlayer != NULL ) m_pExtSoundPlayer->RemoveSound( this );

    if ( m_AmbientInfo.argAllocatorCallback != NULL )
    {
        m_AmbientInfo.argAllocatorCallback->detail_FreeAmbientArg( m_AmbientInfo.arg, this );
        m_AmbientInfo.arg = NULL;
    }

    {
        // プレイヤーの停止
        if ( m_StartedFlag )
        {
            DriverCommand& cmdmgr = DriverCommand::GetInstance();

            DriverCommandPlayer* command =
                cmdmgr.AllocCommand<DriverCommandPlayer>();
            command->id = DRIVER_COMMAND_PLAYER_STOP;
            command->player = GetBasicSoundPlayerHandle();
            command->flag = m_FadeOutFlag;
            cmdmgr.PushCommand(command);

            m_StartedFlag = false;
        }
        m_PlayerAvailableFlag = false;
        m_PlayerState = PLAYER_STATE_STOP;

        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
        command->id = DRIVER_COMMAND_PLAYER_FINALIZE;
        command->player = GetBasicSoundPlayerHandle();
        cmdmgr.PushCommand(command);
    }

    if (m_SoundStopCallback != NULL)
    {
        m_SoundStopCallback();
        m_SoundStopCallback = NULL;
    }

    m_FadeOutFlag = false;
    m_State = STATE_FINALIZED;
}

/*---------------------------------------------------------------------------*
  Name:         StartPrepared

  Description:  プリペアが完了していたらサウンドをスタートする

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::StartPrepared()
{
    m_StartFlag = true;
}

/*---------------------------------------------------------------------------*
  Name:         Stop

  Description:  サウンド停止

  Arguments:    fadeFrames - フェードアウトフレーム

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Stop( int fadeFrames )
{
    if ( ( fadeFrames <= 0 ) || ( m_PauseState == PAUSE_STATE_PAUSED ) ||
         ( m_StartFlag == false && m_StartedFlag == false ) )   // StartPrepared 前
    {
        Finalize();
        return;
    }

    int frames = static_cast<int>( fadeFrames * m_FadeVolume.GetValue() );
    m_FadeVolume.SetTarget( 0.0f, frames );
    SetPlayerPriority( PRIORITY_MIN );
    m_AutoStopFlag = false;
    m_PauseState = PAUSE_STATE_NORMAL;
    m_UnPauseFlag = false;
    m_FadeOutFlag = true;
    m_MuteState = MUTE_STATE_NORMAL;
}

/*---------------------------------------------------------------------------*
  Name:         Pause

  Description:  サウンドの一時停止または再開

  Arguments:    flag       - 一時停止または再開
                fadeFrames - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Pause( bool flag, int fadeFrames )
{
    int frames;
    if ( flag )
    {
        // 一時停止処理
        switch( m_PauseState ) {
        case PAUSE_STATE_NORMAL:
        case PAUSE_STATE_PAUSING:
        case PAUSE_STATE_UNPAUSING:
            frames = static_cast<int>( fadeFrames * m_PauseFadeVolume.GetValue() );
            if ( frames <= 0 ) frames = 1;
            m_PauseFadeVolume.SetTarget( 0.0f, frames);
            m_PauseState = PAUSE_STATE_PAUSING;
            m_UnPauseFlag = false;
            break;
        case PAUSE_STATE_PAUSED:
            // do nothing
            return;
        default:
            NW_ASSERTMSG( false, "Unexpected pause state %d", m_PauseState );
            return;
        }
    }
    else
    {
        switch( m_PauseState ) {
        case PAUSE_STATE_NORMAL:
            // do nothing
            return;
        case PAUSE_STATE_PAUSING:
        case PAUSE_STATE_UNPAUSING:
        case PAUSE_STATE_PAUSED:
            frames = static_cast<int>( fadeFrames * ( 1.0f - m_PauseFadeVolume.GetValue() ) );
            if ( frames <= 0 ) frames = 1;
            m_PauseFadeVolume.SetTarget( 1.0f, frames);
            m_PauseState = PAUSE_STATE_UNPAUSING;
            m_UnPauseFlag = true;
            break;
        default:
            NW_ASSERTMSG( false, "Unexpected pause state %d", m_PauseState );
            return;
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         Mute

  Description:  サウンドのミュートまたはミュート解除

  Arguments:    flag       - ミュートまたはミュート解除
                fadeFrames - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Mute(bool flag, int fadeFrames)
{
    int frames;
    if ( flag )
    {
        // ミュート処理
        switch ( m_MuteState )
        {
        case MUTE_STATE_NORMAL:
        case MUTE_STATE_MUTING:
        case MUTE_STATE_UNMUTING:
            frames = static_cast<int>( fadeFrames * m_MuteFadeVolume.GetValue() );
            if ( frames <= 0 ) frames = 1;
            m_MuteFadeVolume.SetTarget( 0.0f, frames );
            m_MuteState = MUTE_STATE_MUTING;
            break;
        case MUTE_STATE_MUTED:
            // do nothing
            return;
        default:
            NW_ASSERTMSG( false, "Unexpected mute state %d", m_MuteState );
            return;
        }
    }
    else
    {
        switch ( m_MuteState )
        {
        case MUTE_STATE_NORMAL:
            // do nothing
            return;
        case MUTE_STATE_MUTING:
        case MUTE_STATE_UNMUTING:
        case MUTE_STATE_MUTED:
            frames = static_cast<int>( fadeFrames * ( 1.0f - m_MuteFadeVolume.GetValue() ) );
            if ( frames <= 0 ) frames = 1;
            m_MuteFadeVolume.SetTarget( 1.0f, frames );
            m_MuteState = MUTE_STATE_UNMUTING;
            break;
        default:
            NW_ASSERTMSG( false, "Unexpected mute state %d", m_MuteState );
            return;
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetAutoStopCounter

  Description:  サウンドが自動停止するまでのフレーム数を設定する

  Arguments:    frames - 自動停止するまでのフーレム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetAutoStopCounter( int frames )
{
    m_AutoStopCounter = frames;
    m_AutoStopFlag = ( frames > 0 );
}

/*---------------------------------------------------------------------------*
  Name:         FadeIn

  Description:  フェードイン

  Arguments:    frames - 変化フレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::FadeIn( int frames )
{
    if ( m_FadeOutFlag )
    {
        return;
    }
    if ( m_UpdateCounter == 0 )
    {
        m_FadeVolume.InitValue( 0.0f );
        m_FadeVolume.SetTarget( 1.0f, frames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         IsPause

  Description:  一時停止中かどうかを取得

  Arguments:    None.

  Returns:      一時停止中ならば true
 *---------------------------------------------------------------------------*/
bool BasicSound::IsPause() const
{
    return (m_PauseState == PAUSE_STATE_PAUSING) || (m_PauseState == PAUSE_STATE_PAUSED);
}

/*---------------------------------------------------------------------------*
  Name:         IsMute

  Description:  ミュート中かどうかを取得

  Arguments:    None.

  Returns:      ミュート中ならば true
 *---------------------------------------------------------------------------*/
bool BasicSound::IsMute() const
{
    return (m_MuteState == MUTE_STATE_MUTING) || (m_MuteState == MUTE_STATE_MUTED);
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  サウンドの更新

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::Update()
{
    driver::BasicSoundPlayer* player = GetBasicSoundPlayerHandle();
    NW_ASSERT_NOT_NULL(player);

    if ( m_PlayerAvailableFlag && GetBasicSoundPlayerHandle()->IsPlayFinished() )
    {
        Finalize();
        return;
    }

    if ( player->IsFinalizedForCannotAllocateResource() )
    {
        ClearIsFinalizedForCannotAllocatedResourceFlag();
        Finalize();
        return;
    }

    if (IsPrepared() == false)
    {
        return;
    }

    // 自動停止カウンター処理
    if ( m_AutoStopFlag )
    {
        if ( m_AutoStopCounter == 0 )
        {
            if ( ( m_PauseState == PAUSE_STATE_NORMAL ) ||
                 ( m_PauseState == PAUSE_STATE_UNPAUSING ) )
            {
                Stop( 0 );
                return;
            }
        }
        else
        {
            --m_AutoStopCounter;
        }
    }

    // スタート前の処理
    bool playerStartFlag = false;
    if ( ! m_StartedFlag )
    {
        if ( ! m_StartFlag ) return;
        if ( ! IsPrepared() ) return;

        m_PlayerState = PLAYER_STATE_PLAY;
        playerStartFlag = true;
    }

    // カウンタ更新
    if ( m_PlayerState == PLAYER_STATE_PLAY )
    {
        if ( m_UpdateCounter < 0xffffffff ) m_UpdateCounter++;
    }
    if ( m_PlayerState == PLAYER_STATE_PLAY && m_PauseState != PAUSE_STATE_PAUSED )
    {
        if ( m_PlayingCounter < 0xffffffff ) m_PlayingCounter++;
    }

    bool pauseFadeIsFinished = m_PauseFadeVolume.IsFinished();

    // フェードフレームの更新
    switch ( m_PauseState )
    {
    case PAUSE_STATE_PAUSING:
        m_PauseFadeVolume.Update();
        break;
    case PAUSE_STATE_UNPAUSING:
        m_PauseFadeVolume.Update();
        UpdateMoveValue();
        break;
    case PAUSE_STATE_NORMAL:
        UpdateMoveValue();
        break;
    }

    // 外部パラメータの更新
    if ( m_AmbientInfo.argUpdateCallback != NULL )
    {
        m_AmbientInfo.argUpdateCallback->detail_UpdateAmbientArg(
            m_AmbientInfo.arg,
            this
        );
    }

    if ( m_AmbientInfo.paramUpdateCallback != NULL )
    {
        SoundAmbientParam ambientParam;
        if ( m_UpdateCounter > 0 )
        {
            ambientParam.volume            = m_AmbientParam.volume;
            ambientParam.pitch             = m_AmbientParam.pitch;
            ambientParam.lpf               = m_AmbientParam.lpf;
            ambientParam.biquadFilterValue = m_AmbientParam.biquadFilterValue;
            ambientParam.biquadFilterType  = m_AmbientParam.biquadFilterType;
            ambientParam.userData          = m_AmbientParam.userData;
            ambientParam.priority          = m_AmbientParam.priority;

            ambientParam.outputLineFlag    = m_AmbientParam.outputLineFlag;
            ambientParam.tvParam           = m_AmbientParam.tvParam;
            for ( int i = 0; i < DRC_OUT_COUNT; i++ )
            {
                ambientParam.drcParam[i]   = m_AmbientParam.drcParam[i];
            }
        }
        else
        {
            ambientParam.userData = 0;
        }

        m_AmbientInfo.paramUpdateCallback->detail_UpdateAmbientParam(
            m_AmbientInfo.arg,
            m_Id,
            &ambientParam
        );

        m_AmbientParam.volume            = ambientParam.volume;
        m_AmbientParam.pitch             = ambientParam.pitch;
        m_AmbientParam.lpf               = ambientParam.lpf;
        m_AmbientParam.biquadFilterValue = ambientParam.biquadFilterValue;
        m_AmbientParam.biquadFilterType  = ambientParam.biquadFilterType;
        m_AmbientParam.userData          = ambientParam.userData;
        m_AmbientParam.priority          = ambientParam.priority;

        m_AmbientParam.outputLineFlag    = ambientParam.outputLineFlag;
        m_AmbientParam.tvParam           = ambientParam.tvParam;
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            m_AmbientParam.drcParam[i]   = ambientParam.drcParam[i];
        }

    }

    if ( m_pSoundActor != NULL )
    {
        m_ActorParam = m_pSoundActor->detail_GetActorParam();
    }

    UpdateParam();

    // フェードアウト完了チェック
    if ( m_FadeOutFlag && m_FadeVolume.IsFinished() )
    {
        m_FadeOutFlag = false;
        Finalize();
        return;
    }

    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    // プレイヤーの開始処理
    if ( playerStartFlag )
    {
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
        command->id = DRIVER_COMMAND_PLAYER_START;
        command->player = GetBasicSoundPlayerHandle();
        cmdmgr.PushCommand(command);

        m_StartedFlag = true;
        m_StartFlag = false;
    }

    // ポーズステータスチェック
    if ( m_PauseState == PAUSE_STATE_PAUSING )
    {
        if (pauseFadeIsFinished)
        {
            DriverCommandPlayer* command =
                cmdmgr.AllocCommand<DriverCommandPlayer>();
            command->id = DRIVER_COMMAND_PLAYER_PAUSE;
            command->player = GetBasicSoundPlayerHandle();
            command->flag = true;
            cmdmgr.PushCommand(command);

            m_PauseState = PAUSE_STATE_PAUSED;
        }
    }
    else if ( m_PauseState == PAUSE_STATE_UNPAUSING )
    {
        if ( m_PauseFadeVolume.IsFinished() )
        {
            m_PauseState = PAUSE_STATE_NORMAL;
        }
    }
    if ( m_UnPauseFlag ) {
        DriverCommandPlayer* command =
            cmdmgr.AllocCommand<DriverCommandPlayer>();
        command->id = DRIVER_COMMAND_PLAYER_PAUSE;
        command->player = GetBasicSoundPlayerHandle();
        command->flag = false;
        cmdmgr.PushCommand(command);

        m_UnPauseFlag = false;
    }

    // ミュートステータスチェック
    if ( m_MuteState == MUTE_STATE_MUTING )
    {
        if ( m_MuteFadeVolume.IsFinished() )
        {
            m_MuteState = MUTE_STATE_MUTED;
        }
    }
    else if ( m_MuteState == MUTE_STATE_UNMUTING )
    {
        if ( m_MuteFadeVolume.IsFinished() )
        {
            m_MuteState = MUTE_STATE_NORMAL;
        }
    }
}

void BasicSound::UpdateMoveValue()
{
    m_FadeVolume.Update();
    m_MuteFadeVolume.Update();
    m_CommonParam.Update();
}

void BasicSound::UpdateParam()
{
    OnUpdateParam();    // 各サウンドごとの処理

    // パラメータの更新
    f32 volume = 1.0f;
    if ( m_MuteState == MUTE_STATE_MUTED )
    {
        volume = 0;
    }
    else
    {
        volume *= m_InitVolume;
        volume *= GetSoundPlayer()->GetVolume();
        volume *= m_CommonParam.GetVolume();
        volume *= m_FadeVolume.GetValue();
        volume *= m_PauseFadeVolume.GetValue();
        volume *= m_MuteFadeVolume.GetValue();
        volume *= m_AmbientParam.volume;
        volume *= m_ActorParam.volume;
    }

    f32 pitch = 1.0f;
    pitch *= m_Pitch;
    pitch *= m_AmbientParam.pitch;
    pitch *= m_ActorParam.pitch;

    f32 lpfFreq = m_LpfFreq;
    lpfFreq += m_AmbientParam.lpf;
    lpfFreq += GetSoundPlayer()->GetLpfFreq();
    lpfFreq += m_ActorParam.lpf;

    int biquadFilterType = m_BiquadFilterType;
    f32 biquadFilterValue = m_BiquadFilterValue;
    if ( biquadFilterType == BIQUAD_FILTER_TYPE_INHERIT )
    {
        biquadFilterType = GetSoundPlayer()->GetBiquadFilterType();
        biquadFilterValue = GetSoundPlayer()->GetBiquadFilterValue();

        if( biquadFilterType == BIQUAD_FILTER_TYPE_INHERIT )
        {
            biquadFilterType = m_AmbientParam.biquadFilterType;
            biquadFilterValue = m_AmbientParam.biquadFilterValue;
        }
    }

    u32 outputLineFlag = m_OutputLineFlag;
    if ( m_AmbientParam.outputLineFlag != -1 )
    {
        outputLineFlag = m_AmbientParam.outputLineFlag;
    }


    OutputParam param[OUTPUT_DEVICE_COUNT];
    for ( int i = 0; i < OUTPUT_DEVICE_COUNT; i++ )
    {
        param[i] = m_OutputParam[i];
        ApplyCommonParam( param[i] );   // 全体パラメータの適用
    }

    // TV
    OutputParam* main = &param[OUTPUT_DEVICE_MAIN];
    {
        main->volume *= ( GetSoundPlayer()->GetOutputVolume( OUTPUT_DEVICE_MAIN )
                * m_AmbientParam.tvParam.volume
                * m_ActorParam.tvVolume );
        main->pan += ( m_AmbientParam.tvParam.pan + m_ActorParam.tvPan );
        main->span += m_AmbientParam.tvParam.span;
        main->mainSend += GetSoundPlayer()->GetOutputMainSend( OUTPUT_DEVICE_MAIN );
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            main->fxSend[i] += ( GetSoundPlayer()->GetOutputFxSend(OUTPUT_DEVICE_MAIN, static_cast<AuxBus>(i))
                    + m_AmbientParam.tvParam.fxSend[i] );
        }
    }

    // DRC
    OutputParam* drc[DRC_OUT_COUNT];
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        drc[i] = &param[OUTPUT_DEVICE_DRC+i];
        drc[i]->volume *= ( GetSoundPlayer()->GetOutputVolume( OUTPUT_DEVICE_DRC )
                * m_AmbientParam.drcParam[i].volume
                * m_ActorParam.drcVolume[i] );
        drc[i]->pan += ( m_AmbientParam.drcParam[i].pan + m_ActorParam.drcPan[i] );
        drc[i]->span += m_AmbientParam.drcParam[i].span;
        drc[i]->mainSend += GetSoundPlayer()->GetOutputMainSend( OUTPUT_DEVICE_DRC );
        for ( int j = 0; j < AUX_BUS_NUM; j++ )
        {
            drc[i]->fxSend[j] += ( GetSoundPlayer()->GetOutputFxSend( OUTPUT_DEVICE_DRC, static_cast<AuxBus>(j) )
                    + m_AmbientParam.drcParam[i].fxSend[j] );
        }
    }

    // リモコン出力パラメータ
    RemoteOutputParam remoteParam[REMOTE_OUT_COUNT];
    for ( u32 i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        RemoteOutputParam& rRemoteParam = remoteParam[i];
        rRemoteParam = m_RemoteParam[i];
#if defined (NW_PLATFORM_WIN32)
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
        rRemoteParam.volume *= GetSoundPlayer()->GetRemoteOutVolume( i );
#if defined (NW_PLATFORM_WIN32)
#pragma warning(pop)
#endif
    }


    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    DriverCommandPlayerParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerParam>();
    command->id                = DRIVER_COMMAND_PLAYER_PARAM;
    command->player            = GetBasicSoundPlayerHandle();
    command->volume            = volume;
    command->pitch             = pitch;
    command->lpfFreq           = lpfFreq;
    command->biquadFilterType  = biquadFilterType;
    command->biquadFilterValue = biquadFilterValue;
    command->outputLineFlag    = outputLineFlag;
    command->tvParam           = *main;
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        command->drcParam[i] = *drc[i];
    }
    for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        command->remoteParam[i] = remoteParam[i];
    }


    cmdmgr.PushCommand(command);
}

void BasicSound::ApplyCommonParam( OutputParam& param )
{
    // param.volume *= m_CommonParam.GetVolume();
    param.mixMode = m_CommonParam.mixMode;
    param.pan += m_CommonParam.pan;
    param.span += m_CommonParam.span;
    param.mainSend += m_CommonParam.mainSend;
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        param.fxSend[i] += m_CommonParam.fxSend[i];
    }
}

/*---------------------------------------------------------------------------*
  Name:         AttachPlayerHeap

  Description:  PlayerHeapを結びつける

  Arguments:    pHeap - 結びつけるPlayerHeap

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::AttachPlayerHeap( PlayerHeap* pHeap )
{
    NW_NULL_ASSERT( pHeap );
    NW_ASSERT( m_pPlayerHeap == NULL );

    m_pPlayerHeap = pHeap;
}

/*---------------------------------------------------------------------------*
  Name:         DetachPlayerHeap

  Description:  PlayerHeapを切り離す

  Arguments:    heap - 切り離すPlayerHeap

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::DetachPlayerHeap( PlayerHeap* pHeap )
{
    NW_NULL_ASSERT( pHeap );
    NW_ASSERT( pHeap == m_pPlayerHeap );
    (void)pHeap;

    m_pPlayerHeap = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         AttachSoundPlayer

  Description:  SoundPlayerを結びつける

  Arguments:    player - 結びつけるSoundPlayer

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::AttachSoundPlayer( SoundPlayer* player )
{
    NW_NULL_ASSERT( player );
    NW_ASSERT( m_pSoundPlayer == NULL );

    m_pSoundPlayer = player;
}

/*---------------------------------------------------------------------------*
  Name:         DetachSoundPlayer

  Description:  SoundPlayerを切り離す

  Arguments:    player - 切り離すSoundPlayer

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::DetachSoundPlayer( SoundPlayer* player )
{
    NW_NULL_ASSERT( player );
    NW_ASSERT( player == m_pSoundPlayer );
    (void)player;

    m_pSoundPlayer = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         AttachSoundActor

  Description:  SoundActorを結びつける

  Arguments:    actor - 結びつけるSoundActor

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::AttachSoundActor( SoundActor* actor )
{
    NW_NULL_ASSERT( actor );
    NW_ASSERT( m_pSoundActor == NULL );

    m_pSoundActor = actor;
}

/*---------------------------------------------------------------------------*
  Name:         DetachSoundActor

  Description:  SoundActorを切り離す

  Arguments:    actor - 切り離すSoundActor

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::DetachSoundActor( SoundActor* actor )
{
    NW_NULL_ASSERT( actor );
    NW_ASSERT( actor == m_pSoundActor );
    (void)actor;

    m_pSoundActor = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         AttachExternalSoundPlayer

  Description:  ExternalSoundPlayerを結びつける

  Arguments:    extPlayer - 結びつけるExternalSoundPlayer

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::AttachExternalSoundPlayer( ExternalSoundPlayer* extPlayer )
{
    NW_NULL_ASSERT( extPlayer );
    NW_ASSERT( m_pExtSoundPlayer == NULL );

    m_pExtSoundPlayer = extPlayer;
}

/*---------------------------------------------------------------------------*
  Name:         DetachExternalSoundPlayer

  Description:  ExternalSoundPlayerを切り離す

  Arguments:    extPlayer - 切り離すExternalSoundPlayer

  Returns:      None.
  *---------------------------------------------------------------------------*/
void BasicSound::DetachExternalSoundPlayer( ExternalSoundPlayer* extPlayer )
{
    NW_NULL_ASSERT( extPlayer );
    NW_ASSERT( extPlayer == m_pExtSoundPlayer );
    (void)extPlayer;

    m_pExtSoundPlayer = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         GetRemainingFadeFrames

  Description:  フェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *---------------------------------------------------------------------------*/
int BasicSound::GetRemainingFadeFrames() const
{
    return m_FadeVolume.GetRemainingCount();
}

/*---------------------------------------------------------------------------*
  Name:         GetRemainingPauseFadeFrames

  Description:  一時停止フェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *---------------------------------------------------------------------------*/
int BasicSound::GetRemainingPauseFadeFrames() const
{
    return m_PauseFadeVolume.GetRemainingCount();
}

/*---------------------------------------------------------------------------*
  Name:         GetRemainingMuteFadeFrames

  Description:  ミュートフェードの残りフレームを取得する

  Arguments:    なし

  Returns:      残りフレーム
 *---------------------------------------------------------------------------*/
int BasicSound::GetRemainingMuteFadeFrames() const
{
    return m_MuteFadeVolume.GetRemainingCount();
}


/*---------------------------------------------------------------------------*
  Name:         SetPlayerPriority

  Description:  プレイヤープライオリティを設定する

  Arguments:    priority - プレイヤープライオリティ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetPlayerPriority( int priority )
{
    NW_MINMAX_ASSERT( priority, PRIORITY_MIN, PRIORITY_MAX );
    m_Priority = static_cast<u8>( priority );
    if ( m_pSoundPlayer != NULL )
    {
        m_pSoundPlayer->detail_SortPriorityList( this );
    }

    OnUpdatePlayerPriority();
}

/*---------------------------------------------------------------------------*
  Name:         SetInitialVolume

  Description:  初期ボリュームの設定

  Arguments:    volume - ボリューム

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetInitialVolume( f32 volume )
{
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_InitVolume = volume;
}
f32 BasicSound::GetInitialVolume() const
{
    return m_InitVolume;
}

/*---------------------------------------------------------------------------*
  Name:         SetVolume

  Description:  ボリュームの変更

  Arguments:    volume - ボリューム

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetVolume( f32 volume, int frames )
{
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_CommonParam.SetVolume(volume ,frames);

}
f32 BasicSound::GetVolume() const
{
    return m_CommonParam.GetVolume();
}

/*---------------------------------------------------------------------------*
  Name:         SetPitch

  Description:  音程の変更

  Arguments:    pitch - 変更する比率

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetPitch( f32 pitch )
{
    NW_ASSERT( pitch >= 0.0f );
    m_Pitch = pitch;
}
f32 BasicSound::GetPitch() const
{
    return m_Pitch;
}

/*---------------------------------------------------------------------------*
  Name:         SetLpfFreq

  Description:  ローパスフィルタのカットオフ周波数の変更

  Arguments:    lpfFreq - カットオフ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetLpfFreq( f32 lpfFreq )
{
    m_LpfFreq = lpfFreq;
}
f32 BasicSound::GetLpfFreq() const
{
    return m_LpfFreq;
}

/*---------------------------------------------------------------------------*
  Name:         SetBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    type  - フィルタの種類 (-1 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetBiquadFilter( int type, f32 value )
{
    m_BiquadFilterType = static_cast<s8>( type );
    m_BiquadFilterValue = value;
}
void BasicSound::GetBiquadFilter( int* type, f32* value ) const
{
    if ( type != NULL )
    {
        *type = m_BiquadFilterType;
    }
    if ( value != NULL )
    {
        *value = m_BiquadFilterValue;
    }
}

void BasicSound::SetOutputLine( u32 lineFlag )
{
    m_OutputLineFlag = lineFlag;
}

u32 BasicSound::GetOutputLine() const
{
    return m_OutputLineFlag;
}

void BasicSound::ResetOutputLine()
{
    m_OutputLineFlag = m_pSoundPlayer->GetDefaultOutputLine();
}


//
// 全体向け設定
//
void BasicSound::SetMixMode(MixMode mixMode)
{
    m_CommonParam.mixMode = mixMode;
}
MixMode BasicSound::GetMixMode()
{
    return m_CommonParam.mixMode;
}
void BasicSound::SetPan( f32 pan )
{
    m_CommonParam.pan = pan;
}
f32 BasicSound::GetPan() const
{
    return m_CommonParam.pan;
}

void BasicSound::SetSurroundPan( f32 span )
{
    m_CommonParam.span = span;
}
f32 BasicSound::GetSurroundPan() const
{
    return m_CommonParam.span;
}

void BasicSound::SetMainSend( f32 send )
{
    m_CommonParam.mainSend = send;
}
f32 BasicSound::GetMainSend() const
{
    return m_CommonParam.mainSend;
}

void BasicSound::SetFxSend( AuxBus bus, f32 send )
{
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );
    m_CommonParam.fxSend[bus] = send;
}
f32 BasicSound::GetFxSend( AuxBus bus ) const
{
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );
    return m_CommonParam.fxSend[bus];
}


//
// 出力先別設定
//
void BasicSound::SetOutputVolume(OutputDevice device, f32 volume)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    m_OutputParam[device].volume = volume;
}

void BasicSound::SetOutputChannelMixParameter(OutputDevice device, u32 srcChNo, MixParameter param)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    NW_ASSERT_MAXLT(srcChNo, WAVE_CHANNEL_MAX);

    for ( int i = 0; i < CHANNEL_INDEX_NUM; i++ )
    {
        m_OutputParam[device].mixParameter[srcChNo].ch[i] = param.ch[i];
    }
}

void BasicSound::SetOutputPan(OutputDevice device, f32 pan)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    m_OutputParam[device].pan = pan;
}

void BasicSound::SetOutputSurroundPan(OutputDevice device, f32 span)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    m_OutputParam[device].span = span;
}

void BasicSound::SetOutputMainSend(OutputDevice device, f32 send)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    m_OutputParam[device].mainSend = send;
}

void BasicSound::SetOutputFxSend(OutputDevice device, AuxBus bus, f32 send)
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    NW_ASSERT_MINMAXLT(bus, 0, AUX_BUS_NUM);
    m_OutputParam[device].fxSend[bus] = send;
}

f32 BasicSound::GetOutputVolume(OutputDevice device) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    return m_OutputParam[device].volume;
}
MixParameter BasicSound::GetOutputChannelMixParameter(OutputDevice device, u32 srcChNo) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    return m_OutputParam[device].mixParameter[srcChNo];
}
f32 BasicSound::GetOutputPan(OutputDevice device) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    return m_OutputParam[device].pan;
}
f32 BasicSound::GetOutputSurroundPan(OutputDevice device) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    return m_OutputParam[device].span;
}
f32 BasicSound::GetOutputMainSend(OutputDevice device) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    return m_OutputParam[device].mainSend;
}
f32 BasicSound::GetOutputFxSend(OutputDevice device, AuxBus bus) const
{
    NW_ASSERT_MINMAXLT(device, 0, OUTPUT_DEVICE_COUNT);
    NW_ASSERT_MINMAXLT(bus, 0, AUX_BUS_NUM);
    return m_OutputParam[device].fxSend[bus];
}


void BasicSound::SetRemoteOutVolume( u32 remoteIndex, f32 volume )
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    NW_ASSERT( volume >= 0.0f );

    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }
    m_RemoteParam[remoteIndex].volume = volume;
}
f32 BasicSound::GetRemoteOutVolume( u32 remoteIndex ) const
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    return m_RemoteParam[remoteIndex].volume;
}

void BasicSound::SetRemoteMainSend( u32 remoteIndex, f32 send )
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    m_RemoteParam[remoteIndex].mainSend = send;
}
f32 BasicSound::GetRemoteMainSend( u32 remoteIndex ) const
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    return m_RemoteParam[remoteIndex].mainSend;
}

void BasicSound::SetRemoteFxSend( u32 remoteIndex, f32 send )
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    m_RemoteParam[remoteIndex].fxSend = send;
}
f32 BasicSound::GetRemoteFxSend( u32 remoteIndex ) const
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    return m_RemoteParam[remoteIndex].fxSend;
}

void BasicSound::SetPanMode( PanMode mode )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerPanParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerPanParam>();
        command->id = DRIVER_COMMAND_PLAYER_PANMODE;
        command->player = GetBasicSoundPlayerHandle();
        command->panMode = static_cast<u8>(mode);
        cmdmgr.PushCommand(command);
    }
}

void BasicSound::SetPanCurve( PanCurve curve )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerPanParam* command =
            cmdmgr.AllocCommand<DriverCommandPlayerPanParam>();
        command->id = DRIVER_COMMAND_PLAYER_PANCURVE;
        command->player = GetBasicSoundPlayerHandle();
        command->panCurve = static_cast<u8>(curve);
        cmdmgr.PushCommand(command);
    }
}

void BasicSound::SetFrontBypass( bool isFrontBypass )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerFrontBypass* command =
            cmdmgr.AllocCommand<DriverCommandPlayerFrontBypass>();
        command->id = DRIVER_COMMAND_PLAYER_FRONT_BYPASS;
        command->player = GetBasicSoundPlayerHandle();
        command->isFrontBypass = isFrontBypass;
        cmdmgr.PushCommand(command);
    }
}

void BasicSound::SetRemoteFilter( u8 filter )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerRemoteFilter* command =
            cmdmgr.AllocCommand<DriverCommandPlayerRemoteFilter>();
        command->id = DRIVER_COMMAND_PLAYER_REMOTE_FILTER;
        command->player = GetBasicSoundPlayerHandle();
        command->remoteFilter = filter;
        cmdmgr.PushCommand(command);
    }
}

void BasicSound::SetVoiceRendererType( VoiceRendererType mode )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandPlayerRendererType* command =
            cmdmgr.AllocCommand<DriverCommandPlayerRendererType>();
        command->id = DRIVER_COMMAND_PLAYER_RENDERER_TYPE;
        command->player = GetBasicSoundPlayerHandle();
        command->rendererType = static_cast<u8>(mode);
        cmdmgr.PushCommand(command);
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetAmbientInfo

  Description:  アンビエント情報を設定する

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void BasicSound::SetAmbientInfo( const AmbientInfo& ambientArgInfo )
{
    // AmbientArg の複製を作成
    NW_NULL_ASSERT( ambientArgInfo.argAllocatorCallback );
    void* ambientArg =
        ambientArgInfo.argAllocatorCallback->detail_AllocAmbientArg( ambientArgInfo.argSize );
    if ( ambientArg == NULL )
    {
        NW_WARNING( ambientArg != NULL, "Failed to alloc AmbientArg." );
        return;
    }
    std::memcpy( ambientArg, ambientArgInfo.arg, ambientArgInfo.argSize );

    // AmbientInfo の設定
    m_AmbientInfo = ambientArgInfo;
    m_AmbientInfo.arg = ambientArg;
}

/*---------------------------------------------------------------------------*
  Name:         GetAmbientPriority [static]

  Description:  アンビエントプライオリティの取得

  Arguments:    ambientInfo - アンビエント情報
                soundId - サウンドID

  Returns:      アンビエントプライオリティ値を返す
  *---------------------------------------------------------------------------*/
int BasicSound::GetAmbientPriority( const AmbientInfo& ambientInfo, u32 soundId )
{
    if ( ambientInfo.paramUpdateCallback == NULL ) return 0;

    int priority = ambientInfo.paramUpdateCallback->detail_GetAmbientPriority(
        ambientInfo.arg,
        soundId
    );

    return priority;
}

// ハンドル関数
bool BasicSound::IsAttachedGeneralHandle()
{
    return m_pGeneralHandle != NULL;
}

bool BasicSound::IsAttachedTempGeneralHandle()
{
    return m_pTempGeneralHandle != NULL;
}

void BasicSound::DetachGeneralHandle()
{
    m_pGeneralHandle->DetachSound();
}

void BasicSound::DetachTempGeneralHandle()
{
    m_pTempGeneralHandle->DetachSound();
}

void BasicSound::SetId( u32 id )
{
    m_Id = id;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

