﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_BankFileReader.h>

namespace nw {
namespace snd {
namespace internal {

namespace
{

const u32 SIGNATURE_INFO_BLOCK_BANK  = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );

const u32 SUPPORTED_FILE_VERSION_BANK = 0x00010000;
const u32 CURRENT_FILE_VERSION_BANK   = 0x00010000;

bool IsValidFileHeaderBank( const void* bankFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader* header =
        reinterpret_cast<const ut::BinaryFileHeader*>( bankFile );
#else
    const BinaryFileHeader* header =
        reinterpret_cast<const BinaryFileHeader*>( bankFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header->signature == BankFileReader::SIGNATURE_FILE,
            "invalid file signature. bank file is not available." );
    if ( header->signature != BankFileReader::SIGNATURE_FILE ) return false;

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_BANK <= header->version) &&
                                   (header->version <= CURRENT_FILE_VERSION_BANK) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfbnk file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_BANK, CURRENT_FILE_VERSION_BANK, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace

BankFileReader::BankFileReader()
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL ),
  m_IsInitialized( false )
{
}

BankFileReader::BankFileReader( const void* bankFile )
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL ),
  m_IsInitialized( false )
{
    Initialize( bankFile );
}

void BankFileReader::Initialize( const void* bankFile )
{
    if ( bankFile == NULL )
    {
        return;
    }

    if ( ! IsValidFileHeaderBank( bankFile ) )
    {
        return;
    }

    m_pHeader = reinterpret_cast<const BankFile::FileHeader*>( bankFile );

    const BankFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NW_NULL_ASSERT( infoBlock );
    NW_ASSERT( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_BANK );

    if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_BANK )
    {
        return;
    }

    m_pInfoBlockBody = &infoBlock->body;

    m_IsInitialized = true;
}

void BankFileReader::Finalize()
{
    if ( m_IsInitialized )
    {
        m_pHeader = NULL;
        m_pInfoBlockBody = NULL;
        m_IsInitialized = false;
    }
}

bool BankFileReader::ReadVelocityRegionInfo(
        VelocityRegionInfo* info,
        int programNo,
        int key,
        int velocity ) const
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    // イレギュラーな programNo を指定された場合
    if ( programNo < 0 || programNo >= m_pInfoBlockBody->GetInstrumentCount() )
    {
        return false;
    }

    // ベロシティリージョンを取得
    const BankFile::Instrument* instrument =
        m_pInfoBlockBody->GetInstrument( programNo );
    if ( instrument == NULL )
    {
        return false;
    }

    const BankFile::KeyRegion* keyRegion = instrument->GetKeyRegion( key );
    if ( keyRegion == NULL )
    {
        return false;
    }

    const BankFile::VelocityRegion* velocityRegion =
        keyRegion->GetVelocityRegion( velocity );
    if ( velocityRegion == NULL )
    {
        return false;
    }

    // 波形アーカイブ ID / 波形アーカイブ内インデックスを取得
    NW_ASSERT( velocityRegion->waveIdTableIndex < m_pInfoBlockBody->GetWaveIdCount() );
    const Util::WaveId* pWaveId =
        m_pInfoBlockBody->GetWaveId( velocityRegion->waveIdTableIndex );

    if ( pWaveId == NULL )
    {
        return false;
    }

    if ( pWaveId->waveIndex == 0xffffffff )   // 当該リージョンに波形ファイルが割り当たっていない
    {
        NW_WARNING( false,
                "This region [programNo(%d) key(%d) velocity(%d)] is not assigned wave file.",
                programNo, key, velocity );
        return false;
    }

    // info への書き込み
    info->waveArchiveId     = pWaveId->waveArchiveId;
    info->waveIndex         = pWaveId->waveIndex;
    const BankFile::RegionParameter* regionParameter = velocityRegion->GetRegionParameter();
    if ( regionParameter == NULL )
    {
        info->originalKey       = velocityRegion->GetOriginalKey();
        info->volume            = velocityRegion->GetVolume();
        info->pan               = velocityRegion->GetPan();
    #ifdef NW_PLATFORM_RVL
        info->surroundPan       = velocityRegion->GetSurroundPan();
    #endif /* NW_PLATFORM_RVL */
        info->pitch             = velocityRegion->GetPitch();
        info->isIgnoreNoteOff   = velocityRegion->IsIgnoreNoteOff();
        info->keyGroup          = velocityRegion->GetKeyGroup();
        info->interpolationType = velocityRegion->GetInterpolationType();
        info->adshrCurve        = velocityRegion->GetAdshrCurve();
    }
    else
    {
        info->originalKey       = regionParameter->originalKey;
        info->volume            = regionParameter->volume;
        info->pan               = regionParameter->pan;
    #ifdef NW_PLATFORM_RVL
        info->surroundPan       = regionParameter->surroundPan;
    #endif /* NW_PLATFORM_RVL */
        info->pitch             = regionParameter->pitch;
        info->isIgnoreNoteOff   = regionParameter->isIgnoreNoteOff;
        info->keyGroup          = regionParameter->keyGroup;
        info->interpolationType = regionParameter->interpolationType;
        info->adshrCurve        = regionParameter->adshrCurve;
    }

    return true;
}

const Util::WaveIdTable*
BankFileReader::GetWaveIdTable() const
{
    if ( m_IsInitialized == false )
    {
        return NULL;
    }
    return &m_pInfoBlockBody->GetWaveIdTable();
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

