﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/config.h>
#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/snd_AnimSoundFileReader.h>

namespace nw {
namespace snd {
namespace internal {


namespace {

const u32 SIGNATURE_FILE       = NW_UT_MAKE_SIGWORD( 'F', 'A', 'S', 'D' );
// const u32 SIGNATURE_DATA_BLOCK = NW_UT_MAKE_SIGWORD( 'D', 'A', 'T', 'A' );

const u32 SUPPORTED_FILE_VERSION = 0x01000000;  // サポートする最低バージョン
const u32 CURRENT_FILE_VERSION   = 0x01000000;  // サポートする最新バージョン

bool IsValidFileHeader( const void* bfasdFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( bfasdFile );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( bfasdFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header.signature == SIGNATURE_FILE, "invalid file signature." );
    if ( header.signature != SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfasd file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION, CURRENT_FILE_VERSION, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

AnimSoundFileReader::AnimSoundFileReader()
: m_pAnimEventTable( NULL ),
  m_FrameSize( 0 )
{}

bool AnimSoundFileReader::Initialize( const void* bfasdFile )
{
    if ( ! IsValidFileHeader( bfasdFile ) )
    {
        return false;
    }

    const AnimSoundFile::FileHeader* header =
        reinterpret_cast<const AnimSoundFile::FileHeader*>( bfasdFile );

    const AnimSoundFile::DataBlockBody* body = &header->GetDataBlock()->body;
    NW_NULL_ASSERT( body );

    if ( ! body )
    {
        return false;
    }

    const AnimSoundFile::AnimEventTable* table = body->GetAnimEventTable();
    NW_NULL_ASSERT( table );
    if ( ! table )
    {
        return false;
    }

    m_pAnimEventTable = table;
    m_FrameSize = body->frameSize;
    return true;
}

void AnimSoundFileReader::Finalize()
{
    m_pAnimEventTable = NULL;
    m_FrameSize = 0;
}

void AnimSoundFileReader::Dump()
{
#ifdef NW_CONSOLE_ENABLE
    NW_LOG("*** %s begin ***\n", __FUNCTION__);
    NW_LOG("FrameSize : %3d\n", GetFrameSize());

    u32 eventCount = GetEventCount();
    NW_LOG("EventCount: %3d\n", eventCount);
    for ( u32 i = 0; i < eventCount; i++ )
    {
        const AnimSoundFile::AnimEvent* event = GetAnimEvent( i );
        if ( event == NULL )
        {
            NW_LOG("[%3d] event is null\n", i);
        }
        else
        {
            const AnimSoundFile::FrameInfo* frameInfo = &event->frameInfo;
            u8 frameFlag = frameInfo->frameFlag;
            // PC 版でうまく表示されないので一旦 auto 変数に格納する
            s32 startFrame = frameInfo->startFrame;
            s32 endFrame = frameInfo->endFrame;
            NW_LOG("[%3d] start(%d) end(%d) eventType(%s)\n",
                    i, startFrame, endFrame,
                    (frameFlag & AnimSoundFile::FrameInfo::FRAME_FLAG_TRIGGER_EVENT) ?
                        "TRIG" : "RANGE");

            const AnimSoundFile::EventInfo* eventInfo = event->GetEventInfo();
            u32 optionFlag = eventInfo->optionFlag;
            f32 pitch = eventInfo->pitch;
            u32 userParam = eventInfo->userParam;
            NW_LOG("      (%s) vol(%d) pitch(%2.6f) userParam(0x%08x)\n",
                    eventInfo->GetSoundLabel(), eventInfo->volume,
                    pitch, userParam);
            NW_LOG("      notStopSoundWhenAnimFinish?(%d) seqVarEnable?(%d) seqVar(%d)\n",
                    (optionFlag & AnimSoundFile::EventInfo::
                        OPTION_FLAG_IS_NOT_STOP_SOUND_WHEN_ANIMATION_FINISH) ? 1 : 0,
                    (optionFlag & AnimSoundFile::EventInfo::
                        OPTION_FLAG_IS_ENABLE_SEQUENCE_VARIABLE) ? 1 : 0,
                    eventInfo->sequenceVariableNo );
            const char* PLAY_DIRECTION_STRING[] = { "BOTH", "FORWARD", "BACKWARD" };
            NW_LOG("      playDirection(%s)\n",
                    PLAY_DIRECTION_STRING[eventInfo->playDirection] );

        }
    }
    NW_LOG("*** %s end *****\n", __FUNCTION__);
#endif // NW_CONSOLE_ENABLE
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

