﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_AnimEventPlayer.h>
#include <nw/snd/snd_SequenceSoundHandle.h>
#include <nw/snd/snd_SoundStartable.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace snd {
namespace internal {

namespace
{

s16 ConvertSpeedF32ToS16( float speed )
{
    static const s32 S16_SPEED_MIN = -32768;
    static const s32 S16_SPEED_MAX = 32767;
    s32 s32Speed = static_cast<s32>( speed * 100 );
    s16 s16Speed = static_cast<s16>( ut::Clamp( s32Speed, S16_SPEED_MIN, S16_SPEED_MAX ) );
    return s16Speed;
}

} // anonymous namespace

AnimEventPlayer::AnimEventPlayer()
: m_pEventInfo( NULL ),
  m_IsStopWhenFinalize( false )
{
}

AnimEventPlayer::~AnimEventPlayer()
{
    Finalize();
}

void AnimEventPlayer::InitParam(
        const AnimSoundFile::EventInfo& eventInfo, bool isStopWhenFinalize )
{
    // 初期パラメータ設定
    m_Handle.SetVolume( static_cast<f32>( eventInfo.volume ) / 127.0f );

    if ( eventInfo.pitch != 1.0f )
    {
        m_Handle.SetPitch( eventInfo.pitch );
    }
    m_pEventInfo = &eventInfo;
    m_IsStopWhenFinalize =
        isStopWhenFinalize && (! eventInfo.IsNotStopSoundWhenAnimationFinish());
}

void AnimEventPlayer::Finalize()
{
    // 再生中のサウンドを停止する
    if ( m_Handle.IsAttachedSound() && m_IsStopWhenFinalize )
    {
        m_Handle.Stop( 0 );
    }
}

bool AnimEventPlayer::StartEvent(
    const AnimSoundFile::EventInfo& eventInfo,
    SoundStartable& starter,
    bool isStopWhenFinalize
)
{
    // 再生
    if ( eventInfo.placeForSoundId == SoundArchive::INVALID_ID )
    {
        const char* soundLabel = eventInfo.GetSoundLabel();
        if ( ! starter.StartSound( &m_Handle, soundLabel ).IsSuccess() )
        {
            return false;
        }
    }
    else
    {
        if ( ! starter.StartSound( &m_Handle, eventInfo.placeForSoundId ).IsSuccess() )
        {
            return false;
        }
    }

    InitParam( eventInfo, isStopWhenFinalize );
    return true;
}

bool AnimEventPlayer::HoldEvent(
    const AnimSoundFile::EventInfo& eventInfo,
    SoundStartable& starter,
    bool isStopWhenFinalize
)
{
    // 再生
    if ( eventInfo.placeForSoundId == SoundArchive::INVALID_ID )
    {
        const char* soundLabel = eventInfo.GetSoundLabel();
        if ( ! starter.HoldSound( &m_Handle, soundLabel ).IsSuccess() )
        {
            return false;
        }
    }
    else
    {
        if ( ! starter.HoldSound( &m_Handle, eventInfo.placeForSoundId ).IsSuccess() )
        {
            return false;
        }
    }

    // HoldSoundの自動停止をやめる
    m_Handle.detail_GetAttachedSound()->SetAutoStopCounter( 0 );

    InitParam( eventInfo, isStopWhenFinalize );
    return true;
}

void AnimEventPlayer::ForceStop()
{
    m_Handle.Stop( 0 );
    m_pEventInfo = NULL;
}

void AnimEventPlayer::UpdateFrame()
{
    // 自動停止をチェック
    if ( ! m_Handle.IsAttachedSound() )
    {
        m_pEventInfo = NULL;
    }
}

void AnimEventPlayer::WritePlaySpeedToSequenceVariable( u8 sequenceVariableNo, f32 speed )
{
    // ローカル変数反映
    if ( sequenceVariableNo < 16 )
    {
        if ( ! m_Handle.IsAttachedSound() )
        {
            return;
        }
        SequenceSoundHandle handle( &m_Handle );
        handle.WriteVariable( sequenceVariableNo, ConvertSpeedF32ToS16( speed ) );
    }
    // グローバル変数反映
    else if ( sequenceVariableNo < 32 )
    {
        SequenceSoundHandle::WriteGlobalVariable(
                sequenceVariableNo - 16,
                ConvertSpeedF32ToS16( speed ) );
    }
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

