﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_WavStream.h>

#include <nw/snd/fnd/basis/sndfnd_Endian.h>
#include <nw/snd/fnd/basis/sndfnd_Alignment.h>
#include <nw/snd/fnd/io/sndfnd_FileStream.h>
#include <nw/snd/fnd/binary/sndfnd_WavBinary.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

template<typename TChunk>
bool ReadChunkBodyWav(Stream* stream, TChunk& chunk)
{
    NW_NULL_ASSERT(stream);

    u32 size = sizeof(TChunk) - sizeof(ChunkHeader);

    return stream->Read(
        ut::AddOffsetToPtr(&chunk, sizeof(ChunkHeader)),
        size) == size;
}

template<typename TSample>
void EndianSwapWav(TSample* samples, u32 sampleCount)
{
#if defined(NW_SND_FND_IS_BIG_ENDIAN)
    // ビッグエンディアンの場合、.wav のサンプル値（LE）をエンディアン変換する
    for( u32 i = 0; i < sampleCount; ++i)
    {
        samples[i] = nw::snd::internal::fnd::endian::ByteSwap(samples[i]);
    }
#else
    (void)samples;
    (void)sampleCount;
#endif
}

}

//---------------------------------------------------------------------------
WavStreamReader::WavStreamReader() :
m_Stream(NULL),
m_BlockAlignment(0),
m_Padding1(0),
m_Padding2(0),
m_DataOffset(0),
m_DataSize(0),
m_CurrentDataPosition(Stream::INVALID_POSITION)
{
}

WavStreamReader::WavStreamReader(Stream* stream) :
m_Stream(NULL),
m_BlockAlignment(0),
m_Padding1(0),
m_Padding2(0),
m_DataOffset(0),
m_DataSize(0),
m_CurrentDataPosition(Stream::INVALID_POSITION)
{
    Open(stream);
}

//---------------------------------------------------------------------------
FndResult
WavStreamReader::Open(Stream* stream)
{
    NW_ASSERT_NOT_NULL(stream);
    NW_ASSERTMSG(stream->IsOpened(), "stream must be opened.\n");

    if(!stream->CanSeek())
    {
        NW_ASSERTMSG(false, "not seekable stream is not supported.\n");
        return FndResult(SNDFND_RESULT_INVALID_ARGUMENT);
    }

    m_Stream = stream;
    m_DataOffset = 0;
    m_DataSize = 0;
    m_CurrentDataPosition = Stream::INVALID_POSITION;

    return FndResult(SNDFND_RESULT_TRUE);
}

//---------------------------------------------------------------------------
void
WavStreamReader::Close()
{
    m_Stream = NULL;
}

//---------------------------------------------------------------------------
FndResult
WavStreamReader::ReadHeader()
{
    if(!IsOpened())
    {
        NW_ASSERTMSG(false, "WavStreamReader must be opened.\n");
        return FndResult(SNDFND_RESULT_NOT_OPENED);
    }

    NW_ASSERT(m_DataOffset == 0);
    NW_ASSERT(m_DataSize == 0);

    FndResult result = ReadRiffChunk();

    if(!result.IsTrue())
    {
        return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
    }

    bool isFmtChunkFound = false;
    const u32 fileSize = m_Stream->GetSize();
    u32 currentPosition = sizeof(RiffChunk);

    for ( ;; )
    {
        ChunkHeader chunkHeader;
        m_Stream->Read(&chunkHeader, sizeof(ChunkHeader), &result);

        if(!result.IsTrue())
        {
            return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
        }

        currentPosition += sizeof(ChunkHeader);

        switch(chunkHeader.id)
        {
        case FmtChunk::ValidID:
            if(chunkHeader.size == FmtChunk::GetValidBodySize())
            {
                FmtChunk fmtChunk;
                if(!ReadChunkBodyWav(m_Stream, fmtChunk))
                {
                    return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
                }

                m_WaveFormat.channels = static_cast<u8>(fmtChunk.channels);
                m_WaveFormat.bitsPerSample = static_cast<u8>(fmtChunk.bitsPerSample);
                m_WaveFormat.samplesPerSec = fmtChunk.samplesPerSec;
                m_BlockAlignment = static_cast<u8>(fmtChunk.channels) * static_cast<u8>(fmtChunk.bitsPerSample) / 8;

                isFmtChunkFound = true;

                currentPosition += FmtChunk::GetValidBodySize();

                // チャンクデータを読み込み済みなので continue します。
                continue;
            }
            break;

        case DataChunk::ValidID:
            m_DataOffset = currentPosition;
            m_DataSize = chunkHeader.size;
            break;
        }

        // 次のチャンク開始位置がファイル終端なら終了します。
        if(fileSize <= m_Stream->GetCurrentPosition() + chunkHeader.size)
        {
            break;
        }

        // 次のチャンクまでシークします。
        result = m_Stream->Seek(chunkHeader.size, Stream::SEEK_ORIGIN_CURRENT);

        if(result.IsFailed())
        {
            return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
        }

        currentPosition += chunkHeader.size;
    }

    if(!isFmtChunkFound ||
        (m_DataOffset == 0 || fileSize <= m_DataOffset) ||
        m_DataSize == 0)
    {
        return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG("[%s] Channels       : %d\n", __FUNCTION__, m_WaveFormat.channels);
    NW_LOG("[%s] SamplesPerSec  : %d\n", __FUNCTION__, m_WaveFormat.samplesPerSec);
    NW_LOG("[%s] BitsPerSample  : %d\n", __FUNCTION__, m_WaveFormat.bitsPerSample);
    NW_LOG("[%s] DataSize       : %d\n", __FUNCTION__, m_DataSize);
#endif

    return FndResult(SNDFND_RESULT_TRUE);
}

//---------------------------------------------------------------------------
u32
WavStreamReader::ReadFrames(void* buffer, u32 frames, FndResult* result /*= NULL*/)
{
    if(!IsOpened() || !IsHeaderRead())
    {
        NW_ASSERTMSG(false, "WavStreamReader must be opened.\n");

        if(result != NULL)
        {
            *result = FndResult(SNDFND_RESULT_NOT_OPENED);
        }

        return 0;
    }

    // 初めて読み込む場合は、データ開始位置にシークします。
    if(m_CurrentDataPosition == Stream::INVALID_POSITION)
    {
        FndResult seekResult = m_Stream->Seek(m_DataOffset, Stream::SEEK_ORIGIN_BEGIN);

        if(seekResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = seekResult;
            }

            return 0;
        }

        m_CurrentDataPosition = 0;
    }

    u32 tryReadBytes = ut::Min(m_DataSize - m_CurrentDataPosition, frames * m_BlockAlignment);

    if(tryReadBytes == 0)
    {
        if(result != NULL)
        {
            *result = FndResult(SNDFND_RESULT_FALSE);
        }

        return 0;
    }

    FndResult readResult(SNDFND_RESULT_TRUE);
    u32 readBytes = m_Stream->Read(buffer, tryReadBytes, &readResult);

    m_CurrentDataPosition += readBytes;

    if(result != NULL)
    {
        *result = readResult;
    }

    switch(m_WaveFormat.bitsPerSample)
    {
    case 16:
        EndianSwapWav<s16>(static_cast<s16*>(buffer), readBytes / sizeof(s16));
        break;

    case 32:
        EndianSwapWav<s32>(static_cast<s32*>(buffer), readBytes / sizeof(s32));
        break;
    }

    return readBytes / m_BlockAlignment;
}

//---------------------------------------------------------------------------
FndResult
WavStreamReader::ReadRiffChunk()
{
    FndResult result(SNDFND_RESULT_TRUE);

    RiffChunk riffChunk;
    m_Stream->Read(&riffChunk, sizeof(RiffChunk), &result);

    if(!result.IsTrue())
    {
        return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
    }

    if(!riffChunk.IsValid())
    {
        return FndResult(SNDFND_RESULT_NOT_SUPPORTED);
    }

    return result;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
