﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_HioFileStreamImpl.h>

#include <nw/snd/fnd/io/sndfnd_File.h>

#if defined(NW_PLATFORM_CTR)

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::Read(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Read(buf, length, result);
    }

    return ReadDirect(buf, length, result);
}

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::Write(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Write(buf, length, result);
    }

    return WriteDirect(buf, length, result);
}

//---------------------------------------------------------------------------
FndResult
HioFileStreamImpl::Seek(s32 offset, SeekOrigin origin)
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Seek(offset, origin);
    }

    return SeekDirect(offset, origin);
}

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::GetCurrentPosition() const
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.GetCurrentPosition();
    }

    return m_CurrentPosition;
}

//---------------------------------------------------------------------------
bool
HioFileStreamImpl::CanRead() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
HioFileStreamImpl::CanWrite() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
HioFileStreamImpl::CanSeek() const
{
    return IsOpened();
}

//---------------------------------------------------------------------------
void
HioFileStreamImpl::EnableCache(void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);

    if(m_StreamCache.IsInitialized())
    {
        m_StreamCache.Finalize();
    }

    void* alignedBuffer = ut::RoundUp(buffer, FileTraits::IO_BUFFER_ALIGNMENT);

    m_StreamCache.Initialize(
        &m_DirectStream,
        alignedBuffer,
        length - ut::GetOffsetFromPtr(buffer, alignedBuffer));
}

//---------------------------------------------------------------------------
void
HioFileStreamImpl::DisableCache()
{
    m_StreamCache.Finalize();
}

//---------------------------------------------------------------------------
void
HioFileStreamImpl::ValidateAlignment(const void* buf) const
{
    if(ut::RoundUp(buf, FileTraits::IO_BUFFER_ALIGNMENT) != buf)
    {
        NW_FATAL_ERROR(
            "invalid alignment : buf(0x%08x) must be aligned by %d.\n",
            buf,
            FileTraits::IO_BUFFER_ALIGNMENT);
    }
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif
