﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
u32
GetSeekPosition(FileStreamImpl& target, s32 offset, FileStreamImpl::SeekOrigin origin)
{
    u32 result = 0;

    switch(origin)
    {
    case FileStreamImpl::SEEK_ORIGIN_BEGIN:
        result = offset <= 0 ? 0 : offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_END:
        result = offset >= 0 ? target.GetSize() : target.GetSize() - offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_CURRENT:
        if(offset > 0)
        {
            result = target.GetCurrentPosition() + static_cast<u32>(offset);

            u32 size = target.GetSize();

            if(result > size)
            {
                result = size;
            }
        }
        else if(offset < 0)
        {
            if(target.GetCurrentPosition() > static_cast<u32>(-offset))
            {
                result = target.GetCurrentPosition() - static_cast<u32>(-offset);
            }
            else
            {
                result = 0;
            }
        }
        else
        {
            result = target.GetCurrentPosition();
        }
        break;

    default:
        NW_FATAL_ERROR("invalid seek origin.\n");
        break;
    }

    return result;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() :
m_FsClient(NULL),
m_FsCmdBlock(NULL),
m_FsFileHandle(FS_INVALID_HANDLE_VALUE),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl(FSClient* fsClient, FSCmdBlock* fsCmdBlock, FSFileHandle fsFileHandle) :
m_FsClient(fsClient),
m_FsCmdBlock(fsCmdBlock),
m_FsFileHandle(fsFileHandle),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    NW_ASSERT_NOT_NULL(fsClient);
    NW_ASSERT_NOT_NULL(fsCmdBlock);
    NW_ASSERT(fsFileHandle != FS_INVALID_HANDLE_VALUE);
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Open(FSClient* fsClient, FSCmdBlock* fsCmdBlock, FSFileHandle fsFileHandle)
{
    NW_ASSERT_NOT_NULL(fsClient);
    NW_ASSERT_NOT_NULL(fsCmdBlock);
    NW_ASSERTMSG(!IsOpened(), "FileStreamImpl is already opened.\n");

    m_FsClient = fsClient;
    m_FsCmdBlock = fsCmdBlock;
    m_FsFileHandle = fsFileHandle;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close()
{
    if(!IsOpened())
    {
        return;
    }

    FSCloseFile(m_FsClient, m_FsCmdBlock, m_FsFileHandle, FS_RET_NO_ERROR);

    m_FsClient = NULL;
    m_FsCmdBlock = NULL;
    m_FsFileHandle = FS_INVALID_HANDLE_VALUE;
    m_FileSize = INVALID_SIZE;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const
{
    return m_FsClient != NULL;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    if(m_FileSize == INVALID_SIZE)
    {
        FSStat fsStat;

        if(FSGetStatFile(m_FsClient, m_FsCmdBlock, m_FsFileHandle, &fsStat, 0) != FS_STATUS_OK)
        {
            return 0;
        }

        m_FileSize = fsStat.size;
    }

    return m_FileSize;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    FndResult readResult(SNDFND_RESULT_TRUE);
    u32 readFileLength = 0;

    FSStatus fsReadResult = FSReadFile(m_FsClient, m_FsCmdBlock, buf, 1, length, m_FsFileHandle, 0, FS_RET_ALL_ERROR);

    if(fsReadResult < 0)
    {
        switch(fsReadResult)
        {
        case FS_STATUS_ACCESS_ERROR:
            readResult = FndResult(SNDFND_RESULT_IO_INVALID_ACCESS);
            break;

        default:
            readResult = FndResult(SNDFND_RESULT_IO_ERROR);
            break;
        }
    }
    else
    {
        readFileLength = static_cast<u32>(fsReadResult);
        readResult = length == readFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readFileLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSReadFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, fsReadResult);
#endif

    return readFileLength;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    u32 writtenLength = 0;
    FndResult writeResult(SNDFND_RESULT_TRUE);

    m_FileSize = INVALID_SIZE;

    FSStatus fsWriteResult = FSWriteFile(m_FsClient, m_FsCmdBlock, buf, 1, length, m_FsFileHandle, 0, FS_RET_ALL_ERROR);

    if(fsWriteResult < 0)
    {
        switch(fsWriteResult)
        {
        case FS_STATUS_ACCESS_ERROR:
            writeResult = FndResult(SNDFND_RESULT_IO_INVALID_ACCESS);
            break;

        default:
            writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
            break;
        }
    }
    else
    {
        writtenLength = static_cast<u32>(fsWriteResult);
        writeResult = length == writtenLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += writtenLength;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSWriteFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, fsWriteResult);
#endif

    return writtenLength;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    u32 seekPosition = GetSeekPosition(*this, offset, origin);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] try FSSetPosFile : curPos=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, seekPosition);
#endif

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_ASSERTMSG(
        seekPosition < GetSize(),
        "[%s][%08x] seek to out of range : curPos=0x%08x, size=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, GetSize(), seekPosition);
#endif

    FSStatus fsSeekResult = FSSetPosFile(
        m_FsClient,
        m_FsCmdBlock,
        m_FsFileHandle,
        seekPosition,
        FS_RET_ALL_ERROR
        );

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSSetPosFile : curPos=%08x, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, fsSeekResult);
#endif

    if(fsSeekResult != FS_STATUS_OK)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    m_CurrentPosition = seekPosition;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
