﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/snd/aac/sndaac_AacDecoder.h>
#include <nw/snd/snd_SoundArchive.h>

//#define NW_SND_AAC_ENABLE_DEBUG_DUMP

namespace
{
    static const u32 AACDEC_BLOCK_SAMPLE_COUNT = HEAACDEC_PCM_FRAME_LENGTH;
    static const u32 AACDEC_INPUT_STREAM_MAX_SIZE = HEAACDEC_STREAM_MAX_SIZE;
    static const u32 AACDEC_PCM_SIZE = AACDEC_BLOCK_SAMPLE_COUNT * sizeof(s16);
    //static const u32 AACDEC_CHANNEL_MAX = 2;
    static const u32 ADTS_HEADER_SIZE = 7;
    static const u32 ADTS_HEADER_EXT_SIZE = 2;
    static const u32 DECODE_BLOCK_ONCE_NUM = 1;

    static const u32 SAMPLING_RATE[] =
    {
        96000, 88200, 64000, 48000, 44100, 32000, 24000, 22050, 16000, 12000, 11025, 8000
    };

    struct HeaderInfo
    {
        HeaderInfo()
        : headerSize(0)
        , channelCount(0)
        , sampleRateIndex(0)
        , frameLength(0)
        , isExistCRC(false)
        {}

        u32 headerSize;
        u32 channelCount;
        u32 sampleRateIndex;
        u32 frameLength;
        bool isExistCRC;
    };

    u32 GetFrameLength(const u8* adtsHeader)
    {
        return ((adtsHeader[3] & 0x03) << 11) | (adtsHeader[4] << 3) | ((adtsHeader[5] & 0xE0) >> 5);
    }

    u32 GetChannelCount(const u8* adtsHeader)
    {
        return static_cast<u32>( ((adtsHeader[2] << 2) & 0x04) | ((adtsHeader[3] >> 6) & 0x03) );
    }

    u32 GetSampleRateIndex(const u8* adtsHeader)
    {
        return static_cast<u32>( (adtsHeader[2] >> 2) & 0x0F );
    }

    bool IsExistCRC(const u8* adtsHeader)
    {
        return (adtsHeader[1] & 0x01) == 0x00;
    }

#ifdef NW_SND_AAC_ENABLE_DEBUG_DUMP
    void DebugDumpHeader(const u8* adtsHeader)
    {
        NW_LOG("[ADTS]");
        for (s32 i = 0; i < 7; ++i)
        {
            NW_LOG("%x ", adtsHeader[i]);
        }
        NW_LOG("FreqIndex(%d)\n", (adtsHeader[2] >> 2) & 0x0F);
    }
#endif

    void ReadAdtsHeader(nw::ut::FileStream* fileStream, u8* inputBitStream, HeaderInfo* info)
    {
        NW_ASSERT_NOT_NULL(info);

        fileStream->Read( inputBitStream, ADTS_HEADER_SIZE );
#ifdef NW_SND_AAC_ENABLE_DEBUG_DUMP
        DebugDumpHeader(inputBitStream);
#endif

        info->headerSize = ADTS_HEADER_SIZE;
        info->channelCount = GetChannelCount(inputBitStream);
        info->sampleRateIndex = GetSampleRateIndex(inputBitStream);
        info->isExistCRC = IsExistCRC(inputBitStream);
        info->frameLength = GetFrameLength(inputBitStream);
        if (info->isExistCRC)
        {
            fileStream->Read( inputBitStream, ADTS_HEADER_EXT_SIZE );
            info->headerSize += ADTS_HEADER_EXT_SIZE;
        }
        info->frameLength -= info->headerSize;
    }

    HEAACDEC_AAC_INFO s_AacInfo;
    nw::snd::aac::internal::AacDecoder s_AacDecorder;
    u32 s_StreamNum = 1;
}

namespace nw {
namespace snd {
namespace aac {

namespace internal {

void AacDecoder::Initialize( void* buffer, u32 size, u32 streamNum )
{
    NW_ALIGN32_ASSERT( buffer );
    NW_ASSERT( m_WorkBufferSize != 0 );

    m_pInputDataBuffer = nw::ut::RoundUp(buffer, 64);
    buffer = nw::ut::RoundUp(ut::AddOffsetToPtr(m_pInputDataBuffer, AACDEC_INPUT_STREAM_MAX_SIZE), 64);
    m_WorkBufferHeap.Initialize(buffer, nw::ut::RoundUp(m_WorkBufferSize, 64) * streamNum, m_WorkBufferSize, streamNum, 64);
}

void AacDecoder::Finalize()
{
    m_WorkBufferHeap.Finalize();
    m_pInputDataBuffer = NULL;

    m_pAacInfo = NULL;
    m_WorkBufferSize = 0;
}

u32 AacDecoder::GetAacDecBufferSize( HEAACDEC_AAC_INFO* aacInfo )
{
    NW_ASSERT_NOT_NULL( aacInfo );
    m_pAacInfo = aacInfo;

    s32 bufferSize = 0;
    s32 result = HEAACDECMemoryRequirement(m_pAacInfo, &bufferSize);
    if (result != HEAACDEC_RET_OK)
    {
        switch(result)
        {
        case HEAACDEC_RET_BUFFER_ERR:
            NW_ASSERTMSG(false, "HEAACDEC_RET_BUFFER_ERR\n");
            break;
        case HEAACDEC_RET_ERR_PARAM:
            NW_ASSERTMSG(false, "HEAACDEC_RET_ERR_PARAM\n");
            break;
        case HEAACDEC_RET_STREAM_ERR:
            NW_ASSERTMSG(false, "HEAACDEC_RET_STREAM_ERR\n");
            break;
        case HEAACDEC_RET_INVALID_ERR:
            NW_ASSERTMSG(false, "HEAACDEC_RET_INVALID_ERR\n");
            break;
        default:
            NW_ASSERT(false);
            break;
        }
    }

    m_WorkBufferSize = bufferSize;

    return nw::ut::RoundUp(bufferSize, 64);
}

void* AacDecoder::AllocWorkBuffer()
{
    return m_WorkBufferHeap.Alloc();
}

void AacDecoder::FreeWorkBuffer(void* buffer)
{
    m_WorkBufferHeap.Free(buffer);
}

bool AacDecoder::ReadDataInfo( ut::FileStream* fileStream, DataInfo* info, void* workBuffer )
{
    NW_ASSERT_NOT_NULL( m_pAacInfo );
    if ( m_pInputDataBuffer == NULL )
    {
        NW_WARNING( m_pInputDataBuffer != NULL, "nw::snd::aac is not initialized");
        return false;
    }

    fileStream->Seek( 0, ut::FILE_STREAM_SEEK_BEGIN );
    u8* inputBitStream = static_cast<u8*>(m_pInputDataBuffer);
    HeaderInfo headerInfo;
    ReadAdtsHeader(fileStream, inputBitStream, &headerInfo);
    m_pAacInfo->smpFreq = headerInfo.sampleRateIndex;
    m_pAacInfo->channel = headerInfo.channelCount;

    s32 result = HEAACDECClose(workBuffer);
    if (result != HEAACDEC_RET_OK)
    {
        NW_WARNING( false, "HEAACDECClose is Failed (%d)", result );
        //NW_ASSERTMSG( false, "HEAACDECClose is Failed");
        return false;
    }

    result = HEAACDECOpen(workBuffer, m_WorkBufferSize, m_pAacInfo);
    if (result != HEAACDEC_RET_OK)
    {
        NW_WARNING( false, "HEAACDECOpen is Failed (%d)", result );
        //NW_ASSERTMSG( false, "HEAACDECOpen is Failed" );
        return false;
    }

    u32 readSize = fileStream->Read( inputBitStream, headerInfo.frameLength );
    if (readSize <= 0)
    {
        NW_ASSERTMSG( false, "AAC FileRead is Failed" );
        return false;
    }

    result = HEAACDECBegin(workBuffer, inputBitStream, headerInfo.frameLength);
    if (result != HEAACDEC_RET_OK)
    {
        NW_WARNING( false, "HEAACDECBegin is Failed (%d)", result );
        //NW_ASSERTMSG( false, "HEAACDECBegin is Failed");
    }

    info->channelCount = headerInfo.channelCount;
    NW_ASSERT(headerInfo.sampleRateIndex < 12);
    info->sampleRate = SAMPLING_RATE[headerInfo.sampleRateIndex];
    info->blockSampleCount = AACDEC_BLOCK_SAMPLE_COUNT * DECODE_BLOCK_ONCE_NUM;
    info->blockSize = AACDEC_BLOCK_SAMPLE_COUNT * DECODE_BLOCK_ONCE_NUM * sizeof(s16);

    return true;
}

bool AacDecoder::Skip( ut::FileStream* fileStream )
{
    for (s32 i = 0; i < DECODE_BLOCK_ONCE_NUM; ++i)
    {
        if (!SingleBlockSkip(fileStream))
        {
            return false;
        }
    }

    return true;
}

bool AacDecoder::Decode( ut::FileStream* fileStream, int channelCount, DecodeType decodeType, s16* decodedDataArray[], void* workBuffer )
{
    if ( m_pInputDataBuffer == NULL )
    {
        NW_WARNING( m_pInputDataBuffer != NULL, "nw::snd::aac is not initialized");
        return false;
    }

    u8* inputBitStream = static_cast<u8*>(m_pInputDataBuffer);

    // MultiBlockDecode内部でdecodedDataArrayのポインタを書き換えるのでコピーを渡す
    s16* dataArray[nw::snd::internal::STRM_CHANNEL_NUM];
    for( unsigned int i=0; i<channelCount; i++ )
    {
        dataArray[i] = decodedDataArray[i];
    }
    MultiBlockDecode(fileStream, inputBitStream, channelCount, decodeType, dataArray, workBuffer);

    return true;
}

void AacDecoder::MultiBlockDecode(ut::FileStream* fileStream, u8* inputBitStream, int channelCount, DecodeType decodeType, s16* decodedDataArray[], void* workBuffer)
{
    NW_ASSERT_NOT_NULL(inputBitStream);

    bool isLoop = decodeType == DECODE_TYPE_LOOP;
    s32 roopCount = isLoop ? DECODE_BLOCK_ONCE_NUM * 2 : DECODE_BLOCK_ONCE_NUM;

    bool isIdling = decodeType == DECODE_TYPE_IDLING;
    if (isIdling)
    {
        // N個の最後のブロックへ移動
        for (s32 i = 0; i < DECODE_BLOCK_ONCE_NUM - 1; ++i)
        {
            if (!SingleBlockSkip(fileStream))
            {
                return;
            }
        }

        SingleBlockDecode(fileStream, inputBitStream, decodedDataArray, workBuffer);
    }
    else
    {
        for (s32 i = 0; i < roopCount; ++i)
        {
            SingleBlockDecode(fileStream, inputBitStream, decodedDataArray, workBuffer);

            for (s32 ch = 0; ch < channelCount; ++ch)
            {
                decodedDataArray[ch] = static_cast<s16*>(ut::AddOffsetToPtr(decodedDataArray[ch], AACDEC_PCM_SIZE));
            }
        }
    }
}

void AacDecoder::SingleBlockDecode(ut::FileStream* fileStream, u8* inputBitStream, s16* decodedDataArray[], void* workBuffer)
{
    HeaderInfo headerInfo;
    ReadAdtsHeader(fileStream, inputBitStream, &headerInfo);

    u32 readSize = fileStream->Read(inputBitStream, headerInfo.frameLength);
    if (readSize <= 0)
    {
        NW_ASSERTMSG( false, "AAC FileRead is Failed" );
        return;
    }

    u16 pcmLength;
    u16 outputFreqIndex;
    s32 result = HEAACDECExecute(workBuffer, inputBitStream, headerInfo.frameLength, decodedDataArray, &pcmLength, &outputFreqIndex);
    if (result != HEAACDEC_RET_OK)
    {
        NW_WARNING( false, "HEAACDECExecute is Failed (%d)", result );
        //NW_ASSERTMSG( false, "HEAACDECExecute is Failed" );
        return;
    }
}

bool AacDecoder::SingleBlockSkip(ut::FileStream* fileStream)
{
    u32 pos = fileStream->Tell();

    u8* inputBitStream = static_cast<u8*>(m_pInputDataBuffer);
    HeaderInfo headerInfo;
    ReadAdtsHeader(fileStream, inputBitStream, &headerInfo);

    if ( headerInfo.frameLength <= 0 )
    {
        return false;
    }

    fileStream->Seek(pos + headerInfo.headerSize + headerInfo.frameLength, ut::FILE_STREAM_SEEK_BEGIN);

    return true;
}

} // namespace internal


void Initialize( void* buffer, u32 size )
{
    s_AacDecorder.Initialize( buffer, size, s_StreamNum );

    nw::snd::internal::driver::StreamSoundLoader::SetStreamDataDecoder( &s_AacDecorder );
}

void Finalize()
{
    nw::snd::internal::driver::StreamSoundLoader::SetStreamDataDecoder(NULL);

    s_AacDecorder.Finalize();
}

u32 GetAacDecBufferSize( const SoundArchive* arc )
{
    u32 result = 0;

    s_StreamNum = 1;
    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        s_StreamNum = soundArchivePlayerInfo.streamSoundMax;
    }

    // 仕様上もっとも大きな組み合わせでひとまずメモリを確保
    s_AacInfo.smpFreq = HEAACDEC_48000HZ;

    // 6ch 対応は、CafeSDK 2.11.00 以降
#if defined(NW_SND_CONFIG_ENABLE_6CHAAC)
    s_AacInfo.channel = 6;
#else
    s_AacInfo.channel = 2;
#endif

    result += nw::ut::RoundUp(AACDEC_INPUT_STREAM_MAX_SIZE, 64);
    result += nw::ut::RoundUp(s_AacDecorder.GetAacDecBufferSize( &s_AacInfo ), 64) * s_StreamNum;

    return result;
}

} // namespace nw::snd::aac
} // namespace nw::snd
} // namespace nw


