﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <exception>

#include <string>
#include <cstring>

#include <nw/font.h>
#include <nw/lyt.h>

#include <nw/vwrlyt/win/vwrlyt_DirResourceAccessorWin.h>

#include <windows.h>

#if defined(NW_VWRLYT_ENABLE)

namespace nw
{
namespace vwrlyt
{

class DirResourceAccessorImpl : public DirResourceAccessor
{
    static const u8 RESOURCE_ALIGNMENT = 4;
    static const u8 FONT_RESOURCE_ALIGNMENT = 128;
    static const u8 TEXTURE_RESOURCE_ALIGNMENT = 128;

public:
    DirResourceAccessorImpl(const char* pBasePath);

    virtual ~DirResourceAccessorImpl();

    // 指定したパス以下のリソースをアクセサに関連付けます。
    virtual void Attach(const char* resPath);

    // アクセサに関連づけられたリソースを破棄します。
    virtual void Detach();

    // リソースの関連づけが行われているか調べます。
    virtual bool IsAttached() const
    {
        return !m_BasePath.empty();
    }

    // 関連付けられたリソースのパスを取得します。
    virtual const char* GetResourceBasePath() const
    {
        return m_BasePath.c_str();
    }

    virtual const char* FindFile(
        lyt::ResType resType,
        const char* fileName);

    virtual void* GetResource(
        lyt::ResType resType,
        const char* name,
        u32* pSize);

    virtual const lyt::TextureInfo* GetTexture(const char* name);

    virtual font::Font* GetFont(const char* name);

    virtual const nw::lyt::ArchiveShaderInfo* GetShader(const char* name);
protected:
    virtual font::Font* ReferFont(const char* name);

    virtual void AddFont(const char* name, font::Font* pFont, bool own);

    class DirResourceLink
    {
    public:
        DirResourceLink()
        {}

        void Set(
            lyt::ResType resType,
            void* pResource,
            u32 size,
            const char* pName)
        {
            m_ResType = resType;
            m_pResource = pResource;
            m_Size = size;
            m_Name = pName;
        }

        ut::LinkListNode m_Link;

        lyt::ResType m_ResType;
        void* m_pResource;
        u32 m_Size;
        std::string m_Name;
    };

    typedef ut::LinkList<DirResourceLink, offsetof(DirResourceLink, m_Link)>  DirResourceList;

    void*
    SearchFile(
        lyt::ResType resType,
        const char* pBasePath,
        const char* pName,
        u32* pSize);

    DirResourceLink* SearchResource(
        lyt::ResType resType,
        const char* pName);

    bool RegisterResource(
        lyt::ResType resType,
        void* pResource,
        u32 size,
        const char* pName);

    const std::string GetFilePath(
        const char* pResPath,
        lyt::ResType resType,
        const char* pFileName,
        const char* pDirSeparator);

    DirResourceList m_ResourceList;
    std::string m_BasePath;
    lyt::FontContainer m_FontList;
    lyt::TextureContainer m_TextureList;
    char m_FoundPath[MAX_PATH];
    bool m_FindContinue;
};

DirResourceAccessor*
DirResourceAccessor::Create(const char *pBasePath)
{
    return new DirResourceAccessorImpl(pBasePath);
}


/*---------------------------------------------------------------------------*
  @brief コンストラクタ
 *---------------------------------------------------------------------------*/
DirResourceAccessorImpl::DirResourceAccessorImpl(
    const char* pBasePath
)
: m_BasePath(pBasePath),
  m_FindContinue(false)
{
}

DirResourceAccessorImpl::~DirResourceAccessorImpl()
{
    this->Detach();
}

DirResourceAccessorImpl::DirResourceLink*
DirResourceAccessorImpl::SearchResource(
    lyt::ResType resType,
    const char* name
)
{
    for (DirResourceList::Iterator it = m_ResourceList.GetBeginIter(); it != m_ResourceList.GetEndIter(); ++it)
    {
        if (it->m_ResType == resType && it->m_Name == name)
        {
            return &(*it);
        }
    }

    return NULL;
}

// リソースパスの設定
void DirResourceAccessorImpl::Attach(const char* resPath)
{
    NW_ASSERT_NOT_NULL(resPath);

    if (IsAttached())
    {
        if (0 != std::strcmp(GetResourceBasePath(), resPath))
        {
            // 既にAttachしているパスと異なる場合は Detach
            Detach();
        }
    }
    if ( ! IsAttached())
    {
        m_BasePath = resPath;
    }
}

// 読み込んでいる全てのリソースを開放
void DirResourceAccessorImpl::Detach()
{
    m_FontList.Finalize();
    m_TextureList.Finalize();
    while (!m_ResourceList.empty())
    {
        DirResourceList::iterator top = m_ResourceList.begin();
        DirResourceLink* pResLink = &(*top);
        lyt::Layout::FreeMemory(pResLink->m_pResource);
        m_ResourceList.erase(top);
        delete pResLink;
    }

    m_BasePath = "";
}

const char* DirResourceAccessorImpl::FindFile(
    lyt::ResType resType,
    const char* pPrevFilePath)
{
    // フォルダがロックされてしまう問題があるため、効率は良くないが毎回一から探す
    std::string findPattern = this->GetFilePath(m_BasePath.c_str(), resType, "*", "\\");
    WIN32_FIND_DATAA findData;
    HANDLE handle = ::FindFirstFileA(findPattern.c_str(), &findData);

    if (handle == INVALID_HANDLE_VALUE)
    {
        return NULL;
    }
    else
    {
        if (pPrevFilePath == NULL)
        {
            // 最初のファイル
            // 隠しファイルは無視する。
            while (findData.cFileName[0] == '.')
            {
                if (! ::FindNextFileA(handle, &findData))
                {
                    ::FindClose(handle);
                    return NULL;
                }
            }
        }
        else
        {
            // 二つ目以降のファイル
            while (::strcmp(findData.cFileName, pPrevFilePath) != 0)
            {
                if (! ::FindNextFileA(handle, &findData))
                {
                    ::FindClose(handle);
                    return NULL;
                }
            }

            // ひとつ前に探したファイルが見つかった。その次を探す
            if (! ::FindNextFileA(handle, &findData))
            {
                ::FindClose(handle);
                return NULL;
            }
        }

        ::FindClose(handle);
        ut::strcpy(m_FoundPath, findData.cFileName);
        return m_FoundPath;
    }
}

bool
DirResourceAccessorImpl::RegisterResource(
    lyt::ResType resType,
    void*       pResource,
    u32         size,
    const char* pName)
{
    NW_ASSERT_NOT_NULL(pResource);
    NW_ASSERT_NOT_NULL(pName);

    DirResourceLink* pDirResourceLink = new DirResourceLink();
    NW_ASSERT_NOT_NULL(pDirResourceLink);

    pDirResourceLink->Set(resType, pResource, size, pName);
    m_ResourceList.push_back(pDirResourceLink);

    return true;
}

void*
DirResourceAccessorImpl::GetResource(
     lyt::ResType resType,
    const char* name,
    u32* pSize
)
{
    NW_ASSERT_NOT_NULL(name);

    size_t len = ::strlen(name);
    if (len == 0)
    {
        return NULL;
    }

    DirResourceLink* pDirResourceLink = this->SearchResource(resType, name);
    if (pDirResourceLink)
    {
        if (pSize)
        {
            *pSize = pDirResourceLink->m_Size;
        }
        return pDirResourceLink->m_pResource;
    }
    else
    {
        u32 size;
        void* buffer = NULL;

        if ((buffer = SearchFile(resType, m_BasePath.c_str(), name, &size)) != 0)
        {
            RegisterResource(resType, buffer, size, name);
            if (pSize)
            {
                *pSize = size;
            }
            return buffer;
        }

        return NULL;
    }
}

font::Font*
DirResourceAccessorImpl::GetFont(const char* name)
{
    font::Font* pFont = this->ReferFont(name);

    if (pFont == NULL)
    {
        pFont = this->LoadFont(name);

        if (pFont != NULL)
        {
            this->AddFont(name, pFont, true);
        }
    }

    return pFont;
}

font::Font*
DirResourceAccessorImpl::ReferFont(const char* name)
{
    // フォントを所有しているか。
    {
        font::Font* pFont = m_FontList.FindFontByName(name);
        if (pFont != NULL)
        {
            return pFont;
        }
    }

    // .bffnt のフォント生成はライブラリに任せる。
    return NULL;
}

void
DirResourceAccessorImpl::AddFont(const char* name, font::Font* pFont, bool own)
{
    (void)m_FontList.RegistFont(name, pFont, own);
}

const lyt::TextureInfo*
DirResourceAccessorImpl::GetTexture(const char *name)
{
    const lyt::TextureInfo* texInfo = m_TextureList.FindTextureByName(name);
    if (texInfo)
    {
        return texInfo;
    }
    else
    {
        lyt::TextureInfo* newTextureInfo = m_TextureList.RegistTexture(name);

        if (newTextureInfo)
        {
            this->LoadTexture(newTextureInfo, name);

            if (newTextureInfo->GetTextureObject() == lyt::TextureInfo::INVALID)
            {
                NW_WARNING(false, "Can't load texture : %s\n", name);
            }
        }
        else
        {
            NW_WARNING(false, "Can't register texture : %s\n", name);
        }

        return newTextureInfo;
    }
}

void*
DirResourceAccessorImpl::SearchFile(
    lyt::ResType resType,
    const char* pBasePath,
    const char* pName,
    u32* pSize
)
{
    NW_ASSERT_NOT_NULL(pName);

    HANDLE hFindDir;
    WIN32_FIND_DATAA findDataDir;

    std::string searchDir = std::string(pBasePath) + "/*";

    if ((hFindDir = ::FindFirstFileA(searchDir.c_str(), &findDataDir)) != INVALID_HANDLE_VALUE)
    {
        do
        {
            if (::strcmp(".", findDataDir.cFileName) == 0 ||
                ::strcmp("..", findDataDir.cFileName) == 0)
            {
                continue;
            }

            std::string dirName = std::string(pBasePath) + "/" + findDataDir.cFileName + "/";
            std::string searchFile = dirName + pName;

            HANDLE hFindFile;
            WIN32_FIND_DATAA findDataFile;

            if ((hFindFile = ::FindFirstFileA(searchFile.c_str(), &findDataFile)) == INVALID_HANDLE_VALUE)
            {
                continue;
            }

            std::string fileName = dirName + findDataFile.cFileName;
            HANDLE handle = CreateFileA(fileName.c_str(), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
            if(handle == INVALID_HANDLE_VALUE)
            {
                NW_FATAL_ERROR("can not open file (%s).", fileName.c_str());
            }

            u8 alignment = RESOURCE_ALIGNMENT;
            switch (resType)
            {
            case lyt::res::RESOURCETYPE_FONT:
            case lyt::res::RESOURCETYPE_ARCHIVEFONT:
                alignment = FONT_RESOURCE_ALIGNMENT;
                break;

            case lyt::res::RESOURCETYPE_TEXTURE:
                alignment = TEXTURE_RESOURCE_ALIGNMENT;
                break;

            default:
                break;
            }

            /// 下位32bitのみ使う
            DWORD file_size_high = 0;
            DWORD fileSize = GetFileSize(handle, &file_size_high);
            void* buffer = lyt::Layout::AllocMemory(fileSize, alignment);
            NW_ASSERT_NOT_NULL(buffer);

            DWORD read_size = 0;
            BOOL read_success = ReadFile(handle, buffer, fileSize, &read_size, NULL);
            NW_ASSERT(read_success);
            CloseHandle(handle);

            ::FindClose(hFindFile);
            ::FindClose(hFindDir);

            if (pSize)
            {
                *pSize = fileSize;
            }

            return buffer;

        } while (::FindNextFileA(hFindDir, &findDataDir));

        ::FindClose(hFindDir);
    }

    return NULL;
}

const std::string
DirResourceAccessorImpl::GetFilePath(
    const char* pResPath,
    lyt::ResType resType,
    const char* pFileName,
    const char* pDirSeparator)
{
    std::string path;

    if (pResPath)
    {
        path += pResPath;
        path += pDirSeparator;
    }

    path += char(resType >> 24);
    path += char(resType >> 16);
    path += char(resType >> 8);
    path += char(resType >> 0);

    if (pFileName)
    {
        path += pDirSeparator;
        path += pFileName;
    }

    return path;
}

const nw::lyt::ArchiveShaderInfo*
DirResourceAccessorImpl::GetShader(const char* name)
{
    NW_UNUSED_VARIABLE(name);
    return NULL;
}

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
