﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/vwrlyt/vwrlyt_Viewer.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
//#include <tentative/seadGetOpt.h>

#if defined(NW_VWRLYT_ENABLE)

namespace nw
{
namespace vwrlyt
{

static const f32 DRC_WIDTH = 854.f;
static const f32 DRC_HEIGHT = 480.f;

#if defined(NW_PLATFORM_CAFE)
static const f32 DRC_TOUCH_WIDTH = 1280.f;
static const f32 DRC_TOUCH_HEIGHT = 720.f;
#endif
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
static const f32 DRC_TOUCH_WIDTH = 854.f;
static const f32 DRC_TOUCH_HEIGHT = 480.f;
#endif

const Viewer::ViewSizeValue Viewer::VIEWSIZE_VALUES[VIEWSIZE_NUM_MAX] =
{
    {1280.f, 720.f},
    {1920.f, 1080.f},
    {1280.f, 720.f},
    {854.f, 480.f}
};

const char* Viewer::VIEWSIZE_DESCRIPTIONS[VIEWSIZE_NUM_MAX] =
{
    "original",
    "FullHD",
    "HD",
    "DRC"
};

Viewer::Viewer()
 : m_AnimateOnce(false)
 , m_Layout(NULL)
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
 , m_ResourceAccessor(*DirResourceAccessor::Create())
#endif
 , m_Allocator(NULL)
 , m_pControlCreator(NULL)
 , m_ViewSize(VIEWSIZE_ORIGINAL)
 , m_IsPerspectiveProjection(false)
 , m_Fps(FPS_60)
 , m_PerspectiveFovyDeg(40.f)
 , m_PerspectiveNear(1.f)
 , m_PerspectiveFar(10000.f)
 , m_IsViewMtxDirty(false)
 , m_pUIRenderer(NULL)
 , m_bMenuVisible(true)
 , m_IsDisableAllButton(false)
 , m_MenuPage(PAGE_ID_ANIM)
 , m_MenuPagePrev(PAGE_ID_ANIM)
{
    m_PreviewOption.reset();
}

Viewer::~Viewer()
{
}

void Viewer::Initialize(lyt::GraphicsResource* graphicsResource, ut::IAllocator* allocator, const lyt::Size& mainViewportSize)
{
    m_Allocator = allocator;
    m_MainViewportSize = mainViewportSize;
    m_DrawInfo.SetGraphicsResource(graphicsResource);

    InitializeMenu();

#if defined(NW_PLATFORM_CAFE)
    m_HIOListener.StartListen();
#endif
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    m_ScreenShot.SetSize(static_cast<u32>(mainViewportSize.width), static_cast<u32>(mainViewportSize.height));
#endif
}

void Viewer::Finalize()
{
    m_DwWindowManager.Finalize();
}

bool Viewer::ConvertHostToFsPath(nw::ut::BufferedSafeString *outFsPath, const char *mountPoint, const char *inHostPath)
{
    NW_ASSERT_NOT_NULL(outFsPath);
    NW_ASSERT_NOT_NULL(mountPoint);
    NW_ASSERT_NOT_NULL(inHostPath);

    ut::SafeString src(inHostPath);

    if (src.isEmpty())
    {
        return false;
    }

    ut::SafeString part;

    if (0 == strncmp("/cygdrive/", src.c_str(), 10) && isalpha(src[10]) && src[11] == '/')
    {
        // CYGWIN形式
        outFsPath->format("%s/%c", mountPoint, src.c_str() + 10);
        part = src.getPart(11);
    }
    else if (isalpha(src[0]) && src[1] == ':')
    {
        // DOS形式
        outFsPath->format("%s/%c", mountPoint, src[0]);
        part = src.getPart(2);
    }
    else
    {
        // パスが不正。
        return false;
    }

    ut::SafeString::iterator itor = part.begin();
    ut::SafeString::iterator end = part.end();
    for (; itor != end; ++itor)
    {
        outFsPath->append(*itor == '\\'? '/' : *itor);
    }

    return true;
}

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
bool Viewer::PreviewByCommandLineOption(int argc, const char** argv)
{
    NW_UNUSED_VARIABLE(argc)
    NW_UNUSED_VARIABLE(argv)

    m_PreviewOption.reset();

#if 0 // @@@
    char option;
    sead::GetOpt opt(__argc, (const char**)__argv, "w:h:l:cm:n:s:f:g");

    while ((option = opt.next()) != sead::GetOpt::END)
    {
        switch (option)
        {
        case 'm':
                // モードを指定する
                if (opt.arg()[0] == 'c')
                {
                        m_PreviewOption.previewMode = MODE_CONTROL;
                }
                break;
        case 'n':
                // 複数レイアウトがあった場合のルートのレイアウト名を指定する
                m_PreviewOption.layoutName.copy(opt.arg());
                m_PreviewOption.layoutName.append(".bflyt");
                break;
        case 's':
                // スクリーンショットを撮影します。引数でファイル名を指定します。
                m_ScreenShot.SetFileName(opt.arg());
                break;
        case 'f':
                // スクリーンショットを撮影する際のフレームを指定します。
                m_ScreenShot.SetFrame(sead::StringUtil::parseF32(opt.arg()));
                break;
        case 'g':
                // sRGBライトを行い、ガンマ値を補正します。
                m_PreviewOption.isSRGBWriteEnable = true;
                break;
        case '?':
                NW_ERR("illegal option[%c]", opt.option());
                break;
        }
    }
#endif // @@@

    // スクリーンショットが有効なら必ずアニメモード
    if (m_ScreenShot.IsNeedToTake() && m_PreviewOption.previewMode == MODE_CONTROL)
    {
        NW_LOG("preview mode must be anim when taking screen shot.\n");
        m_PreviewOption.previewMode = MODE_ANIMATION;
    }

#if 0 // @@@
    if (opt.getNonOptionArgNum() > 0)
    {
        m_PreviewOption.path.copy(opt.getNonOptionArg(0));
        StartPreview(m_PreviewOption);
        return true;
    }
    else
    {
        return false;
    }
#else // @@@
    return true;
#endif // @@@
}

LRESULT Viewer::PreviewByMessageProcCallback(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    NW_UNUSED_VARIABLE(hWnd)
    NW_UNUSED_VARIABLE(wParam)

    switch(message)
    {
    case WM_COPYDATA:
        PCOPYDATASTRUCT cds = NULL;
        LayoutProtocol  *protocol = NULL;
        char    *verb = NULL;

        cds = (PCOPYDATASTRUCT)lParam;
        protocol = (LayoutProtocol*)cds->lpData;
        verb = protocol->Verb;

        if(::strcmp(verb, "Open") == 0)
        {
            PreviewByCommandStr(protocol->Data);
        }

        return 1;
    }

    return 0;
}
#endif

bool Viewer::PreviewByCommandStr(const char* commandStr)
{
    m_PreviewOption.reset();

    ParseCommandStr(&m_PreviewOption, commandStr);

    StartPreview(m_PreviewOption);

    return true;
}

void Viewer::UpdateLayout()
{
    if (m_Layout)
    {
        m_Layout->AnimateAndUpdateAnimFrame();
        m_DrawInfo.SetProjectionMtx(*(const math::MTX44*)&m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(*(const math::MTX34*)&m_CameraViewMatrix);
        m_Layout->CalculateMtx(m_DrawInfo, m_IsViewMtxDirty);
        m_AnimateOnce = true;
        m_IsViewMtxDirty = false;
    }
}

void Viewer::UpdateSystem()
{
#if defined(NW_PLATFORM_CAFE)
    if (m_HIOListener.GetMessage())
    {
        PreviewByCommandStr(m_HIOListener.GetMessage());
        m_HIOListener.ClearMessage();
    }
#endif

    if (m_Layout == NULL)
    {
        return;
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    if (m_ScreenShot.IsNeedToTake())
    {
        if (m_ScreenShot.GetState() == ScreenShot::cState_NotReady)
        {
            if (m_AnimationMgr.GetAnimator())
            {
                m_AnimationMgr.GetAnimator()->Stop(m_ScreenShot.GetFrame());
            }
            m_Layout->Animate();
            m_Layout->CalculateMtx(m_DrawInfo, m_IsViewMtxDirty);
            m_ScreenShot.Prepare(m_Allocator);

            this->SetMenuVisible(false);

#if 0 // @@@
            NW_INSTANCE( sead::ProcessMeter )->setVisible( false );
#endif // @@@
            m_AnimateOnce = true;
            m_IsViewMtxDirty = false;
        }
        return;
    }
#endif
}

Viewer::MenuEvent Viewer::GetMenuEvent(const nw::internal::dw::Inputs& inputs)
{
    if (inputs.GetPad() != NULL)
    {
        const nw::dev::Pad& pad = *inputs.GetPad();
        if (pad.IsTrig(pad.MASK_A))
        {
            return MENUEVENT_INVOKE;
        }

        if (pad.IsTrig(pad.MASK_B))
        {
            return MENUEVENT_CANCEL;
        }

        if (pad.IsTrig(pad.MASK_LEFT))
        {
            return MENUEVENT_LEFT;
        }

        if (pad.IsTrig(pad.MASK_RIGHT))
        {
            return MENUEVENT_RIGHT;
        }

        if (pad.IsTrig(pad.MASK_X))
        {
            return MENUEVENT_X;
        }
    }

    return MENUEVENT_NONE;
}

bool Viewer::OnUpdateFocusedInput(ItemId itemId, const nw::internal::dw::Inputs& inputs)
{
    MenuEvent event = this->GetMenuEvent(inputs);

    switch (itemId)
    {
    case ITEM_ID_ANIM_FRAME:
        {
            lyt::Animator* animator = m_AnimationMgr.GetAnimator();
            if (animator)
            {
                if (event == MENUEVENT_LEFT)
                {
                    animator->StopCurrent();
                    if (animator->IsMinFrame())
                    {
                        animator->StopAtMax();
                    }
                    else
                    {
                        animator->SetFrame(animator->GetFrame() - 1);
                    }
                }
                else if (event == MENUEVENT_RIGHT)
                {
                    animator->StopCurrent();
                    if (animator->IsMaxFrame())
                    {
                        animator->StopAtMin();
                    }
                    else
                    {
                        animator->SetFrame(animator->GetFrame() + 1);
                    }
                }
            }
        }
        break;
    case ITEM_ID_ANIM_TAG:
    case ITEM_ID_ANIM_TAG_NUM:
        if (m_AnimationMgr.GetAnimationNum() > 0)
        {
            if (event == MENUEVENT_LEFT)
            {
                if (m_AnimationMgr.GetCurrentAnimationNo() == 0)
                {
                    m_AnimationMgr.SetCurrentAnimationNo(m_AnimationMgr.GetAnimationNum() - 1);
                }
                else
                {
                    m_AnimationMgr.SetCurrentAnimationNo(m_AnimationMgr.GetCurrentAnimationNo() - 1);
                }
                m_AnimationMgr.StartAnimation();
            }
            else if (event == MENUEVENT_RIGHT)
            {
                if (m_AnimationMgr.GetAnimationNum() <= m_AnimationMgr.GetCurrentAnimationNo() + 1)
                {
                    m_AnimationMgr.SetCurrentAnimationNo(0);
                }
                else
                {
                    m_AnimationMgr.SetCurrentAnimationNo(m_AnimationMgr.GetCurrentAnimationNo() + 1);
                }
                m_AnimationMgr.StartAnimation();
            }
        }
        break;
    case ITEM_ID_ANIM_TARGET_MODE:
        {
            if (event == MENUEVENT_LEFT || event == MENUEVENT_RIGHT)
            {
                if (m_AnimationMgr.GetTargetMode() == AnimationManager::TARGET_MODE_PANE)
                {
                    m_AnimationMgr.SetTargetMode(AnimationManager::TARGET_MODE_GROUP);
                }
                else
                {
                    m_AnimationMgr.SetTargetMode(AnimationManager::TARGET_MODE_PANE);
                }
            }
        }
        break;
    case ITEM_ID_ANIM_TARGET:
    case ITEM_ID_ANIM_TARGET_NUM:
        if (m_AnimationMgr.GetCurrentTargetNum() > 0)
        {
            if (event == MENUEVENT_LEFT)
            {
                if (m_AnimationMgr.GetCurrentTargetNo() == 0)
                {
                    m_AnimationMgr.SetCurrentTargetNo(m_AnimationMgr.GetCurrentTargetNum() - 1);
                }
                else
                {
                    m_AnimationMgr.SetCurrentTargetNo(m_AnimationMgr.GetCurrentTargetNo() - 1);
                }
                m_AnimationMgr.StartAnimation();
            }
            else if (event == MENUEVENT_RIGHT)
            {
                if (m_AnimationMgr.GetCurrentTargetNum() <= m_AnimationMgr.GetCurrentTargetNo() + 1)
                {
                    m_AnimationMgr.SetCurrentTargetNo(0);
                }
                else
                {
                    m_AnimationMgr.SetCurrentTargetNo(m_AnimationMgr.GetCurrentTargetNo() + 1);
                }
                m_AnimationMgr.StartAnimation();
            }
        }
        break;
    case ITEM_ID_CONTROL_FORCE_OFF:
        if (event == MENUEVENT_CANCEL)
        {
            m_ButtonGroup.ForceOffAll();
        }
        break;
    case ITEM_ID_CONTROL_CANCEL_SELECT:
        if (event == MENUEVENT_CANCEL)
        {
            m_ButtonGroup.CancelAll();
        }
        break;
    case ITEM_ID_CONTROL_DISABLE_ALL:
        if (event == MENUEVENT_LEFT || event == MENUEVENT_RIGHT)
        {
            bool is_set_disable = ! m_IsDisableAllButton;
            nw::ctrl::ButtonList& buttonList = m_ButtonGroup.GetButtonList();
            nw::ctrl::ButtonList::Iterator it_end = buttonList.GetEndIter();
            for (nw::ctrl::ButtonList::Iterator it = buttonList.GetBeginIter(); it != it_end; ++it)
            {
                nw::ctrl::AnimButton* button = nw::ut::DynamicCast<nw::ctrl::AnimButton*>(&(*it));
                if (button) {
                    button->SetActive( ! is_set_disable);
                    button->PlayDisableAnim(is_set_disable);
                }
            }
            m_ControlMenuItemDisableAll.Format("disable all button: %s", is_set_disable ? "true" : "false");
            m_IsDisableAllButton = is_set_disable;
        }
        break;
    case ITEM_ID_ANIM_SIZE:
    case ITEM_ID_CONTROL_SIZE:
        if (event == MENUEVENT_RIGHT)
        {
            if (m_ViewSize == VIEWSIZE_ORIGINAL)
            {
                SetViewSize(VIEWSIZE_DRC);
            }
            else
            {
                SetViewSize(static_cast<ViewSize>(static_cast<int>(m_ViewSize) - 1));
            }
        }
        else if (event == MENUEVENT_LEFT)
        {
            if (m_ViewSize == VIEWSIZE_DRC)
            {
                SetViewSize(VIEWSIZE_ORIGINAL);
            }
            else
            {
                SetViewSize(static_cast<ViewSize>(static_cast<int>(m_ViewSize) + 1));
            }
        }
        break;
    case ITEM_ID_ANIM_FPS:
    case ITEM_ID_CONTROL_FPS:
        if (event == MENUEVENT_RIGHT || event == MENUEVENT_LEFT)
        {
            if (m_Fps == FPS_60)
            {
                m_Fps = FPS_30;
            }
            else
            {
                m_Fps = FPS_60;
            }
        }
        break;
    case ITEM_ID_ANIM_CAMERA:
        if (event == MENUEVENT_RIGHT || event == MENUEVENT_LEFT)
        {
            m_IsPerspectiveProjection = ! m_IsPerspectiveProjection;
            SetViewSize(m_ViewSize);
        }
        else if (m_IsPerspectiveProjection && event == MENUEVENT_X)
        {
            this->SetMenuPage(PAGE_ID_PERSPECIVE_CONFIG);
        }
        break;
    case ITEM_ID_PERSPECIVE_FOVY:
        if (event == MENUEVENT_RIGHT)
        {
            m_PerspectiveFovyDeg += 1.f;
            SetViewSize(m_ViewSize);
        }
        else if (event == MENUEVENT_LEFT)
        {
            m_PerspectiveFovyDeg -= 1.f;
            SetViewSize(m_ViewSize);
        }
        break;
    case ITEM_ID_PERSPECIVE_NEAR:
        if (event == MENUEVENT_RIGHT)
        {
            m_PerspectiveNear += 1.f;
            SetViewSize(m_ViewSize);
        }
        else if (event == MENUEVENT_LEFT)
        {
            m_PerspectiveNear -= 1.f;
            SetViewSize(m_ViewSize);
        }
        break;
    case ITEM_ID_PERSPECIVE_FAR:
        if (event == MENUEVENT_RIGHT)
        {
            m_PerspectiveFar += 1.f;
            SetViewSize(m_ViewSize);
        }
        else if (event == MENUEVENT_LEFT)
        {
            m_PerspectiveFar -= 1.f;
            SetViewSize(m_ViewSize);
        }
        break;
    }

    if (ITEM_ID_PERSPECIVE_FOVY <= itemId && itemId <= ITEM_ID_PERSPECIVE_FAR && event == MENUEVENT_X)
    {
        this->SetMenuPageToPrev();
    }

    return false;
}

bool Viewer::OnUpdatePointerInput(ItemId itemId, const nw::internal::dw::Inputs& inputs)
{
    NW_UNUSED_VARIABLE(itemId)
    NW_UNUSED_VARIABLE(inputs)

    return false;
}

void Viewer::UpdateInputs(const nw::internal::dw::Inputs& inputs, const dev::Pad* vpad)
{
    if (this->GetMenuVisible())
    {
        if (!vpad || !m_DwWindowManager.IsPressWindowControlKey(*vpad))
        {
            m_DwWindowManager.UpdateInputs(inputs);
        }
        if (!inputs.GetPad() || !m_DwWindowManager.IsPressWindowControlKey(*inputs.GetPad()))
        {
            m_DwWindowManager.UpdateInputs(nw::internal::dw::Inputs(vpad, NULL));
        }
    }

    if (m_PreviewOption.previewMode == MODE_CONTROL && m_Layout != NULL)
    {
        bool is_pointer_on = (inputs.GetMouse() && inputs.GetMouse()->IsPointerOn());
        bool is_drc_pointer_on = (vpad && (vpad->IsPointerOn() || vpad->IsPointerOffNow()));
        if (is_pointer_on || is_drc_pointer_on)
        {
            // ポインタの位置をレイアウトの空間に変換し、ButtonGroupのUpdateに与える
            lyt::Size viewSize = GetViewSize();
            lyt::Size layoutSize = m_Layout->GetLayoutSize();
            float widthRate = layoutSize.width / viewSize.width;
            float heightRate = layoutSize.height / viewSize.height;
            float x, y; // 画面上のポインタ位置。中心原点、右下が大きくなる座標系
            float viewport_width, viewport_heigth;
            bool is_touch;
            bool is_release;

            if (is_drc_pointer_on)
            {
                // DRCのコントローラのポインタが有効ならそちらを優先する
                x = (vpad->GetPointer().x / DRC_TOUCH_WIDTH - 0.5f) * 2.f;
                y = (vpad->GetPointer().y / DRC_TOUCH_HEIGHT - 0.5f) * 2.f;
                viewport_width = DRC_WIDTH;
                viewport_heigth = DRC_HEIGHT;
                is_touch = vpad->IsTrig(vpad->MASK_TOUCH);
                is_release = vpad->IsRelease(vpad->MASK_TOUCH);
            }
            else
            {
                const dev::Mouse& mouse = *inputs.GetMouse();

                x = 2.0f * mouse.GetPointer().x / m_MainViewportSize.width - 1.0f;
                y = 2.0f * mouse.GetPointer().y / m_MainViewportSize.height - 1.0f;
                viewport_width = m_MainViewportSize.width;
                viewport_heigth = m_MainViewportSize.height;
                is_touch = mouse.IsTrig(mouse.MASK_LBUTTON);
                is_release = mouse.IsRelease(mouse.MASK_LBUTTON);
            }
            math::VEC2 pos(x * (viewport_width / 2.f) * widthRate, - y * (viewport_heigth / 2.f) * heightRate);
            m_ButtonGroup.Update(&pos, is_touch, is_release);
        }
        else
        {
            // ポインタが画面に入っていないときはnullを与えてもよい
            m_ButtonGroup.Update(NULL, false);
        }
    }
    else
    {
        if (inputs.GetPad() != NULL)
        {
            const dev::Pad& pad = *inputs.GetPad();

            if (pad.GetTrigMask() & pad.MASK_A)
            {
                if (m_AnimationMgr.IsAnimationPlaying())
                {
                    m_AnimationMgr.StopAnimation();
                }
                else
                {
                    m_AnimationMgr.StartAnimation();
                }
            }
            else if (pad.GetTrigMask() & pad.MASK_B)
            {
                m_AnimationMgr.ResetAnimation();
            }
        }
    }

    if (inputs.GetPad() != NULL)
    {
        const dev::Pad& pad = *inputs.GetPad();

        if (pad.GetTrigMask() & pad.MASK_START)
        {
            this->SetMenuVisible(!this->GetMenuVisible());
        }
    }

    UpdateMenuText();
}

void Viewer::UpdateMenu()
{
    if (this->GetMenuVisible() && m_pUIRenderer != NULL)
    {
        m_DwWindowManager.Update(*m_pUIRenderer);
    }
}

void Viewer::DrawLayout()
{
    if (m_Layout != NULL && m_AnimateOnce)
    {
        m_DrawInfo.SetProjectionMtx(*(const math::MTX44*)&m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(*(const math::MTX34*)&m_CameraViewMatrix);
        m_Layout->Draw(m_DrawInfo);
    }
}

void Viewer::DrawLayoutDRC()
{
    if (m_Layout != NULL && m_AnimateOnce)
    {
        m_DrawInfo.SetProjectionMtx(*(const math::MTX44*)&m_ProjectionMatrixDRC);
        m_DrawInfo.SetViewMtx(*(const math::MTX34*)&m_CameraViewMatrixDRC);
        m_Layout->Draw(m_DrawInfo);
    }
}

void Viewer::DrawSystem()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    if (m_ScreenShot.IsNeedToTake())
    {
        if (m_ScreenShot.GetState() == ScreenShot::cState_ReadyToTake)
        {
            m_ScreenShot.Take();
        }
    }
#endif

    if (this->GetMenuVisible() && m_pUIRenderer != NULL)
    {
        m_pUIRenderer->ClearBuffer();
        m_DwWindowManager.Draw(*m_pUIRenderer);
    }
}

void Viewer::StartPreview(const PreviewOption& option)
{
    m_AnimateOnce = false;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // 既にアタッチされていたらDetach
    if (m_ResourceAccessor.IsAttached())
    {
        m_ResourceAccessor.Detach();
    }
    m_ResourceAccessor.Attach(option.path.c_str());
#elif defined(NW_PLATFORM_CAFE)
    // 既にアタッチされていたらunload
    if (m_ResourceAccessor.IsAttached())
    {
        u8* arc_data = reinterpret_cast<u8*>(const_cast<void*>(m_ResourceAccessor.GetArchiveDataStart()));
        m_ResourceAccessor.Detach();
        nw::dev::FileDevice::LoadArg arg;
        arg.allocator = m_Allocator;
        NW_INSTANCE(dev::FileDeviceManager)->Unload(arg, arc_data);
    }
    {
        nw::dev::FileDevice::LoadArg arg;
        arg.path = option.path.c_str();
        arg.alignment = nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT;
        arg.allocator = m_Allocator;
        u8* data = NW_INSTANCE(dev::FileDeviceManager)->Load(arg);

        // バイナリリソースのルートディレクトリを指定してリソースアクセサを生成。
        if ( ! m_ResourceAccessor.Attach(data, "."))
        {
            NW_ERR("can not attach layout archive.\n");
        }
    }
#endif

    // レイアウトリソースの読み込み
    if (m_Layout != NULL)
    {
        m_ButtonGroup.FreeAll();
        m_AnimationMgr.Finalize();
        lyt::Layout::DeleteObj(m_Layout);
        m_Layout = NULL;
    }
    m_Layout = lyt::Layout::NewObj<lyt::Layout>();

    NW_INSTANCE( gfnd::Graphics )->LockDrawContext();

    {
        const char* layoutName = NULL;
        // レイアウト名が指定されていないときは探す
        if (option.layoutName.isEmpty())
        {
            layoutName = m_ResourceAccessor.FindFile(lyt::res::RESOURCETYPE_LAYOUT, NULL);
        }
        else
        {
            layoutName = option.layoutName.c_str();
        }
const void* lytRes = m_ResourceAccessor.GetResource(lyt::res::RESOURCETYPE_LAYOUT, layoutName);
NW_ASSERT_NOT_NULL(lytRes);
        if (m_PreviewOption.previewMode == MODE_ANIMATION)
        {
            m_Layout->Build(lytRes, &m_ResourceAccessor);
            m_AnimationMgr.Setup(m_Layout, &m_ResourceAccessor);
            m_AnimationMgr.StartAnimation();
            this->SetMenuPage(PAGE_ID_ANIM);
        }
        else
        {
            if (m_pControlCreator == NULL)
            {
                ctrl::DefaultControlCreator controlCreater(&m_ButtonGroup);
                m_Layout->Build(lytRes, &m_ResourceAccessor, &controlCreater);
            }
            else
            {
                m_Layout->Build(lytRes, &m_ResourceAccessor, m_pControlCreator);
            }
            this->SetMenuPage(PAGE_ID_CONTROL);
        }
        SetViewSize(m_ViewSize);
    }

    NW_INSTANCE( gfnd::Graphics )->UnlockDrawContext();

    this->SetMenuVisible(true);
    m_IsSRGBWriteEnable = option.isSRGBWriteEnable != 0 ? true : false;
}

void Viewer::ParseCommandStr(PreviewOption* option, const char* commandStr)
{
    int optionMode = 0;
    ut::SafeString str(commandStr);
    ut::FixedSafeString<cPathMax> buf;

    ut::SafeString::token_iterator it = str.tokenBegin(" ");
    ut::SafeString::token_iterator end = str.tokenEnd();

    while( it != end )
    {
        it.getAndForward( &buf );

        if (buf.isEqual("-m"))
        {
            optionMode = 1;
            continue;
        }
        else if (buf.isEqual("-n"))
        {
            optionMode = 2;
            continue;
        }
        else if (buf.isEqual("-g"))
        {
            option->isSRGBWriteEnable = true;
            continue;
        }

        switch(optionMode)
        {
        case 1:
            // モードを指定する
            if (buf.at(0) == 'c')
            {
                option->previewMode = MODE_CONTROL;
            }
            else
            {
                option->previewMode = MODE_ANIMATION;
            }
            optionMode = 0;
            break;
        case 2:
            // レイアウト名を指定する
            option->layoutName.copy(buf);
            option->layoutName.append(".bflyt");
            optionMode = 0;
            break;
        case 0:
        default:
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            // PCのファイルシステムにおける絶対パスが指定されます。
            option->path.copy(buf.c_str());
#elif defined(NW_PLATFORM_CAFE)
            option->path.format("%s/%s", m_PreviewPath.c_str(), buf.c_str());
#endif
            // パスにスペースが含まれる可能性があるため、まだendに達していないときはこれ以降の文字列を付け足して関数を抜ける
            if (it != end)
            {
                option->path.append(str.getPart(it.getIndex() - 1));
            }
            return;
        }
    }
}

lyt::Size Viewer::GetViewSize() const
{
    if (m_ViewSize == VIEWSIZE_ORIGINAL && m_Layout != NULL)
    {
        return m_Layout->GetLayoutSize();
    }
    else
    {
        return lyt::Size(VIEWSIZE_VALUES[m_ViewSize].x, VIEWSIZE_VALUES[m_ViewSize].y);
    }
}

void Viewer::SetViewSize(ViewSize viewSize)
{
    m_ViewSize = viewSize;
    m_IsViewMtxDirty = true;

    // drawInfoへの行列の設定
    if (m_Layout != NULL)
    {
        lyt::Size currentViewSize = GetViewSize();
        lyt::Size layoutSize = m_Layout->GetLayoutSize();
        float widthRate = layoutSize.width / currentViewSize.width;
        float heightRate = layoutSize.height / currentViewSize.height;
        float halfWidth = m_MainViewportSize.width / 2.f;
        float halfHeight = m_MainViewportSize.height / 2.f;
        static const float halfWidthDRC = 0.5f * DRC_WIDTH;
        static const float halfHeightDRC = 0.5f * DRC_HEIGHT;
        if (m_IsPerspectiveProjection)
        {
            f32 fovy = math::DegToRad(m_PerspectiveFovyDeg);
            {
                f32 height = halfHeight * heightRate / math::TanRad(fovy / 2.f);
                f32 aspect = (halfWidth * widthRate) / (halfHeight * heightRate);

                math::MTX44PerspectiveRad(&m_ProjectionMatrix, fovy, aspect, m_PerspectiveNear, m_PerspectiveFar);

                math::VEC3 pos(0.f, 0.f, height);
                math::VEC3 up(0.f, 1.f, 0.f);
                math::VEC3 target(0.f, 0.f, 0.f);

                math::MTX34LookAt(&m_CameraViewMatrix, pos, up, target);
            }
            {
                f32 height = halfHeightDRC * heightRate / math::TanRad(fovy / 2.f);
                f32 aspect = (halfWidthDRC * widthRate) / (halfHeightDRC * heightRate);

                math::MTX44PerspectiveRad(&m_ProjectionMatrixDRC, fovy, aspect, m_PerspectiveNear, m_PerspectiveFar);

                math::VEC3 pos(0.f, 0.f, height);
                math::VEC3 up(0.f, 1.f, 0.f);
                math::VEC3 target(0.f, 0.f, 0.f);

                math::MTX34LookAt(&m_CameraViewMatrixDRC, pos, up, target);
            }
        }
        else
        {
            {
                f32 left = -halfWidth * widthRate;
                f32 right = halfWidth * widthRate;
                f32 top = halfHeight * heightRate;
                f32 bottom = -halfHeight * heightRate;
                f32 nnear = 0.0f;
                f32 ffar = 500.0f;

                math::MTX44Ortho(&m_ProjectionMatrix, left, right, bottom, top, nnear, ffar);

                f32 cx = (left + right) * 0.5f;
                f32 cy = (top + bottom) * 0.5f;
                math::VEC3 pos(cx, cy, nnear);
                math::VEC3 up(0.f, 1.f, 0.f);
                math::VEC3 target(cx, cy, nnear - 1);

                math::MTX34LookAt(&m_CameraViewMatrix, pos, up, target);
            }
            {
                f32 left = -halfWidthDRC * widthRate;
                f32 right = halfWidthDRC * widthRate;
                f32 top = halfHeightDRC * heightRate;
                f32 bottom = -halfHeightDRC * heightRate;
                f32 nnear = 0.0f;
                f32 ffar = 500.0f;

                math::MTX44Ortho(&m_ProjectionMatrixDRC, left, right, bottom, top, nnear, ffar);

                f32 cx = (left + right) * 0.5f;
                f32 cy = (top + bottom) * 0.5f;
                math::VEC3 pos(cx, cy, nnear);
                math::VEC3 up(0.f, 1.f, 0.f);
                math::VEC3 target(cx, cy, nnear - 1);

                math::MTX34LookAt(&m_CameraViewMatrixDRC, pos, up, target);
            }
        }
    }
}

void Viewer::InitializeMenu()
{
    lyt::Size size = this->GetViewSize();

    m_DwWindowManager.Initialize(size.width, size.height, this->m_Allocator);

    {
        m_AnimMenuListBox
            .SetViewer(this)
            .AddItem(&m_AnimMenuItemFrame)
            .AddItem(&m_AnimMenuItemTag)
            .AddItem(&m_AnimMenuItemTagNum)
            .AddItem(&m_AnimMenuItemTargetMode)
            .AddItem(&m_AnimMenuItemTarget)
            .AddItem(&m_AnimMenuItemTargetNum)
            .AddItem(&m_AnimMenuItemSize)
            .AddItem(&m_AnimMenuItemFps)
            .AddItem(&m_AnimMenuItemCamera);
        m_AnimMenuListBox.GetItemPlaceHolder(0).SetIsSelected(true);
    }

    {
        m_ControlMenuItemForceOff.SetText("force off all button(B)");
        m_ControlMenuItemCancelSelect.SetText("cancel select all button(B)");
        m_ControlMenuItemDisableAll.SetText("disable all button: false");

        m_ControlMenuListBox
            .SetViewer(this)
            .AddItem(&m_ControlMenuItemForceOff)
            .AddItem(&m_ControlMenuItemCancelSelect)
            .AddItem(&m_ControlMenuItemDisableAll)
            .AddItem(&m_ControlMenuItemSize)
            .AddItem(&m_ControlMenuItemFps);
        m_ControlMenuListBox.GetItemPlaceHolder(0).SetIsSelected(true);
    }

    {
        m_PerspMenuListBox
            .SetViewer(this)
            .AddItem(&m_PerspMenuItemFovy)
            .AddItem(&m_PerspMenuItemNear)
            .AddItem(&m_PerspMenuItemFar);
        m_PerspMenuListBox.GetItemPlaceHolder(0).SetIsSelected(true);
    }

    {
        m_MenuWindow.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO);
        m_MenuWindow.SetBackgroundColor0(nw::ut::Color4f(m_MenuWindow.GetBackgroundColor0(), 0.5f));
        m_MenuWindow.SetBackgroundColor1(nw::ut::Color4f(m_MenuWindow.GetBackgroundColor1(), 0.5f));
        m_DwWindowManager.CreateWindow(&m_MenuWindow, 30.f, 30.f);
    }

#if 0 // @@@
    ///--- デバッグメニューのコントローラの設定
    sead::ControllerWrapper& controller = m_Menu.getController();
#if defined(NW_PLATFORM_CAFE)
    // Wiiリモコンの1番を割り当てる
    controller.registerWith(SEAD_INSTANCE(sead::ControllerMgr)->getControllerByOrder(sead::ControllerDefine::cController_CafeRemote, 0));
#endif
    // EnterとCancelはデフォルトではリピートなし
    controller.setPadRepeat(sead::DebugMenu::cPadMask_Enter | sead::DebugMenu::cPadMask_Cancel , 0, 0);
    // 十字キーも操作しやすい速度にする
    controller.setPadRepeat(sead::DebugMenu::cPadMask_Down | sead::DebugMenu::cPadMask_Up, 30, 5);
    controller.setPadRepeat(sead::DebugMenu::cPadMask_Left | sead::DebugMenu::cPadMask_Right, 20, 2);

#endif // @@@
    UpdateMenuText();

    // 最初は非表示状態にする
    this->SetMenuPage(PAGE_ID_ANIM);
    this->SetMenuVisible(false);
}

void Viewer::UpdateMenuText()
{
    m_AnimMenuItemFrame.Format(" FRAME:%3.0f /%3.0f", m_AnimationMgr.GetCurrentAnimationFrame(), m_AnimationMgr.GetAnimationFrameMax());
    m_AnimMenuItemTag.Format("  ANIM: %s%s", m_AnimationMgr.GetCurrentAnimationName(),
            m_AnimationMgr.IsCurrentAnimationLoop() ? "(Loop)" : "");
    m_AnimMenuItemTagNum.Format("        %2d /%2d", m_AnimationMgr.GetCurrentAnimationNo() + 1, m_AnimationMgr.GetAnimationNum());
    m_AnimMenuItemTargetMode.Format("TARGET: %s", m_AnimationMgr.GetTargetMode() == AnimationManager::TARGET_MODE_PANE ? "Pane" : "Group");
    m_AnimMenuItemTarget.Format(" %s: %s", m_AnimationMgr.GetTargetMode() == AnimationManager::TARGET_MODE_PANE ? " PANE" : "GROUP",
            m_AnimationMgr.GetCurrentTargetName());
    m_AnimMenuItemTargetNum.Format("        %2d /%2d", m_AnimationMgr.GetCurrentTargetNo() + 1, m_AnimationMgr.GetCurrentTargetNum());
    {
            lyt::Size size = GetViewSize();
            m_AnimMenuItemSize.Format("  SIZE: %.0fx%.0f(%s)", size.width, size.height, VIEWSIZE_DESCRIPTIONS[m_ViewSize]);
            m_ControlMenuItemSize.Format("SIZE: %.0fx%.0f(%s)", size.width, size.height, VIEWSIZE_DESCRIPTIONS[m_ViewSize]);
    }
    if (m_Fps == FPS_60)
    {
        m_AnimMenuItemFps.SetText("   FPS: 60fps");
        m_ControlMenuItemFps.SetText(" FPS: 60fps");
    }
    else
    {
        m_AnimMenuItemFps.SetText("   FPS: 30fps");
        m_ControlMenuItemFps.SetText(" FPS: 30fps");
    }
    if (m_IsPerspectiveProjection)
    {
        m_AnimMenuItemCamera.SetText("  PROJ: Perspective(X: config)");
    }
    else
    {
        m_AnimMenuItemCamera.SetText("  PROJ: Ortho");
    }
    m_PerspMenuItemFovy.Format("  FOVY: %.0f (degree)", m_PerspectiveFovyDeg);
    m_PerspMenuItemNear.Format("  NEAR: %.0f", m_PerspectiveNear);
    m_PerspMenuItemFar.Format ("   FAR: %.0f", m_PerspectiveFar);
}

void Viewer::SetMenuVisible(bool visible)
{
    m_bMenuVisible = visible;
    m_DwWindowManager.SetWindowVisibleFlag(m_bMenuVisible);
}

void Viewer::SetMenuPage(PageId pageId)
{
    m_MenuPagePrev = m_MenuPage;
    m_MenuPage = pageId;

    nw::internal::dw::UIElement* pContent = NULL;
    switch (pageId)
    {
    case PAGE_ID_ANIM:
        m_MenuWindow.SetTitle("Animation Mode");
        pContent = &m_AnimMenuListBox;
        break;

    case PAGE_ID_CONTROL:
        m_MenuWindow.SetTitle("Control Mode");
        pContent = &m_ControlMenuListBox;
        break;

    case PAGE_ID_PERSPECIVE_CONFIG:
        m_MenuWindow.SetTitle("Perspective Config (X: back)");
        pContent = &m_PerspMenuListBox;
        break;
    }

    if (pContent)
    {
        m_MenuWindow.SetContent(pContent);

        // ウィンドウタイトルが枠に収まるように調整します。
        if (m_pUIRenderer)
        {
            math::Vector2 titleSize = m_MenuWindow.MeasureTitle(*m_pUIRenderer);
            pContent->SetMinimumWidth(titleSize.x);
        }
    }
}

void Viewer::SetMenuPageToPrev()
{
    this->SetMenuPage(m_MenuPagePrev);
}

#if 0 // @@@
void Viewer::DebugMenuSkin::drawBackground(const sead::Vector3fType& /* pos */, const sead::Vector2f& /* size */)
{
}

void Viewer::DebugMenuSkin::drawTitleBackground(const sead::Vector3fType& pos, const sead::Vector2f& size)
{
    drawQuad(pos, size, sead::Color4f(0.f, 0.f, 0.f, 0.2f), sead::Color4f(0.f, 0.f, 0.f, 0.3f));
}

void Viewer::DebugMenuSkin::drawItemBackground(const sead::Vector3fType& pos, const sead::Vector2f& size)
{
    drawQuad(pos, size, sead::Color4f(0.f, 0.f, 0.f, 0.1f), sead::Color4f(0.f, 0.f, 0.f, 0.2f));
}

float Viewer::DebugMenuSkin::getLineHeight()
{
    return mTextWriter.getFontHeight() + 2.f;
}

#endif // @@@

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
