﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_ArcResourceAccessor.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/ut/ut_String.h>
#include <cctype>

namespace nw
{
namespace lyt
{

namespace
{

const int NW_LYT_MAX_PATH_NAME_LENGTH = 255;

//----------------------------------------
void*
GetResourceSub(
    ArchiveHandle* pArcHandle,
    const char* resRootDir,
    nw::lyt::ResType resType,
    const char* name,
    u32* pSize
)
{
    NW_ASSERTMSG(resType != 0, "please specify resource type. name = %s", name);

    s32 entryNum = -1;

    {
        char resTypeStr[5];
        resTypeStr[0] = u8(resType >> 24);
        resTypeStr[1] = u8(resType >> 16);
        resTypeStr[2] = u8(resType >>  8);
        resTypeStr[3] = u8(resType >>  0);
        resTypeStr[4] = 0;

        char pathName[NW_LYT_MAX_PATH_NAME_LENGTH + 1];
        if (resRootDir[0] == '.' && resRootDir[1] == '\0')
        {
            // ルート指定がトップになっている場合
            ut::snprintf(pathName, NW_LYT_MAX_PATH_NAME_LENGTH + 1, NW_LYT_MAX_PATH_NAME_LENGTH, "%s/%s", resTypeStr, name);
        }
        else
        {
            // ルート指定がトップ以外の場合
            ut::snprintf(pathName, NW_LYT_MAX_PATH_NAME_LENGTH + 1, NW_LYT_MAX_PATH_NAME_LENGTH, "%s/%s/%s", resRootDir, resTypeStr, name);
        }
        entryNum = pArcHandle->ConvertPathToEntryID(pathName);
    }

    if (entryNum != -1)
    {
        ArcFileInfo fileInfo;
        void* resPtr = pArcHandle->GetFileFast(entryNum, &fileInfo);
        NW_ASSERT_NOT_NULL(resPtr);
        if (pSize)
        {
            *pSize = fileInfo.GetLength();
        }

        return resPtr;
    }

    return NULL;
}

}   // namespace

//----------------------------------------
ArcResourceAccessor::ArcResourceAccessor()
:   m_ArcBuf(0)
{
    m_ResRootDir[0] = 0;
}

//----------------------------------------
bool
ArcResourceAccessor::Attach(
    void*       archiveStart,
    const char* resourceRootDirectory
)
{
    NW_ASSERT(! IsAttached());
    NW_ASSERT_NOT_NULL(archiveStart);
    NW_ASSERT_NOT_NULL(resourceRootDirectory);

    bool bSuccess = m_ArcHandle.PrepareArchive(archiveStart);

    if (! bSuccess)
    {
        return false;
    }

    m_ArcBuf = archiveStart;
    // ルートディレクトリ文字列のコピー
    ut::strcpy(m_ResRootDir, ROOTPATH_MAX, resourceRootDirectory);

    return true;
}

//----------------------------------------
void*
ArcResourceAccessor::Detach()
{
    NW_ASSERT(IsAttached());

    m_FontList.Finalize();
    m_TextureList.Finalize();
    m_ShaderList.Finalize();
    void* ret = m_ArcBuf;
    m_ArcBuf = 0;
    return ret;
}

//----------------------------------------
void*
ArcResourceAccessor::GetResource(
    ResType     resType,
    const char* name,
    u32*        pSize
)
{
    return GetResourceSub(&m_ArcHandle, m_ResRootDir, resType, name, pSize);
}

//----------------------------------------
bool
ArcResourceLink::Set(
    void*       archiveStart,
    const char* resourceRootDirectory
)
{
    NW_ASSERT_NOT_NULL(archiveStart);
    NW_ASSERT_NOT_NULL(resourceRootDirectory);

    bool bSuccess = m_ArcHandle.PrepareArchive(archiveStart);

    if (! bSuccess)
    {
        return false;
    }

    // ルートディレクトリ文字列のコピー
    ut::strcpy(m_ResRootDir, ROOTPATH_MAX, resourceRootDirectory);

    return true;
}

//----------------------------------------
const void*
ArcResourceLink::GetArchiveDataStart() const
{
    return m_ArcHandle.GetArchiveDataStart();
}

//----------------------------------------
font::Font*
ArcResourceAccessor::GetFont(const char *name)
{
    font::Font* pFont = m_FontList.FindFontByName(name);

    if (pFont == NULL)
    {
        pFont = this->LoadFont(name);

        if (pFont != NULL)
        {
            (void)m_FontList.RegistFont(name, pFont, true);
        }
    }

    return pFont;
}

//----------------------------------------
FontKey
ArcResourceAccessor::RegistFont(const char* name, font::Font* pFont)
{
    return m_FontList.RegistFont(name, pFont, false);
}

//----------------------------------------
void
ArcResourceAccessor::UnregistFont(FontKey key)
{
    m_FontList.UnregistFont(key);
}

//----------------------------------------
const TextureInfo*
ArcResourceAccessor::GetTexture(const char *name)
{
    const TextureInfo* texInfo = m_TextureList.FindTextureByName(name);
    if (texInfo)
    {
        return texInfo;
    }
    else
    {
        TextureInfo* newTextureInfo = m_TextureList.RegistTexture(name);

        if (newTextureInfo)
        {
            this->LoadTexture(newTextureInfo, name);

            if (newTextureInfo->GetTextureObject() == TextureInfo::INVALID)
            {
                NW_WARNING(false, "Can't load texture : %s\n", name);
            }
        }
        else
        {
            NW_WARNING(false, "Can't register texture : %s\n", name);
        }

        return newTextureInfo;
    }
}

//----------------------------------------
TextureInfo*
ArcResourceAccessor::RegistTexture(const char* name)
{
    return m_TextureList.RegistTexture(name);
}

//----------------------------------------
void
ArcResourceAccessor::UnregistTexture(TextureInfo* info)
{
    m_TextureList.UnregistTexture(info);
}

//----------------------------------------
const ArchiveShaderInfo*
ArcResourceAccessor::GetShader(const char* name)
{
    const ArchiveShaderInfo* pShaderInfo = m_ShaderList.FindShaderByName(name);
    if (pShaderInfo)
    {
        return pShaderInfo;
    }
    ArchiveShaderInfo* pNewShaderInfo = m_ShaderList.RegistShader(name);

    if (pNewShaderInfo)
    {
        this->LoadShader(pNewShaderInfo, name);
    }
    else
    {
        NW_WARNING(false, "Can't register shader : %s\n", name);
    }

    return pNewShaderInfo;
}

//----------------------------------------
ArchiveShaderInfo*
ArcResourceAccessor::RegistShader(const char* name)
{
    return m_ShaderList.RegistShader(name);
}

//----------------------------------------
void
ArcResourceAccessor::UnregistShader(ArchiveShaderInfo* pShader)
{
    m_ShaderList.UnregistShader(pShader);
}

//----------------------------------------
MultiArcResourceAccessor::MultiArcResourceAccessor()
{
}

//----------------------------------------
MultiArcResourceAccessor::~MultiArcResourceAccessor()
{
    DetachAll();
}

//----------------------------------------
void
MultiArcResourceAccessor::Attach(ArcResourceLink* pLink)
{
    NW_ASSERT_NOT_NULL(pLink);

    m_ArcList.PushBack(pLink);
}

//----------------------------------------
ArcResourceLink*
MultiArcResourceAccessor::Detach(const void* archiveStart)
{
    NW_ASSERT_NOT_NULL(archiveStart);

    internal::ArcResourceList::Iterator it_end = m_ArcList.GetEndIter();
    for (internal::ArcResourceList::Iterator it = m_ArcList.GetBeginIter(); it != it_end; ++it)
    {
        if (archiveStart == it->GetArchiveDataStart())
        {
            ArcResourceLink* ret = &(*it);
            m_ArcList.Erase(it);
            return ret;
        }
    }

    return 0;
}

//----------------------------------------
void
MultiArcResourceAccessor::Detach(ArcResourceLink* pLink)
{
    NW_ASSERT_NOT_NULL(pLink);

    m_ArcList.Erase(pLink);
}

//----------------------------------------
void
MultiArcResourceAccessor::DetachAll()
{
    m_ArcList.Clear();
}

//----------------------------------------
void*
MultiArcResourceAccessor::GetResource(
    ResType     resType,
    const char* name,
    u32*        pSize
)
{
    internal::ArcResourceList::Iterator it_end = m_ArcList.GetEndIter();
    for (internal::ArcResourceList::Iterator it = m_ArcList.GetBeginIter(); it != it_end; ++it)
    {
        ArchiveHandle* pArcHandle = it->GetArcHandle();
        if (void* resPtr = GetResourceSub(pArcHandle, it->GetResRootDir(), resType, name, pSize))
        {
            return resPtr;
        }
    }

    return NULL;
}

//----------------------------------------
font::Font*
MultiArcResourceAccessor::GetFont(const char *name)
{
    font::Font* pFont = m_FontList.FindFontByName(name);

    if (pFont == NULL)
    {
        pFont = this->LoadFont(name);

        if (pFont != NULL)
        {
            (void)m_FontList.RegistFont(name, pFont, true);
        }
    }

    return pFont;
}

//----------------------------------------
FontKey
MultiArcResourceAccessor::RegistFont(const char* name, font::Font* pFont)
{
    return m_FontList.RegistFont(name, pFont, false);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregistFont(FontKey key)
{
    m_FontList.UnregistFont(key);
}

//----------------------------------------
const TextureInfo*
MultiArcResourceAccessor::GetTexture(const char *name)
{
    const TextureInfo* texInfo = m_TextureList.FindTextureByName(name);
    if (texInfo)
    {
        return texInfo;
    }
    else
    {
        TextureInfo* newTextureInfo = m_TextureList.RegistTexture(name);

        if (newTextureInfo)
        {
            this->LoadTexture(newTextureInfo, name);

            if (newTextureInfo->GetTextureObject() == TextureInfo::INVALID)
            {
                NW_WARNING(false, "Can't load texture : %s\n", name);
            }
        }
        else
        {
            NW_WARNING(false, "Can't register texture : %s\n", name);
        }

        return newTextureInfo;
    }
}

//----------------------------------------
TextureInfo*
MultiArcResourceAccessor::RegistTexture(const char* name)
{
    return m_TextureList.RegistTexture(name);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregistTexture(TextureInfo* info)
{
    m_TextureList.UnregistTexture(info);
}

//----------------------------------------
const ArchiveShaderInfo*
MultiArcResourceAccessor::GetShader(const char* name)
{
    const ArchiveShaderInfo* pShaderInfo = m_ShaderList.FindShaderByName(name);
    if (pShaderInfo)
    {
        return pShaderInfo;
    }
    ArchiveShaderInfo* pNewShaderInfo = m_ShaderList.RegistShader(name);

    if (pNewShaderInfo)
    {
        this->LoadShader(pNewShaderInfo, name);
    }
    else
    {
        NW_WARNING(false, "Can't register shader: %s\n", name);
    }

    return pNewShaderInfo;
}

//----------------------------------------
ArchiveShaderInfo*
MultiArcResourceAccessor::RegistShader(const char* name)
{
    return m_ShaderList.RegistShader(name);
}

//----------------------------------------
void
MultiArcResourceAccessor::UnregistShader(ArchiveShaderInfo* pShader)
{
    m_ShaderList.UnregistShader(pShader);
}

}   // namespace lyt
}   // namespace nw
