﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ctrl/ctrl_ButtonBase.h>

namespace nw
{
namespace ctrl
{

ButtonBase::ButtonBase()
 : m_State(STATE_OFF_IDLE)
 , m_Flag(MASK_ACCEPT_ON | MASK_ACCEPT_OFF | MASK_ACCEPT_DOWN | MASK_ACCEPT_CANCEL | MASK_ACTIVE)
{
}

ButtonBase::~ButtonBase()
{
}

void ButtonBase::On()
{
    if (IsActive() && IsAcceptAction(ACTION_ON)) {
        m_ActionQueue.PushWithOmit(ACTION_ON);
    }
}

void ButtonBase::Off()
{
    if (IsAcceptAction(ACTION_OFF)) {
        m_ActionQueue.PushWithOmit(ACTION_OFF);
    }
}

void ButtonBase::Down()
{
    if (IsActive() && IsAcceptAction(ACTION_DOWN)) {
        m_ActionQueue.PushWithOmit(ACTION_DOWN);
    }
}

void ButtonBase::Cancel()
{
    if (IsAcceptAction(ACTION_CANCEL)) {
        m_ActionQueue.PushWithOmit(ACTION_CANCEL);
    }
}

void ButtonBase::ForceOff()
{
    m_ActionQueue.Clear();
    ForceChangeState(STATE_OFF_IDLE);
}

void ButtonBase::ForceOn()
{
    m_ActionQueue.Clear();
    ForceChangeState(STATE_ON_IDLE);
}

void ButtonBase::ForceDown()
{
    m_ActionQueue.Clear();
    ForceChangeState(STATE_DOWN_IDLE);
}

void ButtonBase::Update()
{
    ProcessActionFromQueue();

    switch (m_State) {
    case STATE_OFF_IDLE:
        break;
    case STATE_ON:
        if (UpdateOn()) {
            FinishOn();
            ProcessActionFromQueue();
        }
        break;
    case STATE_OFF:
        if (UpdateOff()) {
            FinishOff();
            ProcessActionFromQueue();
        }
        break;
    case STATE_ON_IDLE:
        break;
    case STATE_DOWN:
        if (UpdateDown()) {
            FinishDown();
            ProcessActionFromQueue();
        }
        break;
    case STATE_DOWN_IDLE:
        break;
    case STATE_CANCEL:
        if (UpdateCancel()) {
            FinishCancel();
            ProcessActionFromQueue();
        }
        break;
    };
}

void ButtonBase::SetActive(bool active)
{
    m_Flag.ChangeMask(MASK_ACTIVE, active);
}

bool ButtonBase::ProcessOn()
{
    switch (m_State) {
    case STATE_OFF_IDLE:
        ChangeState(STATE_ON);
        StartOn();
        return true;
    case STATE_ON:
        return true;
    case STATE_OFF:
        ChangeState(STATE_ON);
        StartOn();
        return true;
    case STATE_ON_IDLE:
        return true;
    case STATE_DOWN:
        return false;
    case STATE_DOWN_IDLE:
        return true;
    case STATE_CANCEL:
        return false;
    };

    return true;
}

bool ButtonBase::ProcessOff()
{
    switch (m_State) {
    case STATE_OFF_IDLE:
        return true;
    case STATE_ON:
        ChangeState(STATE_OFF);
        StartOff();
        return true;
    case STATE_OFF:
        return true;
    case STATE_ON_IDLE:
        ChangeState(STATE_OFF);
        StartOff();
        return true;
    case STATE_DOWN:
        return false;
    case STATE_DOWN_IDLE:
        ChangeState(STATE_OFF);
        StartOff();
        return true;
    case STATE_CANCEL:
        return true;
    };

    return true;
}

bool ButtonBase::ProcessDown()
{
    switch (m_State) {
    case STATE_OFF_IDLE:
        ChangeState(STATE_ON);
        StartOn();
        return false;
    case STATE_ON:
        return false;
    case STATE_OFF:
        ChangeState(STATE_ON);
        StartOn();
        return false;
    case STATE_ON_IDLE:
        ChangeState(STATE_DOWN);
        StartDown();
        return true;
    case STATE_DOWN:
        return true;
    case STATE_DOWN_IDLE:
        return true;
    case STATE_CANCEL:
        return false;
    };

    return true;
}

bool ButtonBase::ProcessCancel()
{
    switch (m_State) {
    case STATE_OFF_IDLE:
        return true;
    case STATE_ON:
        return true;
    case STATE_OFF:
        return true;
    case STATE_ON_IDLE:
        return true;
    case STATE_DOWN:
        return false;
    case STATE_DOWN_IDLE:
        ChangeState(STATE_CANCEL);
        StartCancel();
        return true;
    case STATE_CANCEL:
        return true;
    };

    return true;
}

bool ButtonBase::UpdateOn()
{
    return true;
}

bool ButtonBase::UpdateOff()
{
    return true;
}

bool ButtonBase::UpdateDown()
{
    return true;
}

bool ButtonBase::UpdateCancel()
{
    return true;
}

void ButtonBase::StartOn()
{
}

void ButtonBase::StartOff()
{
}

void ButtonBase::StartDown()
{
}

void ButtonBase::StartCancel()
{
}

void ButtonBase::FinishOn()
{
    ChangeState(STATE_ON_IDLE);
}

void ButtonBase::FinishOff()
{
    ChangeState(STATE_OFF_IDLE);
}

void ButtonBase::FinishDown()
{
    ChangeState(STATE_DOWN_IDLE);
}

void ButtonBase::FinishCancel()
{
    ChangeState(STATE_OFF_IDLE);
}

void ButtonBase::ChangeState(State state)
{
    m_State = state;
}

void ButtonBase::ForceChangeState(State state)
{
    m_State = state;
}

void ButtonBase::ProcessActionFromQueue()
{
    if ( ! m_ActionQueue.IsEmpty()) {
        bool isPop = false;
        switch (m_ActionQueue.Peek()) {
        case ACTION_ON:
            isPop = ProcessOn();
            break;
        case ACTION_OFF:
            isPop = ProcessOff();
            break;
        case ACTION_DOWN:
            isPop = ProcessDown();
            break;
        case ACTION_CANCEL:
            isPop = ProcessCancel();
            break;
        }
        if (isPop) {
            m_ActionQueue.Pop();
        }
    }
}

bool ButtonBase::IsDowning() const
{
    if (m_State == STATE_DOWN || m_State == STATE_DOWN_IDLE) {
        return true;
    } else {
        return m_ActionQueue.IsDownExist();
    }
}

//! キューの中にDOWNアクションがあるか調査します。
bool ButtonBase::ActionQueue::IsDownExist() const
{
    for (int i = 0; i < m_ActionNum; i++) {
        if (m_Actions[i] == ACTION_DOWN) {
            return true;
        }
    }
    return false;
}

void ButtonBase::ActionQueue::PushWithOmit(Action action)
{
    /*
    アクションキューの中に追加されるのと同じアクションがないか調べ、あるときは
    アクションを追加する代わりに、そのアクションより後のアクションを削除する。
    ないときは、単純に一番後ろに追加する。

    例えば

    ┌────┐
    │         │
    ├────┤
    │   ON    │
    ├────┤
    │  OFF   │
    ├────┤
    │   DOWN  │ <= 先頭のアクション
    └────┘

    の状態でDOWNアクションが入ってきた時は、OFFとONを削除して

    ┌────┐
    │         │
    ├────┤
    │         │
    ├────┤
    │        │
    ├────┤
    │   DOWN  │
    └────┘

    とする。これは、感覚的には「ＡをしてＢをしてＣをしてＡをする」というのを
    結局「Ａをする」というものに置き換えるということである。
    この仕様により、アクションキューには同じアクションは１つまでしか存在しない。
    さらに、アクションキューの最大要素数がアクションの数（現在は４つ）だけ必要である理由にもなっている。
    */
    for (int i = 0; i < m_ActionNum; i++) {
        if (m_Actions[i] == action) {
            m_ActionNum = i + 1;
            return;
        }
    }

    // 同じアクションはアクションキューには一つしか存在しない、ということはこの時点でキューがいっぱい
    // ということはないはずだが、念のため確認する。
    if (m_ActionNum < QUEUE_SIZE) {
        m_Actions[m_ActionNum] = action;
        m_ActionNum++;
    } else {
        NW_ERR("action queue is full");
    }
}

void ButtonBase::ActionQueue::Pop()
{
    if (m_ActionNum > 0) {
        for (int i = 0; i < m_ActionNum - 1; i++) {
            m_Actions[i] = m_Actions[i + 1];
        }
        m_ActionNum--;
    } else {
        NW_ERR("action queue is empty");
    }
}

} // namespace nw::ctrl
} // namespace nw
