﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_OS_H_
#define NW_WINEXT_OS_H_

#ifndef _ASSEMBLER

#include <winext/types.h>

#include <stdarg.h>

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
    Round APIs
 *---------------------------------------------------------------------------*/

// Most subsystems require 32 byte alignment
#define OSRoundUp32B(x)       (((u32)(x) + 32 - 1) & ~(32 - 1))
#define OSRoundDown32B(x)     (((u32)(x)) & ~(32 - 1))

/*---------------------------------------------------------------------------*
    Basic OS API
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
  Name:         OSGetConsoleType

  Description:  Checks the console type.

  Arguments:    None.

  Returns:      Returns one of the OS_CONSOLE_* values

 *---------------------------------------------------------------------------*/
u32     OSGetConsoleType( void );

#if 0
/*---------------------------------------------------------------------------*
  Name:         OSGetSecurityLevel

  Description:  Get the security level.

  Arguments:    None.

  Returns:      OS_SECURITY_LEVEL_PROD or OS_SECURITY_LEVEL_DEV

 *---------------------------------------------------------------------------*/

#define	OS_SECURITY_LEVEL_PROD	0
#define	OS_SECURITY_LEVEL_DEV	1

u32     OSGetSecurityLevel( void );
#endif

/*---------------------------------------------------------------------------*
  Name:         OSIsDebuggerPresent

  Description:  Indicate if debugger is initialized and connected (open
  				and running).

  Arguments:    None.

  Returns:      TRUE if debugger is connected
                FALSE if debugger is not connected

 *---------------------------------------------------------------------------*/
BOOL    OSIsDebuggerPresent( void );

/*---------------------------------------------------------------------------*
  Name:         OSIsDebuggerInitialized

  Description:  Indicate if debugger is initialized but not necessarily
                connected (open and running).

                When the debugger is initialized you can call OS debug
                functions like OSDebug() or OSDebugStr().

                Calling a OS debug function when OSIsDebuggerInitialized() is
                true, and OSIsDebuggerPresent() is false, will cause the
                process to stop and wait for the debugger to connect.

                Calling any OS debug function when OSIsDebuggerInitialized is
                false will cause a fatal un-handled exception.

  Arguments:    None.

  Returns:      TRUE if debugger is initialized
                FALSE if debugger is not initialized

 *---------------------------------------------------------------------------*/
BOOL    OSIsDebuggerInitialized( void );


/* foreground switching */
void OSSavesDone_ReadyToRelease(void);
void OSReleaseForeground( void );
BOOL OSGetForegroundBucket( void ** appRetArea, u32 * apRetSizeBytes);
BOOL OSGetForegroundBucketFreeArea( void ** appRetArea, u32 * apRetSizeBytes);
#if 0
BOOL OSRequestFastExit(u32 aFlags, BOOL bRunExit);
#endif
BOOL OSSendAppSwitchRequest(PFID aProcId, void *apArgs, u32 aArgsBytes);
BOOL OSGetCallArgs(PFID *apRetCaller, u8 *apBuffer, u32 aBufferBytes);
#if 0
BOOL OSSetScreenCapturePermission(BOOL enabled);
BOOL OSGetScreenCapturePermission(void);
BOOL OSEnableHomeButtonMenu(BOOL enable);
BOOL OSIsColdBoot(void);
BOOL OSIsHomeButtonMenuEnabled(void);
#endif

#define APP_IN_FOREGROUND (OSGetForegroundBucket(NULL, NULL))

/* system policy notifications */
typedef enum _osPolicyRequest
{
    OSPOLICY_Exit=1,
    OSPOLICY_NotifyNetIoStart=2,
    OSPOLICY_NotifyNetIoEnd=3,
    OSPOLICY_NotifyHomeButtonDenied=4,
} OSPolicyRequest;
BOOL OSSendPolicyRequest(OSPolicyRequest aPolicy, PFID aTargetProcId);

void    OSRegisterVersion( const char* id );

void    OSGetArgcArgv(int *apRetArgc, char const ***apRetArgv);

} // namespace winext
} // namespace internal
} // namespace nw

/*---------------------------------------------------------------------------*
    Real-time Clock API
 *---------------------------------------------------------------------------*/

#include <winext/cafe/os/OSTime.h>

namespace nw {
namespace internal {
namespace winext {

/*----------------------------------------------------------------------*
    Debug API
 *----------------------------------------------------------------------*/

#define OSHalt(msg)             OSPanic(__FUNCTION__, __LINE__, msg)

#if 0
//#ifdef _DEBUG

#ifndef ASSERT
#define ASSERT(exp)                                             \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, "Failed assertion " #exp), 0))
#endif

#ifndef ASSERTMSG
#if defined(__STDC_VERSION__) && (199901L <= __STDC_VERSION__) || defined(__ghs__)
#define ASSERTMSG(exp, ...)                                     \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, __VA_ARGS__), 0))
#else
#define ASSERTMSG(exp, msg)                                     \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, (msg)), 0))
#endif
#endif

#ifndef ASSERTMSG1
#define ASSERTMSG1(exp, msg, param1)                            \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, (msg),                 \
                     (param1)), 0))
#endif

#ifndef ASSERTMSG2
#define ASSERTMSG2(exp, msg, param1, param2)                    \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, (msg),                 \
                     (param1), (param2)), 0))
#endif

#ifndef ASSERTMSG3
#define ASSERTMSG3(exp, msg, param1, param2, param3)            \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, (msg),                 \
                     (param1), (param2), (param3)), 0))
#endif

#ifndef ASSERTMSG4
#define ASSERTMSG4(exp, msg, param1, param2, param3, param4)    \
    (void) ((exp) ||                                            \
            (OSPanic(__FILE__, __LINE__, (msg),                 \
                     (param1), (param2), (param3), (param4)), 0))
#endif


#else   // _DEBUG

#ifndef ASSERT
#define ASSERT(exp)                                             ((void) 0)
#endif

#ifndef ASSERTMSG
#if defined(__STDC_VERSION__) && (199901L <= __STDC_VERSION__) || defined(__ghs__)
#define ASSERTMSG(exp, ...)                                     ((void) 0)
#else
#define ASSERTMSG(exp, msg)                                     ((void) 0)
#endif
#endif

#ifndef ASSERTMSG1
#define ASSERTMSG1(exp, msg, param1)                            ((void) 0)
#endif
#ifndef ASSERTMSG2
#define ASSERTMSG2(exp, msg, param1, param2)                    ((void) 0)
#endif
#ifndef ASSERTMSG3
#define ASSERTMSG3(exp, msg, param1, param2, param3)            ((void) 0)
#endif
#ifndef ASSERTMSG4
#define ASSERTMSG4(exp, msg, param1, param2, param3, param4)    ((void) 0)
#endif

//#endif  // _DEBUG

#ifndef OS_STATIC_ASSERT
    #define OS_STATIC_ASSERT(e, message)   \
        typedef int os_static_assert__failed__##message[!(e) ? -1 : 1]
#endif
#endif

void OSPanic  ( const char* file, int line, const char* msg, ... );
void OSReport ( const char* msg, ... );
void OSVReport( const char* msg, va_list list );
void OSConsoleWrite( const char *buf, unsigned long len );
void OSFatal  ( const char* msg );
void OSReportSetNewline(const char *newline);
#if 0
void OSSupressConsoleOutput(BOOL umSuppress, BOOL appSuppress, BOOL kmSuppress);

/*
 * caferun -v <n> : set verbose level (0=off 7 == MAX) determines whether these
 * versions of OSReport emit anything or are skipped.
 */
void OSReportWarn( const char* msg, ...);
void OSReportInfo( const char* msg, ...);
void OSReportVerbose( const char* msg, ...);
#endif

#ifdef __ghs__
/* made this macros to avoid problems if inlining is off
 * note: do { ... } while (NN_STATIC_CONDITION(0))
 * executes body of loop exactly once, and avoids problems
 * if expanded in an if statement like:
 * if (x != 0) OSDebugStr("failed");
 */
#define OSDebugStr(message) do { \
    __SETREG(3, (unsigned int) message); \
    asm("tw 31,r31,r31"); \
} while (NN_STATIC_CONDITION(0))
#define OSDebug() OSDebugStr(0)

#define OSDebugMessageBytes(message, count) do { \
    __SETREG(3, (unsigned int) message); \
    __SETREG(4, count); \
    asm("tw 31,r30,r30"); \
} while (NN_STATIC_CONDITION(0))
#define OSDebugMessage(message) OSDebugMessageBytes(message, 0)
#define OSDebugStrIfConnected(message) do { if (OSIsDebuggerPresent()) OSDebugStr(message); } while (NN_STATIC_CONDITION(0))
#else
extern void OSDebug(void);
extern void OSDebugStr(const char *message);
extern void OSDebugMessage(const char *message);
extern void OSDebugMessageBytes(const char *message, unsigned int count);
extern void OSDebugStrIfConnected(const char *message);
#endif

/*---------------------------------------------------------------------------*
    Get infomation APIs
 *---------------------------------------------------------------------------*/

const char* OSGetAppGamename(void);
const u8    OSGetAppType(void);
#define  OS_APP_TYPE_WC24             0x20
#define  OS_APP_TYPE_IPL              0x40
#define  OS_APP_TYPE_DVD              0x80
#define  OS_APP_TYPE_NAND             0x81

} // namespace winext
} // namespace internal
} // namespace nw

/*---------------------------------------------------------------------------*
    Advanced OS APIs
 *---------------------------------------------------------------------------*/

#include <winext/cafe/os/OSCore.h>      // Processor Core routines and defines
//#include <winext/cafe/os/OSSync.h>      // Synchronization routines and defines
//#include <winext/cafe/os/OSAlarm.h>     // Alarm routines and defines
//#include <winext/cafe/os/OSAtomic.h>    // Atomic operation
//#include <winext/cafe/os/OSCache.h>     // Cache routines and defines
//#include <winext/cafe/os/OSContext.h>   // Context structures and defines
//#include <winext/cafe/os/OSDeviceThread.h> // Threads for device drivers and user callback functions
#include <winext/cafe/os/OSError.h>     // Error handler routines and defines
//#include <winext/cafe/os/OSFastCast.h>  // Fast float/int conversion
#include <winext/cafe/os/OSMessage.h>   // Message routines and defines
#include <winext/cafe/os/OSMutex.h>     // Mutex routines and defines
//#include <winext/cafe/os/OSSemaphore.h> // Semaphore routines and defines
#include <winext/cafe/os/OSEvent.h>
//#include <winext/cafe/os/OSSystemInfo.h>
#include <winext/cafe/os/OSThread.h>    // Thread routines and defines
//#include <winext/cafe/os/OSReset.h>     // Reset APIs
//#include <winext/cafe/os/OSDynLoad.h>
//#include <winext/cafe/os/OSFunctionType.h>
//#include <winext/cafe/os/OSPerformanceMonitor.h>
//#include <winext/cafe/os/OSLaunch.h>
//#include <winext/cafe/os/OSCoroutine.h>
//#include <winext/cafe/os/OSMemory.h>
//#include <winext/cafe/os/OSMemmap.h>    // Memory mapping APIs
//#include <winext/cafe/os/OSPlatform.h>  // Platform information
//#include <winext/cafe/os/OSSysHealth.h>
#include <winext/cafe/os/OSInterrupt.h>

#define WEAK_SYMBOL

#endif // _ASSEMBLER
#endif // NW_WINEXT_OS_H_

