﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_SHAREDPTR_H_
#define NW_UT_SHAREDPTR_H_

#include <nw/types.h>
#include <nw/ut/ut_LockObject.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_ScopedLock.h>

namespace nw {
namespace ut {

//! @briefprivate
typedef nw::ut::NullLockObject DefaultLockObject;

//---------------------------------------------------------------------------
//! @briefprivate
//! @details       シェアードポインタのカウンタクラスです。
//---------------------------------------------------------------------------
template <typename TLockObject = DefaultLockObject>
class SharedPtrCount
{
public:
    //---------------------------------------------------------------------------
    //! @brief        参照カウントを生成します。
    //!
    //! @param[out]   pAllocator メモリを確保するアロケータです。
    //!
    //! @return       生成した参照カウントクラスのインスタンスです。
    //---------------------------------------------------------------------------
    static SharedPtrCount*  Create( nw::ut::IAllocator* pAllocator = NULL )
    {
        if ( pAllocator )
        {
            void* buf = pAllocator->Alloc( sizeof(SharedPtrCount) );
            return new(buf) SharedPtrCount( pAllocator );
        }
        else
        {
            return new SharedPtrCount( NULL );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        参照カウントを破棄します。
    //---------------------------------------------------------------------------
    void Destroy()
    {
        if ( m_pAllocator )
        {
            this->~SharedPtrCount();
            m_pAllocator->Free( this );
        }
        else
        {
            delete this;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        参照カウントをインクリメントします。
    //---------------------------------------------------------------------------
    void AddRef()
    {
        ScopedLock<TLockObject> lock(m_LockObject);
        ++m_Use;
    }

    //---------------------------------------------------------------------------
    //! @brief        参照カウントが 0 でない場合に、参照カウントをインクリメントします。
    //!
    //! @return       インクリメントに成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool TestAddRef()
    {
        ScopedLock<TLockObject> lock(m_LockObject);

        if (m_Use == 0)  // 既にオブジェクトは破棄している
        {
            return false;
        }
        else
        {
            ++m_Use;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        参照カウントをデクリメントします。
    //!
    //! @return       残りの参照カウント数です。
    //---------------------------------------------------------------------------
    s32 Release()
    {
        ScopedLock<TLockObject> lock(m_LockObject);

        return --m_Use;
    }

    //---------------------------------------------------------------------------
    //! @brief        弱参照のカウントをインクリメントします。
    //---------------------------------------------------------------------------
    void WeakAddRef()
    {
        ScopedLock<TLockObject> lock(m_LockObject);

        ++m_Weak;
    }

    //---------------------------------------------------------------------------
    //! @brief        弱参照のカウントをデクリメントします。
    //!
    //! @return       残りの弱参照カウントです。
    //---------------------------------------------------------------------------
    s32 WeakRelease()
    {
        ScopedLock<TLockObject> lock(m_LockObject);

        return --m_Weak;
    }

    //---------------------------------------------------------------------------
    //! @brief        参照カウントを取得します。
    //!
    //! @return       参照カウント数です。
    //---------------------------------------------------------------------------
    s32 GetUseCount() const { return m_Use; }

    //---------------------------------------------------------------------------
    //! @brief        弱参照のカウントを取得します。
    //!
    //! @return       弱参照のカウント数です。
    //---------------------------------------------------------------------------
    s32 GetWeakCount() const { return m_Weak; }

private:

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[out]   pAllocator 破棄時に利用するアロケータです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit SharedPtrCount( nw::ut::IAllocator* pAllocator )
    :   m_Use( 0 ),
        m_Weak( 0 ),
        m_pAllocator( pAllocator )
    {
        m_LockObject.Initialize();
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~SharedPtrCount()
    {
        m_LockObject.Finalize();
    }

    s32                 m_Use;
    s32                 m_Weak;
    nw::ut::IAllocator* m_pAllocator;

    TLockObject  m_LockObject;
    template <typename U, typename ULockObject, typename UDisposer> friend class SharedPtr;
};


//--------------------------------------------------------------------------
//! @brief      オブジェクトを破棄するデフォルトのクラスです。
//---------------------------------------------------------------------------
class
SharedPtrDefaultDisposer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        オブジェくを破棄するスタティック関数です。
    //!
    //! @param[out]   obj       破棄するオブジェクトへのポインタです。
    //! @param[out]   allocator アロケータです。
    //---------------------------------------------------------------------------
    template <typename TObj>
    static void Dispose(TObj* obj, nw::ut::IAllocator* allocator)
    {
        if ( allocator )
        {
            obj->~TObj();
            allocator->Free( obj );
        }
        else
        {
            delete obj;
        }
    }
};

template < typename TObj, typename TLockObject = DefaultLockObject, typename TDisposer = SharedPtrDefaultDisposer > class WeakPtr;

//--------------------------------------------------------------------------
//! @brief      参照カウント式のシェアードポインタクラスです。
//!
//! @details    このクラスを継承して使用する事は非サポートです。
//---------------------------------------------------------------------------
template < typename TObj, typename TLockObject = DefaultLockObject, typename TDisposer = SharedPtrDefaultDisposer >
class SharedPtr
{
public:
    //---------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */  SharedPtr() : m_pObj( NULL ), m_pCnt( NULL ), m_pAllocator( NULL ) {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[out]   pObj       シェアードポインタを作成するオブジェクトのポインタです。
    //! @param[out]   pAllocator メモリを開放するポインタです。
    //! @param[out]   pCnt       参照カウント用オブジェクトです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit SharedPtr(TObj* pObj, nw::ut::IAllocator* pAllocator = NULL, SharedPtrCount<TLockObject>* pCnt = NULL)
     : m_pObj( pObj ),
       m_pCnt( pCnt ),
       m_pAllocator( pAllocator )
    {
        if ( pObj == NULL )
        {
            m_pCnt = NULL;
            return;
        }

        if ( m_pCnt == NULL )
        {
            m_pCnt = SharedPtrCount<TLockObject>::Create( pAllocator );
        }
        m_pCnt->AddRef();
        m_pCnt->WeakAddRef();
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    other   コピー元のシェアードポインタです。
    //---------------------------------------------------------------------------
    /* ctor */ /* implicit */ SharedPtr(const SharedPtr& other)
     : m_pObj( other.m_pObj ),
       m_pCnt( other.m_pCnt ),
       m_pAllocator( other.m_pAllocator )
    {


        if (m_pCnt)
        {
            m_pCnt->AddRef();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    other    コピー元のシェアードポインタです。
    //---------------------------------------------------------------------------
    template <typename Y>
    /* ctor */ /* implicit */ SharedPtr<TObj, TLockObject, TDisposer>(const SharedPtr<Y, TLockObject, TDisposer>& other)
     : m_pObj( other.m_pObj ),
       m_pCnt( other.m_pCnt ),
       m_pAllocator( other.m_pAllocator )
    {
        if (m_pCnt)
        {
            m_pCnt->AddRef();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    other   コピー元のウィークポインタです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit       SharedPtr(const WeakPtr<TObj, TLockObject, TDisposer>& other)
     : m_pObj( other.m_pObj ),
       m_pCnt( other.m_pCnt ),
       m_pAllocator( other.m_pAllocator )
    {
        if (m_pCnt && m_pCnt->TestAddRef())
        {
        }
        else
        {
            m_pCnt = NULL;
            m_pObj = NULL;
            m_pAllocator = NULL;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~SharedPtr()
    {
        if ( m_pCnt )
        {
            if ( ! m_pCnt->Release() )
            {
                TDisposer::Dispose( m_pObj, m_pAllocator );

                if ( ! m_pCnt->WeakRelease() )
                {
                    m_pCnt->Destroy();
                }
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        代入演算子です。
    //!
    //! @param[in]    other   コピー元のシェアードポインタです。
    //!
    //! @return       コピー先への参照を返します。
    //---------------------------------------------------------------------------
    SharedPtr&   operator = (const SharedPtr& other)
    {
        SharedPtr(other).SwapObjPtr(*this);
        return * this;
    }

    //---------------------------------------------------------------------------
    //! @brief        代入演算子です。
    //!
    //! @param[in]    other   コピー元のシェアードポインタです。
    //!
    //! @return       コピー先への参照を返します。
    //---------------------------------------------------------------------------
    template <typename U>
    SharedPtr<TObj, TLockObject, TDisposer>&   operator = (const SharedPtr<U, TLockObject, TDisposer>& other)
    {
        SharedPtr(other).SwapObjPtr(*this);
        return * this;
    }

    //---------------------------------------------------------------------------
    //! @brief        シェアードポインタの参照先をリセットします。
    //!
    //! @param[out]   pObj       シェアードポインタを作成するオブジェクトのポインタです。
    //! @param[out]   pAllocator メモリを開放するポインタです。
    //! @param[out]   pCnt       参照カウント用オブジェクトです。
    //!
    //! @return       リセットを実行したシェアードポインタの参照を返します。
    //---------------------------------------------------------------------------
    SharedPtr&         Reset(TObj* pObj = NULL, nw::ut::IAllocator* pAllocator = NULL, SharedPtrCount<TLockObject>* pCnt = NULL)
    {
        SharedPtr(pObj, pAllocator, pCnt).SwapObjPtr(*this);
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        参照しているオブジェクトを取得します。
    //!
    //! @return       参照しているオブジェクトのポインタです。
    //---------------------------------------------------------------------------
    TObj*              Get() const { return m_pObj; }

    //---------------------------------------------------------------------------
    //! @brief        参照しているオブジェクトを取得します。
    //!
    //! @return       参照しているオブジェクトのリファレンスです。
    //---------------------------------------------------------------------------
    TObj&              operator * () const { NW_ASSERT_NOT_NULL( m_pObj ); return  *m_pObj; }

    //---------------------------------------------------------------------------
    //! @brief        参照しているオブジェクトを取得します。
    //!
    //! @return       参照しているオブジェクトのポインタです。
    //---------------------------------------------------------------------------
    TObj*              operator -> () const { NW_ASSERT_NOT_NULL( m_pObj ); return m_pObj; }

    //---------------------------------------------------------------------------
    //! @brief        参照しているオブジェクトが NULL でないかどうかを取得します。
    //!
    //! @return       参照しているオブジェクトが NULL であれば false, それ以外の場合は true を返します。
    //---------------------------------------------------------------------------
    operator bool () const { return NULL != m_pObj; }

    // デバッグ用
    //---------------------------------------------------------------------------
    //! @brief        デバッグ用関数です。参照カウントの値を取得します。
    //!
    //! @return       参照カウントの値です。
    //---------------------------------------------------------------------------
    s32                RefCount() const
    {
        if (m_pCnt == NULL) { return 0; }
        return m_pCnt->GetUseCount();
    }

private:
    TObj*                        m_pObj;
    SharedPtrCount<TLockObject>* m_pCnt;
    nw::ut::IAllocator*          m_pAllocator;

    //---------------------------------------------------------------------------
    //! @brief        ポインタの指す先をスワップする関数です。
    //!
    //! @param[out]   other   スワップ対象のシェアードポインタです。
    //---------------------------------------------------------------------------
    void                SwapObjPtr(SharedPtr& other)
    {
        std::swap( m_pObj, other.m_pObj );
        std::swap( m_pCnt, other.m_pCnt );
        std::swap( m_pAllocator, other.m_pAllocator );
    }

    template <typename U, typename ULockObject, typename UDisposer> friend class SharedPtr;
    template <typename U, typename ULockObject, typename UDisposer> friend class WeakPtr;
};

//----------------------------------------
//! @name スマートポインタ操作関連
//@{

//---------------------------------------------------------------------------
//! @brief        SharedPtr が同じかどうかを判定します。
//!
//! @param[in]    a       比較対象となる1つ目のオブジェクトです。
//! @param[in]    b       比較対象となる2つ目のオブジェクトです。
//!
//! @return       ポインタが等しい場合は true、異なる場合は false を返します。
//---------------------------------------------------------------------------
template <class T, class U, typename TLockObject, typename TDisposer>
NW_INLINE bool
operator==(SharedPtr<T, TLockObject, TDisposer> const & a, SharedPtr<U, TLockObject, TDisposer> const & b)
{
    return a.Get() == b.Get();
}

//---------------------------------------------------------------------------
//! @brief        SharedPtr が異なるかどうかを判定します。
//!
//! @param[in]    a       比較対象となる1つ目のオブジェクトです。
//! @param[in]    b       比較対象となる2つ目のオブジェクトです。
//!
//! @return       ポインタが異なる場合は true、等しい場合は false を返します。
//---------------------------------------------------------------------------
template <class T, class U, typename TLockObject, typename TDisposer>
NW_INLINE bool
operator!=(SharedPtr<T, TLockObject, TDisposer> const & a, SharedPtr<U, TLockObject, TDisposer> const & b)
{
    return ! (a == b);
}

//@}

//--------------------------------------------------------------------------
//! @brief      SharedPtr への弱参照クラスです。
//!
//! @details    このクラスを継承して使用する事は非サポートです。
//---------------------------------------------------------------------------
template <typename TObj, typename TLockObject, typename TDisposer>
class WeakPtr
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */  WeakPtr()  : m_pObj( NULL ), m_pCnt( NULL ), m_pAllocator( NULL )
    {
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    other   コピー元のシェアードポインタです。
    //---------------------------------------------------------------------------
    template <class U>
    /* ctor */ /* implicit */ WeakPtr(const SharedPtr<U, TLockObject, TDisposer>& other)
        :   m_pObj(other.m_pObj),
            m_pCnt(other.m_pCnt),
            m_pAllocator(other.m_pAllocator)
    {
        if (m_pCnt)
        {
            m_pCnt->WeakAddRef();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    other   コピー元のウィークポインタです。
    //---------------------------------------------------------------------------
    template <class U>
    /* ctor */ /* implicit */ WeakPtr(const WeakPtr<U, TLockObject, TDisposer>& other)
        :   m_pObj(other.m_pObj),
            m_pCnt(other.m_pCnt),
            m_pAllocator(other.m_pAllocator)
    {
        if (m_pCnt)
        {
            m_pCnt->WeakAddRef();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~WeakPtr()
    {
        if ( m_pCnt )
        {
            if ( ! m_pCnt->WeakRelease() )
            {
                m_pCnt->Destroy();
            }
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        ウィークポインタからシェアードポインタを取得します。
    //!
    //! @return       生成したシェアードポインタです。
    //---------------------------------------------------------------------------
    SharedPtr<TObj, TLockObject, TDisposer>    Lock() const
    {
        return SharedPtr<TObj, TLockObject, TDisposer>(*this);
    }

    //---------------------------------------------------------------------------
    //! @brief        代入演算子です。
    //!
    //! @tparam       U ウィークポインタの型です。
    //! @param[in]    rhs コピー元のシェアードポインタです。
    //! @return       コピー先のウィークポインタです。
    //---------------------------------------------------------------------------
    template <class U>
    WeakPtr&           operator = (const SharedPtr<U, TLockObject, TDisposer>& rhs)
    {
        WeakPtr(rhs).SwapObjPtr(*this);
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        代入演算子です。
    //!
    //! @tparam       U ウィークポインタの型です。
    //! @param[in]    other コピー元のウィークポインタです。
    //! @return       コピー先のウィークポインタです。
    //---------------------------------------------------------------------------
    template <class U>
    WeakPtr&            operator = (const WeakPtr<U, TLockObject, TDisposer>& other)
    {
        WeakPtr(other).SwapObjPtr(*this);
        return *this;
    }

private:

    //---------------------------------------------------------------------------
    //! @brief        ポインタの指す先をスワップする関数です。
    //!
    //! @param[out]   other   スワップ対象のウィークポインタです。
    //---------------------------------------------------------------------------
    void                SwapObjPtr(WeakPtr& other)
    {
        std::swap( m_pObj, other.m_pObj );
        std::swap( m_pCnt, other.m_pCnt );
        std::swap( m_pAllocator, other.m_pAllocator );
    }

    TObj*                        m_pObj;
    SharedPtrCount<TLockObject>* m_pCnt;
    nw::ut::IAllocator*          m_pAllocator;

    template <typename U, typename ULockObject, typename UDisposer> friend class SharedPtr;
    template <typename U, typename ULockObject, typename UDisposer> friend class WeakPtr;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_SHAREDPTR_H_
