﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_SAFE_STRING_H_
#define NW_UT_SAFE_STRING_H_

#include <cstring>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_String.h>

namespace nw {
namespace ut {

//-----------------------------------------------------------------------------
// 前方宣言.
template <typename CharType>
class BufferedSafeStringBase;

/**
 * 文字列を安全に扱うクラスの基底.
 *
 */
template <typename CharType>
class SafeStringBase
{
public:
    typedef CharType char_type;

    //!
    //! NULL文字を表します.
    //!
    static const CharType cNullChar;

    //!
    //! NULL文字列を表します.
    //!
    static const CharType cNullString[1];

    //!
    //! 改行文字を表します.
    //!
    static const CharType cLineBreakChar;

    //!
    //! 空文字列(文字列長0の文字列)を返します.
    //!
    static const SafeStringBase cEmptyString;

    //!
    //! 最大で扱える文字列の長さを表します.
    //!
    // TODO: XmlDocumentで扱えるコンテンツのサイズを拡張するため、一時的に最大値を変更します。
    static const s32 cMaximumLength = 256*1024;

public:
    //-------------------------------------------------------------------------
    //! @brief 空文字列を指すコンストラクタです。
    //-------------------------------------------------------------------------
    SafeStringBase()
        : mStringTop( cNullString )
    {}

    //-------------------------------------------------------------------------
    //! @brief 文字列を指すコンストラクタです。
    //!
    //! 暗黙の型変換を許します。
    //! 文字列はヌル終端されていなければなりません。
    //!
    //! @param[in]      str     文字列の先頭を指すポインタです。
    //-------------------------------------------------------------------------
    /* implicit */ SafeStringBase( const CharType* str )
        : mStringTop( str )
    {
        NW_ASSERT_NOT_NULL( mStringTop );
    }

    //-------------------------------------------------------------------------
    //! @brief デストラクタです。
    //-------------------------------------------------------------------------
    virtual ~SafeStringBase()
    {}

public:
    /**
     * @brief イテレータです。
     */
    class iterator
    {
    public:
        //---------------------------------------------------------------------
        //! @brief コンストラクタです。
        //!
        //! @param[in] str 指す文字列を設定します。
        //! @details 位置は0に初期化されます。
        //---------------------------------------------------------------------
        explicit iterator( const SafeStringBase<CharType>* str )
            : mString( str )
            , mIndex( 0 )
        {}

        //---------------------------------------------------------------------
        //! @brief コンストラクタです。
        //!
        //! @param[in] str 指す文字列を設定します。
        //! @param[in] idx 初期位置を設定します。
        //---------------------------------------------------------------------
        iterator( const SafeStringBase<CharType>* str, s32 idx )
            : mString( str )
            , mIndex( idx )
        {}

        virtual ~iterator()
        {}

    public:
        //---------------------------------------------------------------------
        //! @brief        イテレータの前置のインクリメント演算子です。
        //!
        //! @return       インクリメントされたイテレータを返します。
        //---------------------------------------------------------------------
        inline iterator& operator++()
        {
            ++mIndex;
            return *this;
        }

        //---------------------------------------------------------------------
        //! @brief        イテレータの前置のデクリメント演算子です。
        //!
        //! @return       デクリメントされたイテレータを返します。
        //---------------------------------------------------------------------
        inline iterator& operator--()
        {
            --mIndex;
            return *this;
        }

        //---------------------------------------------------------------------
        //! @brief 2つのイテレータが同じものを指すかを返します。
        //!
        //! @param[in]  a       イテレータです。
        //! @param[in]  b       イテレータです。
        //!
        //! @return     2つのイテレータが同じものであれば true を返します。
        //---------------------------------------------------------------------
        friend bool operator==( const iterator& a, const iterator& b )
        {
            return ( a.mString == b.mString && a.mIndex == b.mIndex );
        };

        // TODO: compiler_bug  operator==と引数名が同じだと怒られてしまう
        friend bool operator!=( const iterator& a_, const iterator& b_ )
        {
            return !( a_==b_ );
        }

        const CharType& operator*() const
        {
            return mString->at( mIndex );
        }

        s32 getIndex() const
        {
            return mIndex;
        }

    protected:
        const SafeStringBase*   mString;
        s32 mIndex; ///< 場所.
    };

    //-----------------------------------------------------------------------------

    /**
     *  @brief 区切り文字ごとに移動するイテレータ.
     */
    class token_iterator : public iterator
    {
    public:
        /**
         *  コンストラクタ. 指す文字列と初期位置と区切り文字を指定.
         *
         *  @param[in]      str                     参照先文字列.
         *  @param[in]      delimiter       区切り文字を並べたリスト.
         */
        token_iterator( const SafeStringBase* str, const SafeStringBase& delimiter )
            : iterator( str )
            , mDelimiter( delimiter )
        {}

        token_iterator( const SafeStringBase* str, s32 idx, const SafeStringBase& delimiter )
            : iterator( str, idx )
            , mDelimiter( delimiter )
        {}

        virtual ~token_iterator()
        {}

    public:
        //-------------------------------------------------------------------------
        //! @brief        イテレータの前置のインクリメント演算子です。
        //!
        //! @return       インクリメントされたイテレータを返します。
        //!
        //! 次の区切り文字まで進みます。
        //! - 既に文字列終端を指している場合は何もしません。
        //-------------------------------------------------------------------------
        inline token_iterator& operator++();

        //-------------------------------------------------------------------------
        //! @brief        イテレータの前置のデクリメント演算子です。
        //!
        //! @return       デクリメントされたイテレータを返します。
        //!
        //! 前の区切り文字まで戻ります。
        //! - 既に文字列先頭を指している場合は何もしません.
        //-------------------------------------------------------------------------
        inline token_iterator& operator--();

        //-------------------------------------------------------------------------
        //! @brief 現在の位置から、次の区切り文字までの間の文字列を引数の文字列に上書きする.
        //! @param[out] out 上書きされる文字列です。
        //! @return 上書きした長さです。
        //-------------------------------------------------------------------------
        inline s32 get( BufferedSafeStringBase<CharType>* out ) const;

        //-------------------------------------------------------------------------
        //! @brief 現在の位置から、次の区切り文字までの間の文字列を引数の文字列に上書きし、iteratorを進める.
        //!
        //! @param[out] out 上書きするバッファです。
        //! @return 書き込んだ文字数.
        //!
        //! @details getのあとにインクリメントするよりも高速です.
        //-------------------------------------------------------------------------
        inline s32 getAndForward( BufferedSafeStringBase<CharType>* out );

    private:
        SafeStringBase mDelimiter;  ///< 区切り文字列.
    };

public:
    //! @brief 開始イテレータの取得.
    //! @return 開始イテレータです。
    iterator begin() const
    {
        return iterator( this, 0 );
    };

    /**
     *  @brief 終端イテレータの取得.
     *
     *  @details 関数内でcalcLengthを呼びます.
     *  ループ処理の終了条件として使う場合は、ループの外で一度だけ呼ぶようにすると効率的です.
     *  @return 終端イテレータです。
     *  @code
     *  SafeString::iterator end = mString.end();
     *  for( SafeString::iterator it = mString.begin(); it != end; it++ )
     *  {
     *      // 中略
     *  }
     *  @endcode
     */
    iterator end() const
    {
        return iterator( this, calcLength() + 1 );
    };

public:
    //! @brief 区切り文字ごとに移動する開始イテレータの取得.
    //! @param[in] delimiter 区切り文字を並べたリストです。
    //! @return 開始イテレータです。
    token_iterator tokenBegin( const SafeStringBase<CharType>& delimiter ) const
    {
        return token_iterator( this, 0, delimiter );
    };

    /**
     *  @brief 区切り文字ごとに移動する終了イテレータの取得.
     *
     *  @param[in] delimiter    区切り文字のリストを指定します。
     *  @return 終了イテレータです。
     *  @details
     *  token_iteratorは比較演算子で区切り文字の違いを考慮しないので、単に末尾まで進んだかどうかの比較をするだけなら引数は不要です.
     *  関数内でcalcLengthを呼びます.
     *  ループ処理の終了条件として使う場合は、ループの外で一度だけ呼ぶようにすると効率的です.
     *  @code
     *  SafeString::token_iterator end = mString.tokenEnd();
     *  for( SafeString::token_iterator it = mString.tokenBegin(); it != end; it++ )
     *  {
     *      // 中略
     *  }
     *  @endcode
     */
    token_iterator tokenEnd( const SafeStringBase<CharType>& delimiter = SafeStringBase<CharType>() ) const
    {
        return token_iterator( this, calcLength() + 1, delimiter );
    }

public:
    /**
     *  @brief c形式のポインタに戻す.
     *  @return c形式のポインタです。
     */
    const CharType* c_str() const
    {
        assureTerminationImpl_();
        return mStringTop;
    }

    /**
     *  @brief 指定位置の取得.
     *  @param idx インデックスです。
     *  @return 指定位置の要素です。
     */
    inline const CharType& at( s32 idx ) const;

    /**
     *  @brief 添え字アクセス(const).
     *  @param idx インデックスです。
     *  @return 指定位置の要素です。
     */
    inline const CharType& operator[]( s32 idx ) const
    {
        return at( idx );
    }

    /**
     *  @brief 文字列の長さを取得する.
     *  @return 文字列の長さです。
     */
    inline s32 calcLength() const;

    /**
     *  @brief 文字列の途中から最後までを指す新しいSafeStringBaseを返す.
     *
     *  @return 文字列の途中を参照する参照先.
     *  @param[in]      at      何文字目から先を新しい文字列にするか.
     *  @details この関数が返すのは文字列の途中を参照する(Bufferedでない)SafeStringであり、バッファをクローンするわけではありません.
     */
    const SafeStringBase<CharType> getPart( s32 at ) const
    {
        s32 len = calcLength();
        if ( at < 0 || at > len )
        {
            NW_ERR( "index(%d) out of range[0, %d]", at, len );
            // 範囲外のときは空文字列を返す.
            return SafeStringBase<CharType>::cEmptyString;
        }
        return SafeStringBase<CharType>( mStringTop + at );
    }

    /**
     *  @brief 文字列の途中から最後までを指す新しいSafeStringBaseを返す.
     *
     *  @param[in]      it      どこから先を新しい文字列にするか.
     *  @return 文字列の途中を参照する参照先.
     *  @details この関数が返すのは文字列の途中を参照する(Bufferedでない)SafeStringであり、バッファをクローンするわけではありません.
     */
    const SafeStringBase<CharType> getPart( const iterator& it ) const
    {
        return getPart( it.getIndex() );
    }

    /**
     *  文字列中に引数の文字が含まれているか調べて返す.
     *  @param[in] c 調べる文字です。
     *  @return 文字が含まれている場合 true を返します。
     */
    inline bool include( const CharType& c ) const;

    /**
     *  @brief 文字列中に引数の文字列が含まれているか調べて返す.
     *  @param[in] str 調べる文字列です。
     *  @return 文字列が含まれていれば true を返します。
     */
    inline bool include( const SafeStringBase<CharType>& str ) const
    {
        return ( findIndex( str ) != -1 );
    }

    /**
     *  @brief 文字列が同一かどうか調べる.
     *  @param[in] rhs 比較対象の文字列です。
     *  @return 文字列が同一であれば true を返します。
     */
    inline bool isEqual( const SafeStringBase<CharType>& rhs ) const;

    /**
     *  @brief 文字列が同一かどうか調べる.
     *  @param[in] lhs 比較対象の文字列です。
     *  @param[in] rhs 比較対象の文字列です。
     *  @return 文字列が同一であれば true を返します。
     */
    friend bool operator== ( const SafeStringBase<CharType>& lhs, const SafeStringBase<CharType>& rhs )
    {
        return lhs.isEqual( rhs );
    }

    /**
     *  @brief 文字列が異なるかどうか調べる.
     *  @param[in] lhs 比較対象の文字列です。
     *  @param[in] rhs 比較対象の文字列です。
     *  @return 文字列が異なっていれば true を返します。
     */
    friend bool operator!= ( const SafeStringBase<CharType>& lhs, const SafeStringBase<CharType>& rhs )
    {
        return !lhs.isEqual( rhs );
    }

    /**
     *  @brief 先頭n文字の文字列比較を行う.
     *  @param[in] rhs 比較対象の文字列です。
     *  @param[in] n 文字列比較を行う長さです。
     *  @return 同一であれば 0 を返します。文字列が rhs より小さい場合は負の値を、大きい場合は正の値を返します。
     */
    inline s32 comparen( const SafeStringBase<CharType>& rhs, s32 n ) const;

    /**
     *  @brief 文字列の比較を行う.
     *  @param[in] rhs 比較対象の文字列です。
     *  @return 同一であれば 0 を返します。文字列が rhs より小さい場合は負の値を、大きい場合は正の値を返します。
     */
    inline s32 compare( const SafeStringBase<CharType>& rhs ) const
    {
        return comparen( rhs, cMaximumLength );
    }

    /**
     *  @brief 比較演算子(>).文字列の比較を行います。
     *  @param[in] lhs 比較対象の文字列です。
     *  @param[in] rhs 比較対象の文字列です。
     *  @return lhs が rhs より大きければ true を返します。
     */
    friend bool operator>( const SafeStringBase<CharType>& lhs, const SafeStringBase<CharType>& rhs )
    {
        return lhs.compare( rhs ) > 0;
    }

    /**
     *  比較演算子(<).文字列の比較を行います。
     *  @param[in] lhs 比較対象の文字列です。
     *  @param[in] rhs 比較対象の文字列です。
     *  @return lhs が rhs より小さければ true を返します。
     */
    friend bool operator<( const SafeStringBase<CharType>& lhs, const SafeStringBase<CharType>& rhs )
    {
        return lhs.compare( rhs ) < 0;
    }

    /**
     *  @brief 指定した文字列が最初に現れる場所を探し、その場所へのイテレータを返す.
     *
     *  @param  token   探す文字列.
     *  @return                 文字列が見つかった場合、その文字へのiterator. 見つからなかった場合、end()に等しいiterator.
     */
    inline iterator findIterator( const SafeStringBase<CharType>& token ) const;

    /**
     *  @brief 指定した文字列が最初に現れる場所を探し、その場所への添え字を返す.
     *
     *  @param  token   探す文字列.
     *  @return                 文字列が見つかった場合、その文字への添え字. 見つからなかった場合、-1.
     */
    s32 findIndex( const SafeStringBase<CharType>& token ) const;

    /**
     *  @brief 指定した文字列が最後に現れる場所を探し、その場所へのイテレータを返す.
     *
     *  @param  token   探す文字列.
     *  @return                 文字列が見つかった場合、その文字へのiterator. 見つからなかった場合、end()に等しいiterator.
     */
    inline iterator rfindIterator( const SafeStringBase<CharType>& token ) const;

    /**
     *  @brief 指定した文字列が最後に現れる場所を探し、その場所への添え字を返す.
     *
     *  @param  token   探す文字列.
     *  @return                 文字列が見つかった場合、その文字への添え字. 見つからなかった場合、-1.
     */
    s32 rfindIndex( const SafeStringBase<CharType>& token ) const;
    /**
     *  @brief 文字列が空文字列かどうか返す.
     *  @return 文字列が空文字列であれば true を返します。
     */
    bool isEmpty() const
    {
        return unsafeAt_( 0 ) == cNullChar;
    };

private:
    /**
     *  指定位置の取得(境界チェック無し版).
     */
    const CharType& unsafeAt_( s32 idx ) const
    {
            return mStringTop[ idx ];
    };

protected:
    /**
     *  終端文字があることを保証します.
     *
     *  - BufferedSafeStringBaseで実装されます.
     */
    virtual void assureTerminationImpl_() const {};

protected:
    const CharType*     mStringTop;             ///< 保持する文字列の先頭.
};

template<typename CharType>
const CharType SafeStringBase<CharType>::cNullChar = 0;

template<typename CharType>
const CharType SafeStringBase<CharType>::cLineBreakChar = static_cast<CharType>('\n');

template<typename CharType>
const CharType SafeStringBase<CharType>::cNullString[1] = { SafeStringBase<CharType>::cNullChar };

template<typename CharType>
const SafeStringBase<CharType> SafeStringBase<CharType>::cEmptyString;

template <typename CharType>
inline const CharType&
SafeStringBase<CharType>::at( s32 idx ) const
{
    s32 len = calcLength();
    if ( idx < 0 || idx > len )
    {
        NW_ERR( "index(%d) out of range[0, %d]", idx, len );
        return cNullChar;
    }
    return mStringTop[ idx ];
}

template <typename CharType>
inline s32
SafeStringBase<CharType>::calcLength() const
{
    NW_ASSERT( mStringTop );
    assureTerminationImpl_();
    s32 length = 0;
    for ( ; ; )
    {
        if ( length > cMaximumLength || mStringTop[ length ] == cNullChar )
        {
            break;
        }
        length++;
    }
    if ( length > cMaximumLength )
    {
        // オーバーフロー.
        NW_ERR( "too long string\n" );
        return 0;
    }
    return length;
}

template <typename CharType>
inline bool
SafeStringBase<CharType>::include( const CharType& c ) const
{
    assureTerminationImpl_();
    // calcLengthしないよう修正.
    for( s32 i = 0; i <= cMaximumLength; i++ )
    {
        if ( unsafeAt_( i ) == cNullChar )
        {
            return false;
        }
        if ( unsafeAt_( i ) == c )
        {
            return true;
        }
    }
    return false;
}

template <typename CharType>
inline bool
SafeStringBase<CharType>::isEqual( const SafeStringBase<CharType>& rhs ) const
{
    assureTerminationImpl_();
    if ( c_str() == rhs.c_str() )
    {
        // 先頭が同じなら同一とみなす.
        return true;
    }
    // 1文字ずつ比較.
    // calcLengthしないように修正.
    for( s32 i = 0; i <= cMaximumLength; i++ )
    {
        if ( unsafeAt_( i ) != rhs.unsafeAt_( i ) )
        {
            return false;
        }
        else if ( unsafeAt_( i ) == cNullChar )
        {
            return true;
        }
    }
    // ここに来るのはオーバーフローしたとき.
    NW_ERR( "too long string\n" );
    return false;
}

template <typename CharType>
inline s32
SafeStringBase<CharType>::comparen( const SafeStringBase<CharType>& rhs, s32 n ) const
{
    assureTerminationImpl_();
    if ( c_str() == rhs.c_str() )
    {
        // 先頭が同じなら同一.
        return 0;
    }
    // 比較する長さはcMaximumLength以内.
    if ( n > cMaximumLength )
    {
        NW_ERR( "paramater(%d) out of bounds [0, %d]", n, cMaximumLength );
        n = cMaximumLength;
    }
    // 1文字づつ比較.
    for( s32 i = 0; i < n; i++ )
    {
        if ( unsafeAt_( i ) == cNullChar && rhs.unsafeAt_( i ) == cNullChar )
        {
            return 0;
        }
        else if ( unsafeAt_( i ) == cNullChar )
        {
            return -1;
        }
        else if ( rhs.unsafeAt_( i ) == cNullChar )
        {
            return 1;
        }
        else if ( unsafeAt_( i ) < rhs.unsafeAt_( i ) )
        {
            return -1;
        }
        else if ( unsafeAt_( i ) > rhs.unsafeAt_( i ) )
        {
            return 1;
        }
    }
    return 0;
}

template <typename CharType>
inline typename SafeStringBase<CharType>::iterator
SafeStringBase<CharType>::findIterator( const SafeStringBase<CharType>& token ) const
{
    s32 idx = findIndex( token );
    if ( idx != -1 )
    {
        return SafeStringBase<CharType>::iterator( this, idx );
    }
    else
    {
        return end();
    }
}

template <typename CharType>
inline s32
SafeStringBase<CharType>::findIndex( const SafeStringBase<CharType>& token ) const
{
    s32 len = calcLength();
    s32 token_len = token.calcLength();
    for( s32 i = 0; i <= len - token_len; i++ )
    {
        // getPartを使うよりも下記の実装の方がcalcLengthの回数が減る.
        if ( SafeStringBase<CharType>( &mStringTop[ i ] ).comparen( token, token_len ) == 0 )
        {
            return i;
        }
    }
    return -1;
}

template <typename CharType>
inline typename SafeStringBase<CharType>::iterator
SafeStringBase<CharType>::rfindIterator( const SafeStringBase<CharType>& token ) const
{
    s32 idx = rfindIndex( token );
    if ( idx != -1 )
    {
        return SafeStringBase<CharType>::iterator( this, idx );
    }
    else
    {
        return end();
    }
}

template <typename CharType>
inline s32
SafeStringBase<CharType>::rfindIndex( const SafeStringBase<CharType>& token ) const
{
    s32 len = calcLength();
    s32 token_len = token.calcLength();
    len -= token_len;
    if ( len < 0 )
    {
        return -1;
    }
    for( ; len >= 0; len-- )
    {
        // getPartを使うよりも下記の実装の方がcalcLengthの回数が減る.
        if ( SafeStringBase<CharType>( &mStringTop[ len ] ).comparen( token, token_len ) == 0 )
        {
            break;
        }
    }
    return len;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline typename SafeStringBase<CharType>::token_iterator&
SafeStringBase<CharType>::token_iterator::operator++()
{
    s32 index = this->getIndex();
    s32 length = this->mString->calcLength();
    if ( !( 0 <= index && index <= length ) )
    {
        NW_ERR( "index(%d) out of range [0, %d].\n", index, length );
        return *this;
    }
    for(;;)
    {
        NW_ASSERT( 0 <= index && index <= length );
        const CharType& c = this->mString->unsafeAt_( index );
        if ( c == cNullChar || mDelimiter.include( c ) )
        {
            break;
        }
        index++;
    }
    this->mIndex = index + 1;
    return *this;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline typename SafeStringBase<CharType>::token_iterator&
SafeStringBase<CharType>::token_iterator::operator--()
{
    s32 index = this->getIndex();
    s32 length = this->mString->calcLength();
    if ( index == 0 )
    {
        // 既にbeginの位置なら何もしない.
        return *this;
    }
    else if ( index == 1 )
    {
        // 先頭に区切り文字がある場合対策.
        this->mIndex = 0;
        return *this;
    }
    else if ( !( 0 <= index && index <= length + 1 ) )
    {
        NW_ERR( "index(%d) out of range [0, %d].\n", index, length + 1 );
        return *this;
    }
    index -= 2;
    for(;;)
    {
        NW_ASSERT( 0 <= index && index <= length );
        const CharType& c = this->mString->unsafeAt_( index );
        if ( c == cNullChar || this->mDelimiter.include( c ) )
        {
            break;
        }
        index--;
        if ( index == -1 )
        {
            // 先頭まで来た.
            break;
        }
    }
    this->mIndex = index + 1;
    return *this;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
SafeStringBase<CharType>::token_iterator::get( BufferedSafeStringBase<CharType>* out ) const
{
    token_iterator next_delim = *this;
    ++next_delim;
    s32 size = next_delim.getIndex() - this->getIndex() - 1;
    return out->copy( this->mString->getPart( *this ), size );
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
SafeStringBase<CharType>::token_iterator::getAndForward( BufferedSafeStringBase<CharType>* out )
{
    s32 index = this->getIndex();
    s32 length = this->mString->calcLength();
    if ( !( 0 <= index && index <= length ) )
    {
        NW_ERR( "index(%d) out of range [0, %d].\n", index, length );
        return 0;
    }
    s32 i = 0;
    CharType* raw_out = out->getBuffer();
    for(;;)
    {
        NW_ASSERT( 0 <= index && index <= length );
        const CharType& c = this->mString->unsafeAt_( index );
        if ( c == cNullChar || mDelimiter.include( c ) )
        {
            raw_out[ i ] = cNullChar;
            break;
        }
        raw_out[ i ] = c;
        ++i;
        ++index;
    }
    this->mIndex = index + 1;
    return i;
}

//-----------------------------------------------------------------------------
//! バッファを与えられ、その範囲内で文字列の書き換えが行える文字列クラスです。
//!
//! バッファの終端には、クラス内で自動的にヌル文字を書き込みますので
//! ご注意ください.
//!
//! この終端文字の書き込みはconst関数でも行われる場合があります.
//-----------------------------------------------------------------------------
template <typename CharType>
class BufferedSafeStringBase : public SafeStringBase<CharType>
{
public:
    /**
     *  @brief バッファの先頭と大きさを指定するコンストラクタ.
     *
     *  @param[in]      buffer  バッファの先頭.
     *  @param[in]      size    バッファの大きさは何文字分か.
     *  @details バッファの中身は特に改変せずそのまま使う. ただし、バッファの末尾にNULL文字を入れる処理だけは勝手にやる.
     *  バッファの大きさには1より大きい値を設定しなければならない.
     *  0以下の値を設定すると、エラー(プロダクト版ではnullptrを指す不正な文字列になる).
     */
    BufferedSafeStringBase( CharType* buffer, s32 size )
    {
        this->Construct( buffer, size );
    }

    //-------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @tparam N           配列の要素数です。
    //! @param[in] buffer   文字配列です。
    //!
    //! @details
    //! 文字配列を指定するコンストラクタです。
    //-------------------------------------------------------------------------
    template <s32 N>
    NW_FORCE_INLINE BufferedSafeStringBase( CharType (&buffer)[ N ] )
    {
        this->Construct( buffer, N );
    }

private:
    // コンストラクタの共通の処理です。
    void Construct( CharType* buffer, s32 size )
    {
        NW_ASSERT_NOT_NULL( buffer );
        NW_ASSERTMSG( size > 0, "Invalied buffer size(%d).\n", size );

        if ( buffer && 0 < size )
        {
            this->mStringTop = buffer;
            mBufferSize = size;
            assureTerminationImpl_();
        }
        else
        {
            this->mStringTop = NULL;
            mBufferSize = 0;
        }
    }

public:
    /**
     *  @brief 別のBufferedSafeStringの一部を先頭とするBufferedSafeStringとして作るコンストラクタ.
     *
     *  @param[in] original 元になる BufferedSafeString
     *  @param[in] pos 新しい BufferedSafeString が指す部分文字列の先頭位置
     *  @details バッファは共有されるので、注意する必要がある.
     *  posが元のバッファサイズより大きい場合、エラー(プロダクト版ではnullptrを指す不正な文字列になる).
     */
    BufferedSafeStringBase( BufferedSafeStringBase<CharType>* original, s32 pos )
        : mBufferSize( 0 )
    {
        if ( !original )
        {
            NW_ERR( "original string must not be nullptr." );
            this->mStringTop = NULL;
            return;
        }

        if ( pos < 0 || original->getBufferSize() <= pos )
        {
            NW_ERR( "pos(%d) out of bounds[0,%d)", pos, original->getBufferSize() );
            this->mStringTop = NULL;
            return;
        }

        this->mStringTop = original->getBuffer() + pos;
        mBufferSize = original->getBufferSize() - pos;
        assureTerminationImpl_();
    }

    /**
     *  @brief デストラクタ.
     */
    virtual~ BufferedSafeStringBase()
    {}

private:
    /**
     * コピーコンストラクタは禁止.
     * (実装しない)
     */
    BufferedSafeStringBase( const BufferedSafeStringBase<CharType>& );

    /**
     * 代入も禁止.
     * (実装しない)
     */
    const BufferedSafeStringBase<CharType>& operator = ( const BufferedSafeStringBase<CharType>& );

public:
    /// @name 取得系.
    /// @{
    /**
     *  @brief c形式のポインタに戻す.
     *  @return c形式のポインタです。
     */
    const CharType* c_str() const
    {
        assureTerminationImpl_();
        return this->mStringTop;
    }

    /**
     *  @brief 添え字アクセス(const).
     *  @param[in] idx インデックスです。
     *  @return インデックスに該当する文字です。
     */
    inline const CharType& operator[]( s32 idx ) const;

    /**
     *  @brief バッファサイズを取得する.
     *  @return バッファサイズです。
     */
    s32 getBufferSize() const
    {
        return mBufferSize;
    }

    /// @}

    /// @name copy系.
    /// @{
    /**
     *  別のSafeStringの一部で上書きする.
     *
     *  引数に同じ文字列を指定した場合、何もせずに0を返す.
     *  バッファサイズを超える大きさをコピーしようとした場合、エラー.
     *  プロダクトではコピーできるだけして、コピーできた長さを返す.
     *  全部コピーできなかった場合でも正しく動くように、必要ならアプリ側で返り値をチェックする必要がある.
     *  @param[in]      rhs             コピー元.
     *  @param[in]      size    コピーするサイズ. 負値ならrhs全部.
     *  @return                         実際にコピーしたサイズ.
     */
    inline s32 copy( const SafeStringBase<CharType>& rhs, s32 size = -1 );

    /**
     *  この文字列の指定した場所に文字列をコピーする.
     *
     *  buffer_sizeを超える場合はアサート. アサート無効の場合は追加できるだけ追加する.
     *  atが負値の場合、文字列の末尾から数えた場所にコピーする(-1のとき末尾に付け足す処理と同じ意味になる).
     *  atの絶対値がバッファサイズより大きい場合アサート. プロダクトでは先頭への書き込みとなる.
     *  @param[in]      at                              コピー先の添え字.
     *  @param[in]      src                             コピー元文字列.
     *  @param[in]      cpy_length              コピーする長さ(-1ならsrcすべて).
     *  @return                                         実際にコピーした長さ.
     */
    inline s32 copyAt( s32 at, const SafeStringBase<CharType>& src, s32 cpy_length = -1 );
    /// @}

    /// @name format系.
    /// @{
    /**
     *  書式に従って文字列を構築する.
     *
     *  もともとこのBufferedSafeStringに入っていた文字列は消える.
     *  バッファの大きさを超える分はカットされる(snprintf( buffersize - 1, ...) と同じ).
     *  @param  format_string   書式文字列(Templateを使うとうまく可変長引数がとれないので普通のCharTypeに).
     *  @return BufferedSafeStringBase 自身への参照を返します。
     */
    BufferedSafeStringBase& format( const CharType* format_string, ... );

    /**
     *  書式に従って文字列を構築する(可変長引数リスト指定版).
     *
     *  もともとこのBufferedSafeStringに入っていた文字列は消える.
     *  バッファの大きさを超える分はカットされる(snprintf( buffersize - 1, ...) と同じ).
     *  @param[in]  format_string   書式文字列(Templateを使うとうまく可変長引数がとれないので普通のCharTypeに).
     *  @param[in]  varg            可変長引数リスト
     *  @return                                 書き込み後の文字列の長さ.
     */
    s32 formatV( const CharType* format_string, va_list varg );
    /**
     *  この文字列の末尾に、書式に従って構築した文字列を追記する.
     *
     *  バッファの大きさを超える分はカットされる.
     *  @param  format_string   書式文字列.
     *  @return                                 書き込み後の文字列の長さ.
     */
    s32 appendWithFormat( const CharType* format_string, ... );

    /**
     *  この文字列の末尾に、書式に従って構築した文字列を追記する(可変長引数リスト指定版).
     *
     *  バッファの大きさを超える分はカットされる.
     *  @param[in]  format_string   書式文字列.
     *  @param[in]  varg 可変長引数リスト
     *  @return                                 書き込み後の文字列の長さ.
     */
    s32 appendWithFormatV( const CharType* format_string, va_list varg );
    /// @}

    /// @name append系.
    /// @{
    /**
     *  この文字列の末尾に文字列を追加する.
     *
     *  buffer_sizeを超える場合はエラー終了する.
     *  プロダクト版では、追加できるだけ追加し、実際に追加できた長さを返す.
     *  アプリケーションは全部追加できなかった場合でも適切に動作するように、必要なら返り値をチェックする必要がある.
     *  @param[in]      src                             追加する文字列.
     *  @param[in]      append_length   追加する長さ(-1ならすべて).
     *  @return                                         実際に追加した長さ.
     */
    inline s32 append( const SafeStringBase<CharType>& src, s32 append_length = -1 );

    /**
     *  この文字列の末尾に文字を追加する.
     *
     *  buffer_sizeを超える場合はアサート. アサート無効の場合は何もしない.
     *  アプリケーションは追加できなかった場合でも適切に動作するように、必要なら返り値をチェックする必要がある.
     *  @param[in]      src_chr 追加する文字.
     *  @return                         追加できた文字数(1か0).
     */
    inline s32 append( CharType src_chr );
    /// @}

    /// @name chop系
    /// @{
    /**
     *  末尾から指定した文字数切り落とす.
     *
     *  引数を省略した場合、1文字切り落とす(Perl / Rubyなどのchopと同じ).
     *  chop_numが文字数より多い場合アサート.
     *  プロダクト版では、文字全てを切り落とす.
     *  chop_numが負値の場合もアサート.
     *  プロダクト版では、何もしない.
     *  実際に切り落とした文字数を返す.
     *  @param[in]      chop_num        切り落とす文字数.
     *  @return                                 実際に何文字切り落としたか.
     */
    inline s32 chop( s32 chop_num = 1 );

    /**
     *  末尾の文字が引数の文字に一致する場合、切り落とす.
     *
     *  @param  chop_char       切り落としたい文字.
     *  @return                         実際に切り落とした文字数(1か0).
     */
    inline s32 chopMatchedChar( CharType chop_char );

    /**
     *  末尾の文字が引数の文字リストに含まれている場合、切り落とす.
     *
     *  chopMatchedCharの複数の文字を候補にするバージョン.
     *  @param  chop_char_list  切り落としたい文字リスト.
     *  @return                                 実際に切り落とした文字数(1か0).
     */
    inline s32 chopMatchedChar( const SafeStringBase<CharType>& chop_char_list );

    /**
     *  末尾の文字が改行文字の場合、切り落とす.
     *
     *  Perl / Rubyなどのchompと同じ.
     *  chopMatchedChar( SafeStringBase<CharType>::cLineBreakChar )と同じ動作.
     *  実際に切り落とした文字数を返す(0か1).
     *  @return                                 実際に切り落とした文字数.
     */
    inline s32 chomp()
    {
        return chopMatchedChar( SafeStringBase<CharType>::cLineBreakChar );
    }

    /**
     *  末尾の文字がASCIIコードの規定する印字可能文字でない場合、切り落とす.
     *
     *  chopMatchedChar に全てのASCII印字可能文字以外の文字(ASCIIコードで0x00-0x1F, 0x20(空白), 0x7f(DEL))を渡した場合と同じ動作だが、より高速に動作する.
     *  実際に切り落とした文字数を返す(0か1).
     *  @return 実際に切り落とした文字数.
     */
    inline s32 chopUnprintableAsciiChar();
    /// @}

    /// @name rstrip系.
    /// @{
    /**
     *  指定した文字リストに含まれる文字を、それが続く限り文字列の末尾から全て切り落とす.
     *
     *  末尾についている不要な文字をまとめて削除するときに利用する.
     *  実際に切り落とした文字数を返す.
     *  @param  strip_char_list 切り落としたい文字リスト.
     *  @return                                 実際に切り落とした文字数.
     */
    inline s32 rstrip( const SafeStringBase<CharType>& strip_char_list );

    /**
     *  ASCIIコードの規定する印字可能文字でない文字を、それが続く限り文字列の末尾から全て切り落とす.
     *
     *  末尾についている不要な空白文字や改行文字などをまとめて削除するときに利用する(rubyのrstripと似た動作).
     *  rstrip関数に全てのASCII印字可能文字以外の文字(ASCIIコードで0x00-0x1F, 0x20(空白), 0x7f(DEL))を渡した場合と同じ動作だが、より高速に動作する.
     *  実際に切り落とした文字数を返す.
     *  @return 実際に切り落とした文字数.
     */
    inline s32 rstripUnprintableAsciiChars();
    /// @}

    /// @name trim系.
    /// @{
    /**
     *  この文字列の長さをtrim_length内に収める.
     *
     *  trim_lengthより長い部分を削除する.
     *  もともと文字列長が収まっている(trim_lengthより手前にヌル文字がある)場合でも、trim_lengthの位置にヌル文字を書き加える処理を行う.
     *  trim_lengthに負値を指定するとアサート(プロダクトでは0に切り詰める).
     *  trim_lengthにバッファサイズより長い文字を指定してもアサート(プロダクトでは何もせず文字列長を返す).
     *  @param[in]      trim_length     文字列の長さ.
     *  @return                                 常にtrim_lengthを返す. ただし、プロダクト版でアサート抜けした場合は、0または文字列長を返す.
     */
    inline s32 trim( s32 trim_length );

    /**
     *  文字列の末尾が引数の文字列に一致する場合、切り落とす.
     *
     *  切り落とした後の文字列の長さを返す.
     *  @param  trim_string     切り落としたい文字列.
     *  @return                         切り落とした後の文字数.
     */
    inline s32 trimMatchedString( const SafeStringBase<CharType>& trim_string );
    /// @}

    /// @name その他.
    /// @{
    /**
     *  文字列を空にする.
     *
     *  メモリ全体をクリアするのではなく、先頭にヌル文字を代入しています.
     */
    inline void clear()
    {
        getMutableStringTop_()[ 0 ] = this->cNullChar;
    }

    /**
     *  マルチバイト文字列からこのSafeStringへ変換する.
     *
     *  この変換は文字コードを考慮しない単純なもので、ASCIIの範囲外は切り捨てられる.
     *  元々このバッファに入っていた文字列は消える.
     *  バッファサイズを超えるサイズや変換元文字列より長いサイズを指定した場合エラー.
     *  プロダクト版では、変換できる分だけ変換する.
     *  src_sizeに-1以外の負値を指定するとエラー.
     *  プロダクトでは何もしない.
     *  @param[in]      src                     変換元マルチバイト文字列.
     *  @param[in]      src_size        何文字変換するか(-1のとき、srcすべて).
     *  @return                                 実際に変換した文字数.
     */
    s32 convertFromMultiByteString( const SafeStringBase<char>& src, s32 src_size = -1 )
    {
        return convertFromOtherType_( src, src_size );
    };

    /**
     *  ワイド文字列からこのSafeStringへ変換する.
     *
     *  この変換は文字コードを考慮しない単純なもので、ASCIIの範囲外は切り捨てられる.
     *  元々このバッファに入っていた文字列は消える.
     *  バッファサイズを超えるサイズや変換元文字列より長いサイズを指定した場合エラー.
     *  プロダクト版では、変換できる分だけ変換する.
     *  src_sizeに-1以外の負値を指定するとエラー.
     *  プロダクトでは何もしない.
     *  @param[in]      src                     変換元ワイド文字列.
     *  @param[in]      src_size        何文字変換するか(-1のとき、srcすべて).
     *  @return                                 実際に変換した文字数.
     */
    s32 convertFromWideCharString( const SafeStringBase<char16>& src, s32 src_size = -1 )
    {
        return convertFromOtherType_( src, src_size );
    }

    /**
     *  @brief バッファへのポインタを取得します.
     *
     *  @return バッファへのポインタです。
     *  @details
     *  - 他のライブラリが持っている文字列操作関数へこの文字列を渡すときに使います.
     *  - 他のライブラリの関数内でヌル終端が破壊されたりバッファオーバーランされてしまう恐れがあることに十分気をつけて利用してください.
     *  - sead内の文字列操作関数では、冒頭でバッファ終端にヌル文字を付加するようになっていますので、外部ライブラリでヌル文字を破壊されても最低限の安全性を持ちます.
     *  - 上記のことから、getBufferで取得したポインタをそのまま出力などに渡すより、c_strで再度取得するほうが安全です.
     * @code
     *    FixedSafeString<16> str;
     *    // 外部ライブラリの文字列操作関数.
     *    char* buf = str.getBuffer();
     *    OtherLibrary::someOperation( buf, str.getBufferSize() );
     *    // NW_PRINT( "%s\n", buf ); // <- someOperation でヌル終端が壊されている場合危険！
     *    NW_PRINT( "%s\n", str.c_str() ); // c_str関数を呼んだ時点で、buf[15]にヌル文字が入ることが保障されているので、上のコードより安全.
     * @endcode
     */
    CharType* getBuffer()
    {
        assureTerminationImpl_();
        return getMutableStringTop_();
    }
    /// @}

private:
    /**
     *  mStringTopのconstを外します.
     *
     *  BufferedSafeStringのみで使います.
     */
    CharType* getMutableStringTop_()
    {
        return const_cast<CharType*>( this->mStringTop );
    }

    /**
     *  他の型のSafeStringから強制変換します.
     *
     *  convertFrom系命令内部で呼び出します.
     */
    template<typename Other>
    inline s32 convertFromOtherType_( const SafeStringBase<Other>& src, s32 src_size );

    /**
     *  指定した文字が印字不可能ならtrueを返します.
     */
    inline bool isUnprintableChar_( CharType c ) const;
    /**
     *  format処理の実装部分です.
     */
    static s32 formatImpl_( CharType* dst, s32 dst_size, const CharType* format_string, va_list varg );

protected:
    /**
     *  assureTerminationImpl_の内部実装です.
     *
     *  SafeStringBaseから呼ばれることもあるのでconst_castを使っています.
     */
    virtual void assureTerminationImpl_() const
    {
        BufferedSafeStringBase<CharType>* mutable_ptr = const_cast<BufferedSafeStringBase<CharType>*>( this );
        CharType* mutableStringTop = mutable_ptr->getMutableStringTop_();
        mutableStringTop[ getBufferSize() - 1 ] = mutable_ptr->cNullChar;
    };

private:
    s32 mBufferSize;
};

//-----------------------------------------------------------------------------
template <typename CharType>
inline const CharType&
BufferedSafeStringBase<CharType>::operator[]( s32 idx ) const
{
    if ( idx < 0 || idx > getBufferSize() - 1 )
    {
        NW_ERR( "index(%d) out of range[0, %d]", idx, getBufferSize() - 1 );
        return SafeStringBase<CharType>::cNullChar;
    }
    return SafeStringBase<CharType>::mStringTop[ idx ];
}

template <typename CharType>
inline s32
BufferedSafeStringBase<CharType>::copyAt( s32 at, const SafeStringBase<CharType>& src, s32 cpy_length )
{
    CharType* mutable_string_top = getMutableStringTop_();
    s32 len = this->calcLength();
    if ( at < 0 )
    {
        at = len + at + 1;
        if ( at < 0 )
        {
            NW_ERR( "at(%d) out of range[0, %d]", at, len );
            at = 0;
        }
    }
    if ( cpy_length < 0 )
    {
        cpy_length = src.calcLength();
    }
    if ( at + cpy_length >= getBufferSize() )
    {
        // バッファをはみ出してしまう.
        NW_ERR( "Buffer overflow. (Buffer Size: %d, At: %d, Copy Length: %d)", getBufferSize(), at, cpy_length );
        // PRODUCTではコピーできるだけする.
        cpy_length = getBufferSize() - at - 1;
    }
    // コピーしない( atが大きすぎるときのみ発生 ).
    if ( cpy_length <= 0 )
    {
        return 0;
    }

#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
    std::char_traits<CharType>::copy( mutable_string_top + at, src.c_str(), cpy_length);
#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif
    // 終端文字を壊した可能性がある場合、つけてあげる.
    if ( at + cpy_length > len )
    {
        mutable_string_top[ at + cpy_length ] = SafeStringBase<CharType>::cNullChar;
    }
    return cpy_length;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
BufferedSafeStringBase<CharType>::append( const SafeStringBase<CharType>& src, s32 append_length )
{
    return copyAt( -1, src, append_length );
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
BufferedSafeStringBase<CharType>::append( CharType src_chr )
{
    s32 len = this->calcLength();
    if ( len >= getBufferSize() - 1 )
    {
        NW_ERR( "Buffer overflow. (Buffer Size: %d, Length: %d)", getBufferSize(), len );
        return 0;
    }
    CharType* mutable_string_top = getMutableStringTop_();
    mutable_string_top[ len ] = src_chr;
    mutable_string_top[ len + 1 ] = SafeStringBase<CharType>::cNullChar;
    return 1;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
BufferedSafeStringBase<CharType>::trim( s32 trim_length )
{
    if ( trim_length >= this->getBufferSize() )
    {
        NW_ERR( "trim_length(%d) out of bounds. [0,%d) \n", trim_length, this->getBufferSize() );
        return this->calcLength();
    }
    if ( trim_length < 0 )
    {
        NW_ERR( "trim_length(%d) out of bounds. [0,%d) \n", trim_length, this->getBufferSize() );
        trim_length = 0;
    }
    CharType* mutable_string_top = getMutableStringTop_();
    mutable_string_top[ trim_length ] = SafeStringBase<CharType>::cNullChar;
    return trim_length;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline s32
BufferedSafeStringBase<CharType>::copy( const SafeStringBase<CharType>& rhs, s32 size )
{
    if ( this != &rhs )
    {
        clear();
        return copyAt( 0, rhs, size );
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
template<typename CharType>
template<typename Other>
inline s32
BufferedSafeStringBase<CharType>::convertFromOtherType_( const SafeStringBase<Other>& src, s32 src_size )
{
    // サイズのチェック.
    s32 src_length = src.calcLength();
    if ( src_size == -1 )
    {
        src_size = src_length;
    }
    else if ( src_size < 0 )
    {
        NW_ERR( "src_size(%d) out of bounds [%d,%d]", src_size, 0, src_length );
        return 0;
    }
    else if ( src_size > src_length )
    {
        NW_ERR( "src_size(%d) out of bounds [%d,%d]", src_size, 0, src_length );
        src_size = src_length;
    }
    if ( getBufferSize() <= src_size )
    {
        NW_ERR( "copy_size(%d) out of bounds [%d,%d)", src_size, 0, getBufferSize() );
        src_size = getBufferSize() - 1;
    }
    // 実際のコピー.
    CharType* dst = getMutableStringTop_();
    Other* src_array = const_cast<Other*>( src.c_str() );       // 他のクラス扱いなのでunsafeAtが使えない.
    for( s32 i = 0; i < src_size; i++ )
    {
        dst[ i ] = static_cast<CharType>( src_array[ i ] );
    }
    dst[ src_size ] = BufferedSafeStringBase<CharType>::cNullChar;
    return src_size;
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::chop( s32 chop_num )
{
    s32 len = this->calcLength();
    // 範囲外チェック.
    // 0はエラーではない(意味はない).
    if ( chop_num < 0 || chop_num > len )
    {
        NW_ERR( "chop_num(%d) out of range[0, %d]", chop_num, len );
        if ( chop_num < 0 )
        {
            return 0;
        }
        chop_num = len;
    }
    // 指定した場所にヌル文字を入れることで切り落とす.
    s32 pos = len - chop_num;
    CharType* str = getMutableStringTop_();
    str[ pos ] = SafeStringBase<CharType>::cNullChar;
    return chop_num;
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::chopMatchedChar( CharType chop_char )
{
    s32 len = this->calcLength();
    if ( len == 0 )
    {
        return 0;
    }
    CharType* str = getMutableStringTop_();
    if ( str[ len - 1 ] == chop_char )
    {
        str[ len - 1 ] = SafeStringBase<CharType>::cNullChar;
        return 1;
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::chopMatchedChar( const SafeStringBase<CharType>& chop_char_list )
{
    s32 len = this->calcLength();
    if ( len == 0 )
    {
        return 0;
    }
    CharType* str = getMutableStringTop_();
    if ( chop_char_list.include( str[ len - 1 ] ) )
    {
        str[ len - 1 ] = SafeStringBase<CharType>::cNullChar;
        return 1;
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::chopUnprintableAsciiChar()
{
    s32 len = this->calcLength();
    if ( len == 0 )
    {
        return 0;
    }
    CharType* str = getMutableStringTop_();
    if ( isUnprintableChar_( str[ len - 1 ] ) )
    {
        str[ len - 1 ] = SafeStringBase<CharType>::cNullChar;
        return 1;
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::trimMatchedString( const SafeStringBase<CharType>& trim_string )
{
    s32 len = this->calcLength();
    s32 token_len = trim_string.calcLength();
    if ( len < token_len )
    {
        return len;
    }
    if ( SafeStringBase<CharType>( &this->mStringTop[ len - token_len ] ).comparen( trim_string, token_len ) == 0 )
    {
        CharType* str = getMutableStringTop_();
        str[ len - token_len ] = SafeStringBase<CharType>::cNullChar;
        return len - token_len;
    }
    return len;
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::rstrip( const SafeStringBase<CharType>& strip_char_list )
{
    s32 len = this->calcLength();
    if ( len == 0 )
    {
        return 0;
    }
    CharType* str = getMutableStringTop_();
    s32 count = 0;
    for( s32 i = len - 1; i >= 0; i-- )
    {
        if ( strip_char_list.include( str[ i ] ) )
        {
            str[ i ] = SafeStringBase<CharType>::cNullChar;
            count++;
        }
        else
        {
            break;
        }
    }
    return count;
}

//-----------------------------------------------------------------------------
template<typename CharType>
inline s32
BufferedSafeStringBase<CharType>::rstripUnprintableAsciiChars()
{
    s32 len = this->calcLength();
    if ( len == 0 )
    {
        return 0;
    }
    CharType* str = getMutableStringTop_();
    s32 count = 0;
    for( s32 i = len - 1; i >= 0; i-- )
    {
        if ( isUnprintableChar_( str[ i ] ) )
        {
            str[ i ] = SafeStringBase<CharType>::cNullChar;
            count++;
        }
        else
        {
            break;
        }
    }
    return count;
}

//-----------------------------------------------------------------------------
template <typename CharType>
inline bool
BufferedSafeStringBase<CharType>::isUnprintableChar_( CharType c ) const
{
    if ( c <= 0x20 || c == 0x7f )
    {
        return true;
    }
    else
    {
        return false;
    }
}

//-------------------------------------------------------------------------
//! @brief 自分で固定長のバッファを保持する BufferedSafeString のラッパークラスです。
//!
//! @details
//! バーチャル関数テーブルが生成されないようにするため、継承による実装を避けています。
//-------------------------------------------------------------------------
template <typename CharType>
class FixedSafeStringBase
{
public:
    typedef CharType char_type;
    typedef typename SafeStringBase<CharType>::iterator iterator;
    typedef typename SafeStringBase<CharType>::token_iterator token_iterator;

protected:
    //-------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in] buffer 文字列を格納するバッファです。
    //! @param[in] size   buffer のサイズです。
    //-------------------------------------------------------------------------
    FixedSafeStringBase( CharType* buffer, s32 size )
        : mBuffString( buffer, size )
    {
        this->clear();
    }

private:
    // 禁止
    FixedSafeStringBase( const FixedSafeStringBase<CharType>& rhs );
    FixedSafeStringBase& operator=( const FixedSafeStringBase<CharType>& rhs );

public:
    //-------------------------------------------------------------------------
    //! @brief BufferedSafeString への型変換オペレータです。
    //!
    //! @return BufferedSafeStringBase<CharType> への参照を返します。
    //-------------------------------------------------------------------------
    operator const BufferedSafeStringBase<CharType>& () const
    { return mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief BufferedSafeString への型変換オペレータです。
    //!
    //! @return BufferedSafeStringBase<CharType> への参照を返します。
    //-------------------------------------------------------------------------
    operator BufferedSafeStringBase<CharType>& ()
    { return mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief BufferedSafeString へのポインタを取得します。
    //!
    //! @return BufferedSafeStringBase<CharType> へのポインタを返します。
    //-------------------------------------------------------------------------
    const BufferedSafeStringBase<CharType>* operator&() const
    { return &mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief BufferedSafeString へのポインタを取得します。
    //!
    //! @return BufferedSafeStringBase<CharType> へのポインタを返します。
    //-------------------------------------------------------------------------
    BufferedSafeStringBase<CharType>* operator&()
    { return &mBuffString; }

public:
    //-------------------------------------------------------------------------
    //! @brief 開始イテレータを取得します。
    //!
    //! @return 開始イテレータを返します。
    //-------------------------------------------------------------------------
    iterator begin() const
    { return mBuffString.begin(); }

    //-------------------------------------------------------------------------
    //! @brief 終端イテレータを取得します。
    //!
    //! @return 終端イテレータを返します。
    //!
    //! @details
    //! 関数内で calcLength() を呼びます。
    //! ループ処理の終了条件として使う場合は、ループの外で一度だけ呼ぶようにします。
    //! @code
    //! SafeString::iterator end = mString.end();
    //! for( SafeString::iterator it = mString.begin(); it != end; it++ )
    //! {
    //!     // 中略
    //! }
    //! @endcode
    //-------------------------------------------------------------------------
    iterator end() const
    { return mBuffString.end(); }

    //-------------------------------------------------------------------------
    //! @brief 区切り文字ごとに移動する開始イテレータを取得します。
    //!
    //! @param[in] delimiter    区切り文字のリストを指定します。
    //!
    //! @return 開始イテレータを返します。
    //-------------------------------------------------------------------------
    token_iterator tokenBegin( const SafeStringBase<CharType>& delimiter ) const
    { return mBuffString.tokenBegin( delimiter ); }

    //-------------------------------------------------------------------------
    //! @brief 区切り文字ごとに移動する終了イテレータを取得します。
    //!
    //! @param[in] delimiter    区切り文字のリストを指定します。
    //!
    //! @return 終了イテレータを返します。
    //!
    //! @details
    //!
    //! token_iterator は比較演算子で区切り文字の違いを考慮しないので、
    //! 単に末尾まで進んだかどうかの比較をするだけなら引数は不要です。
    //!
    //! 関数内で calcLength() を呼びます。
    //! ループ処理の終了条件として使う場合は、ループの外で一度だけ呼ぶようにします。
    //! @code
    //! SafeString::token_iterator end = mString.tokenEnd();
    //! for( SafeString::token_iterator it = mString.tokenBegin(); it != end; it++ )
    //! {
    //!     // 中略
    //! }
    //! @endcode
    //-------------------------------------------------------------------------
    token_iterator tokenEnd( const SafeStringBase<CharType>& delimiter = SafeStringBase<CharType>() ) const
    { return mBuffString.tokenEnd( delimiter ); }

    //-------------------------------------------------------------------------
    //! @brief c形式の文字列ポインタを取得します。
    //!
    //! @return 文字列へのポインタを返します。
    //!
    //! @details
    //! ヌル終端を保証するため、配列の末尾にヌル文字が書き込まれます。
    //-------------------------------------------------------------------------
    const CharType* c_str() const
    { return mBuffString.c_str(); }

    //-------------------------------------------------------------------------
    //! @brief 指定位置の文字を取得します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return 指定位置の文字への参照を返します。
    //!
    //! @details
    //! インデックスの範囲が検査されます。
    //-------------------------------------------------------------------------
    const CharType& at( s32 idx ) const
    { return mBuffString.at( idx ); }

    //-------------------------------------------------------------------------
    //! @brief 指定位置の文字を取得します。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return 指定位置の文字への参照を返します。
    //!
    //! @details
    //! インデックスの範囲が検査されます。
    //-------------------------------------------------------------------------
    const CharType& operator[]( s32 idx ) const
    { return mBuffString[ idx ]; }

    //-------------------------------------------------------------------------
    //! @brief 文字列の長さを取得します。
    //!
    //! @return 文字列の長さを返します。
    //!
    //! @details
    //! 文字列の走査を行います。
    //-------------------------------------------------------------------------
    s32 calcLength() const
    { return mBuffString.calcLength(); }

    //-------------------------------------------------------------------------
    //! @brief 指定位置から末尾までの部分文字列を指す SafeStringBase を取得します。
    //!
    //! @param[in] at  部分文字列の先頭位置を指定します。
    //!
    //! @return 部分文字列を指す SafeStringBase への参照を返します。
    //!
    //! @details
    //! この関数は同じ文字列の途中を参照する SafeStringBase を返します。
    //! BufferedSafeStringBase では無いので、バッファをクローンはされません。
    //-------------------------------------------------------------------------
    const SafeStringBase<CharType> getPart( s32 at ) const
    { return mBuffString.getPart( at ); }

    //-------------------------------------------------------------------------
    //! @brief 指定位置から末尾までの部分文字列を指す SafeStringBase を取得します。
    //!
    //! @param[in] it  部分文字列の先頭位置を指定します。
    //!
    //! @return 部分文字列を指す SafeStringBase への参照を返します。
    //!
    //! @details
    //! この関数は同じ文字列の途中を参照する SafeStringBase を返します。
    //! BufferedSafeStringBase では無いので、バッファのクローンはされません。
    //-------------------------------------------------------------------------
    const SafeStringBase<CharType> getPart( const iterator& it ) const
    { return mBuffString.getPart ( it ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列中に指定の文字が含まれているか調べます。
    //!
    //! @param[in] c  文字を指定します。
    //!
    //! @return 文字列中に指定の文字が含まれていた場合には true を返します。
    //-------------------------------------------------------------------------
    bool include( const CharType& c ) const
    { return mBuffString.include( c ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列中に指定の文字列が含まれているか調べます。
    //!
    //! @param[in] str  文字列を指定します。
    //!
    //! @return 文字列中に指定の文字列が含まれていた場合には true を返します。
    //-------------------------------------------------------------------------
    bool include( const SafeStringBase<CharType>& str ) const
    { return mBuffString.include( str ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列が同一かどうか調べます。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が同一であれば true を返します。
    //-------------------------------------------------------------------------
    bool isEqual( const SafeStringBase<CharType>& rhs ) const
    { return mBuffString.isEqual( rhs ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列が同一かどうか調べます。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が同一であれば true を返します。
    //-------------------------------------------------------------------------
    bool operator== ( const SafeStringBase<CharType>& rhs ) const
    { return mBuffString == rhs; }

    //-------------------------------------------------------------------------
    //! @brief 文字列が同一かどうか調べます。
    //!
    //! @param[in] lhs  比較対象の文字列を指定します。
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が同一であれば true を返します。
    //-------------------------------------------------------------------------
    friend bool operator== ( const CharType* lhs, const FixedSafeStringBase<CharType>& rhs )
    { return lhs == rhs.mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief 文字列が異なるかどうか調べます。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が異なれば true を返します。
    //-------------------------------------------------------------------------
    bool operator!= ( const SafeStringBase<CharType>& rhs )
    { return mBuffString != rhs; }

    //-------------------------------------------------------------------------
    //! @brief 文字列が異なるかどうか調べます。
    //!
    //! @param[in] lhs  比較対象の文字列を指定します。
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が異なれば true を返します。
    //-------------------------------------------------------------------------
    friend bool operator!= ( const CharType* lhs, const FixedSafeStringBase<CharType>& rhs )
    { return lhs != rhs.mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief 先頭 n 文字の文字列比較を行います。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //! @param[in] n    比較する文字数を指定します。
    //!
    //! @return 比較結果に応じて -1, 0, 1 のいずれかを返します。
    //-------------------------------------------------------------------------
    s32 comparen( const SafeStringBase<CharType>& rhs, s32 n ) const
    { return mBuffString.comparen( rhs, n ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列の比較を行います。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 比較結果に応じて -1, 0, 1 のいずれかを返します。
    //-------------------------------------------------------------------------
    s32 compare( const SafeStringBase<CharType>& rhs ) const
    { return mBuffString.compare( rhs ); }

    //-------------------------------------------------------------------------
    //! @brief 比較演算子(>)です。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が rhs よりも大きければ true を返します。
    //-------------------------------------------------------------------------
    bool operator>( const SafeStringBase<CharType>& rhs )
    { return mBuffString > rhs; }

    //-------------------------------------------------------------------------
    //! @brief 比較演算子(>)です。
    //!
    //! @param[in] lhs  比較対象の文字列を指定します。
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return lhs が rhs よりも大きければ true を返します。
    //-------------------------------------------------------------------------
    friend bool operator>( const CharType* lhs, const FixedSafeStringBase<CharType>& rhs )
    { return lhs > rhs.mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief 比較演算子(<)です。
    //!
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return 文字列が rhs よりも小さければ true を返します。
    //-------------------------------------------------------------------------
    bool operator<( const SafeStringBase<CharType>& rhs )
    { return mBuffString < rhs; }

    //-------------------------------------------------------------------------
    //! @brief 比較演算子(<)です。
    //!
    //! @param[in] lhs  比較対象の文字列を指定します。
    //! @param[in] rhs  比較対象の文字列を指定します。
    //!
    //! @return lhs が rhs よりも小さければ true を返します。
    //-------------------------------------------------------------------------
    friend bool operator<( const CharType* lhs, const FixedSafeStringBase<CharType>& rhs )
    { return lhs < rhs.mBuffString; }

    //-------------------------------------------------------------------------
    //! @brief 指定した文字列が最初に現れる場所を探し、その場所へのイテレータを返します。
    //!
    //! @param[in] token  探す文字列です。
    //!
    //! @return 指定した文字列の先頭位置を表すイテレータを返します。
    //!         見つからなかった場合には end() に等しいイテレータを返します。
    //-------------------------------------------------------------------------
    iterator findIterator( const SafeStringBase<CharType>& token ) const
    { return mBuffString.findIterator( token ); }

    //-------------------------------------------------------------------------
    //! @brief 指定した文字列が最初に現れる場所を探し、その場所のインデックスを返します。
    //!
    //! @param[in] token  探す文字列です。
    //!
    //! @return 指定した文字列の先頭位置のインデックスを返します。
    //!         見つからなかった場合には -1 を返します。
    //-------------------------------------------------------------------------
    s32 findIndex( const SafeStringBase<CharType>& token ) const
    { return mBuffString.findIndex( token ); }

    //-------------------------------------------------------------------------
    //! @brief 指定した文字列が最後に現れる場所を探し、その場所へのイテレータを返します。
    //!
    //! @param[in] token  探す文字列です。
    //!
    //! @return 指定した文字列の先頭位置のインデックスを返します。
    //!         見つからなかった場合には end() に等しいイテレータを返します。
    //-------------------------------------------------------------------------
    iterator rfindIterator( const SafeStringBase<CharType>& token ) const
    { return mBuffString.rfindIterator( token ); }

    //-------------------------------------------------------------------------
    //! @brief 指定した文字列が最後に現れる場所を探し、その場所のインデックスを返します。
    //!
    //! @param[in] token  探す文字列です。
    //!
    //! @return 指定した文字列の先頭位置のインデックスを返します。
    //!         見つからなかった場合には -1 を返します。
    //-------------------------------------------------------------------------
    s32 rfindIndex( const SafeStringBase<CharType>& token ) const
    { return mBuffString.rfindIndex( token ); }

    //-------------------------------------------------------------------------
    //! @brief 文字列が空文字列かどうか調べます。
    //!
    //! @return 空文字列の場合には true を返します。
    //-------------------------------------------------------------------------
    bool isEmpty() const
    { return mBuffString.isEmpty(); }

    //-------------------------------------------------------------------------
    //! @brief バッファサイズを取得します。
    //!
    //! @return バッファサイズを返します。
    //-------------------------------------------------------------------------
    s32 getBufferSize() const
    { return mBuffString.getBufferSize(); }

    /// @}

    /// @name copy系.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 指定の文字列で上書きします。
    //!
    //! @param[in]      rhs     コピー元です。
    //! @param[in]      size    コピーするサイズです。負値なら rhs 全部をコピーします。
    //!
    //! @return 実際にコピーしたサイズを返します。
    //!         引数に同じ文字列を指定した場合、何もせずに0を返します。
    //!
    //! @details
    //! - バッファサイズを超える大きさをコピーしようとした場合にはエラーとなります。
    //!   Release版では、できるだけコピーをおこない、コピーできた長さを返します。
    //-------------------------------------------------------------------------
    s32 copy( const SafeStringBase<CharType>& rhs, s32 size = -1 )
    { return mBuffString.copy( rhs, size ); }

    //-------------------------------------------------------------------------
    //! @brief 指定した場所に文字列をコピーします。
    //!
    //! @param[in] at           コピー先のインデックスです。
    //!                         負数の場合は末尾からの位置を表します。
    //!                         -1 は末尾に付け足す処理と同じ意味になります。
    //! @param[in] src          コピー元の文字列です。
    //! @param[in] cpy_length   コピーする長さです。負数なら src すべてをコピーします。
    //!
    //! @return 実際にコピーしたサイズを返します。
    //!         引数に同じ文字列を指定した場合、何もせずに0を返します。
    //!
    //! @details
    //! - バッファサイズを超える大きさをコピーしようとした場合にはエラーとなります。
    //!   Release版では、できるだけコピーをおこない、コピーできた長さを返します。
    //!
    //! - at の絶対値がバッファサイズより大きい場合はエラーとなります。
    //!   Release版では先頭への書き込みとなります。
    //-------------------------------------------------------------------------
    s32 copyAt( s32 at, const SafeStringBase<CharType>& src, s32 cpy_length = -1 )
    { return mBuffString.copyAt( at, src, cpy_length ); }

    /// @}

    /// @name format系.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 書式に従って文字列を構築します。
    //!
    //! @param[in] format_string    書式文字列です。
    //!
    //! @return FixedSafeStringBase 自身への参照を返します。
    //!
    //! @details
    //! もともとこの FixedSafeStringBase に入っていた文字列は消えます。
    //! バッファの大きさを超える分はカットされます。
    //!
    //! 関数の引数でフォーマットした文字列を渡すのに便利なように、
    //! format 系のメンバー関数で唯一、自身への参照を返します。
    //-------------------------------------------------------------------------
    FixedSafeStringBase& format( const CharType* format_string, ... );

    //-------------------------------------------------------------------------
    //! @brief 書式に従って文字列を構築します。
    //!
    //! @param[in] format_string    書式文字列です。
    //! @param[in] varg             可変長引数リストです。
    //!
    //! @return 書き込み後の文字列の長さを返します。
    //!
    //! @details
    //! もともとこの FixedSafeStringBase に入っていた文字列は消えます。
    //! バッファの大きさを超える分はカットされます。
    //-------------------------------------------------------------------------
    s32 formatV( const CharType* format_string, va_list varg )
    { return mBuffString.formatV( format_string, varg ); }

    //-------------------------------------------------------------------------
    //! @brief この文字列の末尾に、書式に従って構築した文字列を追加します。
    //!
    //! @param[in] format_string    書式文字列です。
    //!
    //! @return 書き込み後の文字列の長さを返します。
    //!
    //! @details
    //! バッファの大きさを超える分はカットされます。
    //-------------------------------------------------------------------------
    s32 appendWithFormat( const CharType* format_string, ... );

    //-------------------------------------------------------------------------
    //! @brief この文字列の末尾に、書式に従って構築した文字列を追加します。
    //!
    //! @param[in] format_string    書式文字列です。
    //! @param[in] varg             可変長引数リストです。
    //!
    //! @return 書き込み後の文字列の長さを返します。
    //!
    //! @details
    //! バッファの大きさを超える分はカットされます。
    //-------------------------------------------------------------------------
    s32 appendWithFormatV( const CharType* format_string, va_list varg )
    { return mBuffString.appendWithFormatV( format_string, varg ); }

    /// @}

    /// @name append系.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief この文字列の末尾に文字列を追加します。
    //!
    //! @param[in] src              追加する文字列です。
    //! @param[in] append_length    追加する長さです(-1ならすべて)。
    //!
    //! @return 実際に追加した長さを返します。
    //!
    //! @details
    //! - バッファの大きさを超える場合はエラーとなります。
    //!   Release版では、できるだけ追加を行い、実際に追加できた長さを返します。
    //-------------------------------------------------------------------------
    s32 append( const SafeStringBase<CharType>& src, s32 append_length = -1 )
    { return mBuffString.append( src, append_length ); }

    //-------------------------------------------------------------------------
    //! @brief この文字列の末尾に文字を追加します。
    //!
    //! @param[in] src_chr  追加する文字です。
    //!
    //! @return 追加した文字数(1か0)を返します。
    //!
    //! @details
    //! - バッファの大きさを超える場合はエラーとなります。
    //!   Release版では、バッファの大きさを超える場合には追加を行いません。
    //-------------------------------------------------------------------------
    s32 append( CharType src_chr )
    { return mBuffString.append( src_chr ); }

    /// @}

    /// @name chop系
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 末尾から指定数の文字を切り落します。
    //!
    //! @param[in] chop_num     切り落とす文字数です。
    //!
    //! @return 実際に切り落とした文字数を返します。
    //!
    //! @details
    //! 引数を省略した場合、1文字切り落とします。
    //!
    //! - chop_num が文字数より多い場合はアサートします。
    //!   Release版では、文字全てを切り落とします。
    //!
    //! - chop_num が負値の場合もアサートします。
    //!   Release版では、何もしません。
    //-------------------------------------------------------------------------
    s32 chop( s32 chop_num = 1 )
    { return mBuffString.chop( chop_num ); }

    //-------------------------------------------------------------------------
    //! @brief 末尾から指定の文字を切り落します。
    //!
    //! @param[in] chop_char     切り落としたい文字です。
    //!
    //! @return 実際に切り落とした文字数(1か0)を返します。
    //-------------------------------------------------------------------------
    s32 chopMatchedChar( CharType chop_char )
    { return mBuffString.chopMatchedChar( chop_char ); }

    //-------------------------------------------------------------------------
    //! @brief 末尾から指定の文字を切り落します。
    //!
    //! @param[in] chop_char_list   切り落としたい文字のリストです。
    //!
    //! @return 実際に切り落とした文字数(1か0)を返します。
    //!
    //! @details
    //! 末尾の文字が chop_char_list に含まれる場合に切り落とします。
    //-------------------------------------------------------------------------
    s32 chopMatchedChar( const SafeStringBase<CharType>& chop_char_list )
    { return mBuffString.chopMatchedChar( chop_char_list ); }

    //-------------------------------------------------------------------------
    //! @brief 末尾が改行文字の場合に切り落とします。
    //!
    //! @return 実際に切り落とした文字数(1か0)を返します。
    //!
    //! @details
    //! chopMatchedChar( SafeStringBase<CharType>::cLineBreakChar )と同じ動作です。
    //-------------------------------------------------------------------------
    s32 chomp()
    { return mBuffString.chomp(); }

    //-------------------------------------------------------------------------
    //! @brief 末尾がASCIIコードの規定する印字可能文字でない場合に切り落とします。
    //!
    //! @return 実際に切り落とした文字数(1か0)を返します。
    //!
    //! @details
    //! chopMatchedChar に全てのASCII印字可能文字以外の文字
    //! (ASCIIコードで0x00-0x1F, 0x20(空白), 0x7f(DEL))を
    //! 渡した場合と同じ動作ですが、より高速に動作します。
    //-------------------------------------------------------------------------
    s32 chopUnprintableAsciiChar()
    { return mBuffString.chopUnprintableAsciiChar(); }

    /// @}

    /// @name rstrip系.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 末尾から指定の文字をすべて切り落とします。
    //!
    //! @param  strip_char_list     切り落としたい文字のリストです。
    //!
    //! @return 実際に切り落とした文字数を返します。
    //!
    //! @details
    //! 指定した文字リストに含まれる文字を、それが続く限り文字列の末尾から全て切り落とします。
    //-------------------------------------------------------------------------
    s32 rstrip( const SafeStringBase<CharType>& strip_char_list )
    { return mBuffString.rstrip( strip_char_list ); }

    //-------------------------------------------------------------------------
    //! @brief 末尾からASCIIコードの規定する印字可能文字でない文字をすべて切り落とします。
    //!
    //! @return 実際に切り落とした文字数を返します。
    //!
    //! @details
    //! 末尾についている不要な空白文字や改行文字などをまとめて削除するときに利用します。
    //!
    //! rstrip() に全てのASCII印字可能文字以外の文字(ASCIIコードで0x00-0x1F, 0x20(空白), 0x7f(DEL))を
    //! 渡した場合と同じ動作ですが、より高速に動作します。
    //-------------------------------------------------------------------------
    s32 rstripUnprintableAsciiChars()
    { return mBuffString.rstripUnprintableAsciiChars(); }

    /// @}

    /// @name trim系.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 文字列を指定の長さに切り詰めます。
    //!
    //! @param[in] trim_length  文字列の長さです。
    //!
    //! @return 切り詰め後の文字列の長さを返します。
    //!
    //! @details
    //! trim_length より長い部分は削除します。
    //!
    //! もともと文字列長が trim_length 以下に収まっている場合でも、
    //! trim_length の位置にヌル文字を書き加える処理を行います。
    //!
    //! - trim_length に負値を指定するとアサートします。
    //!   Relase 版では0に切り詰めます。
    //!
    //! - trim_length にバッファサイズより長い文字を指定した場合はアサートします。
    //!   Release版では何もせず文字列長を返します。
    //-------------------------------------------------------------------------
    s32 trim( s32 trim_length )
    { return mBuffString.trim( trim_length ); }

    //-------------------------------------------------------------------------
    //! @brief 末尾から指定の文字列を切り落とします。
    //!
    //! @param[in] trim_string  切り落としたい文字列です。
    //!
    //! @return 切り落とした後の文字列の長さを返します。
    //!
    //! @details
    //! 文字列の末尾が trim_string の文字列に一致する場合に切り落とす。
    //-------------------------------------------------------------------------
    s32 trimMatchedString( const SafeStringBase<CharType>& trim_string )
    { return mBuffString.trimMatchedString( trim_string ); }

    /// @}

    /// @name その他.
    /// @{

    //-------------------------------------------------------------------------
    //! @brief 文字列を空にします。
    //!
    //! @details
    //! 先頭にヌル文字を代入します。
    //-------------------------------------------------------------------------
    void clear()
    { mBuffString.clear(); }

    //-------------------------------------------------------------------------
    //! @brief マルチバイト文字列から変換します。
    //!
    //! @param[in] src          変換元のマルチバイト文字列です。
    //! @param[in] src_size     何文字変換するかを指定します(-1のときは src すべて)。
    //!
    //! @return 実際に変換した文字数を返します。
    //!
    //! @details
    //! この変換は文字コードを考慮しない単純なもので、ASCIIの範囲外は切り捨てられます。
    //! 元々このバッファに入っていた文字列は消えます。
    //!
    //! - バッファサイズを超えるサイズや変換元文字列より長いサイズを指定した場合にはエラーとなります。
    //!   Release版では、変換できる分だけ変換します。
    //!
    //! - src_size に-1以外の負値を指定するとエラーとなります。
    //!   Release版では何もしません。
    //-------------------------------------------------------------------------
    s32 convertFromMultiByteString( const SafeStringBase<char>& src, s32 src_size = -1 )
    { return mBuffString.convertFromMultiByteString( src, src_size ); }

    //-------------------------------------------------------------------------
    //! @brief ワイド文字列から変換します。
    //!
    //! @param[in] src          変換元のワイド文字列です。
    //! @param[in] src_size     何文字変換するかを指定します(-1のときは src すべて)。
    //!
    //! @return 実際に変換した文字数を返します。
    //!
    //! @details
    //! この変換は文字コードを考慮しない単純なもので、ASCIIの範囲外は切り捨てられます。
    //! 元々このバッファに入っていた文字列は消えます。
    //!
    //! - バッファサイズを超えるサイズや変換元文字列より長いサイズを指定した場合にはエラーとなります。
    //!   Release版では、変換できる分だけ変換します。
    //!
    //! - src_size に-1以外の負値を指定するとエラーとなります。
    //!   Release版では何もしません。
    //-------------------------------------------------------------------------
    s32 convertFromWideCharString( const SafeStringBase<char16>& src, s32 src_size = -1 )
    { return mBuffString.convertFromWideCharString( src, src_size ); }

    //-------------------------------------------------------------------------
    //! @brief バッファへのポインタを取得します.。
    //!
    //! @return バッファへのポインタです。
    //!
    //! @details
    //! 他のライブラリの文字列操作関数へこの文字列を渡すときに使います。
    //!
    //! - 他のライブラリの関数内でヌル終端が破壊されたりバッファオーバーランされてしまう
    //!   恐れがあることに十分気をつけて利用してください。
    //! - c_str() はバッファ終端にヌル文字を付加しますので、
    //!   外部ライブラリでヌル文字を破壊されても最低限の安全性を持ちます。
    //!
    //! @code
    //!   FixedSafeString<16> str;
    //!   // 外部ライブラリの文字列操作関数.
    //!   char* buf = str.getBuffer();
    //!   OtherLibrary::someOperation( buf, str.getBufferSize() );
    //!   //// someOperation でヌル終端が壊されている場合危険！
    //!   // NW_PRINT( "%s\n", buf );
    //!   //// c_str関数を呼んだ時点で、buf[15]にヌル文字が入ることが保障されているので、上のコードより安全。
    //!   NW_PRINT( "%s\n", str.c_str() );
    //! @endcode
    //-------------------------------------------------------------------------
    CharType* getBuffer()
    { return mBuffString.getBuffer(); }

    /// @}

protected:
    BufferedSafeStringBase<CharType> mBuffString;
};

//-----------------------------------------------------------------------------

/// 安全な文字列(char).
typedef SafeStringBase<char> SafeString;

/// 安全な文字列(char16).
typedef SafeStringBase<char16> SafeString16;

/// 安全なバッファ指定文字列(char).
typedef BufferedSafeStringBase<char> BufferedSafeString;

/// 安全なバッファ指定文字列(char16).
typedef BufferedSafeStringBase<char16> BufferedSafeString16;

//-----------------------------------------------------------------------------

/**
 *  自分で固定長の配列をもつ文字列(char).
 */
template <s32 N> class FixedSafeString : public FixedSafeStringBase<char>
{
    // コンストラクタとoperator=はもう一回実装しないとダメ.
public:
    enum { array_size = N };

    FixedSafeString()
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {}

    explicit FixedSafeString( const SafeStringBase<char>& rhs )
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {
        this->copy( rhs );
    }

    FixedSafeString( const FixedSafeString& rhs )
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {
        this->copy( rhs );
    }

public:
    FixedSafeString& operator=( const SafeStringBase<char>& rhs )
    {
        this->copy( rhs );
        return *this;
    }

    FixedSafeString& operator=( const FixedSafeString& rhs )
    {
        this->copy( rhs );
        return *this;
    }

protected:
    char_type mBuffer[ array_size ];
};

//-----------------------------------------------------------------------------
/**
 *  自分で固定長の配列をもつ文字列(char16).
 */
template <s32 N> class FixedSafeString16 : public FixedSafeStringBase<char16>
{
    // コンストラクタとoperator=はもう一回実装しないとダメ.
public:
    enum { array_size = N };

    FixedSafeString16()
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {}

    explicit FixedSafeString16( const SafeStringBase<char16>& rhs )
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {
        this->copy( rhs );
    }

    FixedSafeString16( const FixedSafeString16& rhs )
        : FixedSafeStringBase<char_type>( mBuffer, array_size )
    {
        this->copy( rhs );
    }

public:
    FixedSafeString16& operator=( const SafeStringBase<char_type>& rhs )
    {
        this->copy( rhs );
        return *this;
    }

    FixedSafeString16& operator=( const FixedSafeString16& rhs )
    {
        this->copy( rhs );
        return *this;
    }

protected:
    char_type mBuffer[ array_size ];
};

//-----------------------------------------------------------------------------

// これらのメソッドの実装はcppにある
template<>
s32 BufferedSafeStringBase<char>::formatV( const char* format_string, va_list varg );
template<>
s32 BufferedSafeStringBase<char16>::formatV( const char16* format_string, va_list varg );
template<>
BufferedSafeStringBase<char>& BufferedSafeStringBase<char>::format( const char* format_string, ... );
template<>
BufferedSafeStringBase<char16>& BufferedSafeStringBase<char16>::format( const char16* format_string, ... );
template<>
s32 BufferedSafeStringBase<char>::appendWithFormatV( const char* format_string, va_list varg );
template<>
s32 BufferedSafeStringBase<char16>::appendWithFormatV( const char16* format_string, va_list varg );
template<>
s32 BufferedSafeStringBase<char>::appendWithFormat( const char* format_string, ... );
template<>
s32 BufferedSafeStringBase<char16>::appendWithFormat( const char16* format_string, ... );
template<>
FixedSafeStringBase<char>& FixedSafeStringBase<char>::format( const char* format_string, ... );
template<>
FixedSafeStringBase<char16>& FixedSafeStringBase<char16>::format( const char16* format_string, ... );
template<>
s32 FixedSafeStringBase<char>::appendWithFormat( const char* format_string, ... );
template<>
s32 FixedSafeStringBase<char16>::appendWithFormat( const char16* format_string, ... );

//-----------------------------------------------------------------------------
} // namespace ut
} // namespace nw

#endif /* NW_SAFE_STRING_H_ */
