﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_RECT_H_
#define NW_UT_RECT_H_

#include <nw/types.h>
//#include <nw/math/arithmetic.h>

namespace nw {
namespace ut {


//--------------------------------------------------------------------------
//! @brief        矩形を表すクラスです。
//---------------------------------------------------------------------------
struct Rect
{
public:
    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    f32 left;     //!< 矩形の左座標です。
    f32 top;      //!< 矩形の上座標です。
    f32 right;    //!< 矩形の右座標です。
    f32 bottom;   //!< 矩形の下座標です。


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{

    //--------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //---------------------------------------------------------------------------
    Rect()
    : left(0),
      top(0),
      right(0),
      bottom(0)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    l       矩形の左座標です。
    //! @param[in]    t       矩形の上座標です。
    //! @param[in]    r       矩形の右座標です。
    //! @param[in]    b       矩形の下座標です。
    //---------------------------------------------------------------------------
    Rect(f32 l, f32 t, f32 r, f32 b)
    : left(l),
      top(t),
      right(r),
      bottom(b)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    v コピー元の Rect です。
    //---------------------------------------------------------------------------
    Rect(const Rect& v)
    : left(v.left),
      top(v.top),
      right(v.right),
      bottom(v.bottom)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~Rect() {}

    //@}

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //--------------------------------------------------------------------------
    //! @brief        矩形の幅を取得します。
    //!
    //! @return       矩形の幅です。
    //---------------------------------------------------------------------------
    f32     GetWidth() const { return right - left; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の高さを取得します。
    //!
    //! @return       矩形の高さです。
    //---------------------------------------------------------------------------
    f32     GetHeight() const { return bottom - top; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の基点となるX座標を取得します。
    //!
    //! @return       矩形の基点となるX座標です。
    //---------------------------------------------------------------------------
    f32     GetX() const { return left; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の基点となるY座標を取得します。
    //!
    //! @return       矩形の基点となるY座標です。
    //---------------------------------------------------------------------------
    f32     GetY() const { return top; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の幅を設定します。
    //!
    //! @param[in]    width   矩形の幅です。
    //---------------------------------------------------------------------------
    void    SetWidth(f32 width) { right = left + width; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の高さを設定します。
    //!
    //! @param[in]    height  矩形の高さです。
    //---------------------------------------------------------------------------
    void    SetHeight(f32 height) { bottom = top + height; }

    //--------------------------------------------------------------------------
    //! @brief        矩形の左上座標を移動します。
    //!
    //! @param[in]    x       矩形の左上座標の x 座標です。
    //! @param[in]    y       矩形の左上座標の y 座標です。
    //---------------------------------------------------------------------------
    void    MoveTo(f32 x, f32 y)
    {
        right  = x + GetWidth();
        left   = x;
        bottom = y + GetHeight();
        top    = y;
    }

    //--------------------------------------------------------------------------
    //! @brief        矩形を並行移動します。
    //!
    //! @param[in]    dx       x 方向へ矩形を移動する移動量です。
    //! @param[in]    dy       y 方向へ矩形を移動する移動量です。
    //---------------------------------------------------------------------------
    void    Move(f32 dx, f32 dy)
    {
        left    += dx;
        right   += dx;
        top     += dy;
        bottom  += dy;
    }

    //--------------------------------------------------------------------------
    //! @brief        左上と右下を設定します。
    //!
    //! @param[in]    l   左端の座標値です。
    //! @param[in]    r   右端の座標値です。
    //! @param[in]    b   下端の座標値です。
    //! @param[in]    t   上端の座標値です。
    //---------------------------------------------------------------------------
    void    SetEdge(f32 l, f32 r, f32 b, f32 t )
    {
        left    = l;
        right   = r;
        bottom  = b;
        top     = t;
    }

    //--------------------------------------------------------------------------
    //! @brief        原点とサイズを設定します。
    //!
    //! @param[in]    x       矩形の基点となるX座標です。
    //! @param[in]    y       矩形の基点となるY座標です。
    //! @param[in]    width   矩形の幅です。
    //! @param[in]    height  矩形の高さです。
    //---------------------------------------------------------------------------
    void    SetOriginAndSize(f32 x, f32 y, f32 width, f32 height )
    {
        left    = x;
        right   = x + width;
        top     = y;
        bottom  = y + height;
    }

    //--------------------------------------------------------------------------
    //! @brief        中心とサイズを設定します。
    //!
    //! @param[in]    x       中心のX座標です。
    //! @param[in]    y       中心のY座標です。
    //! @param[in]    width   矩形の幅です。
    //! @param[in]    height  矩形の高さです。
    //---------------------------------------------------------------------------
    void    SetCenterAndSize(f32 x, f32 y, f32 width, f32 height )
    {
        f32 hx = width * 0.5f;
        f32 hy = height * 0.5f;

        left    = x - hx;
        right   = x + hx;
        top     = y - hy;
        bottom  = y + hy;
    }

    //@}

    //--------------------------------------------------------------------------
    //! @brief        矩形情報を正常な値に正規化します。
    //---------------------------------------------------------------------------
    void    Normalize()
    {
        const f32 l = left;
        const f32 t = top;
        const f32 r = right;
        const f32 b = bottom;

        left    = (r - l) >= 0 ? l : r;
        right   = (r - l) >= 0 ? r : l;
        top     = (b - t) >= 0 ? t : b;
        bottom  = (b - t) >= 0 ? b : t;
    }
};



} /* namespace ut */
} /* namespace nw */

#endif //  NW_UT_RECT_H_
