﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MOVEPTR_H_
#define NW_UT_MOVEPTR_H_

#include <nw/ut/ut_TypeTraits.h>

namespace nw
{
namespace ut
{

namespace internal
{

//---------------------------------------------------------------------------
//! @brief        MovePtr 用デリーターの基本クラスです。
//!
//! @tparam       TObject 削除するオブジェクトの型です。
//---------------------------------------------------------------------------
template<typename TObject>
class MovePtrDeleterBase
{
public:
    typedef void* (*Deleter)(TObject*);

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    deleter オブジェクトを削除するためのデリータです。
    //!
    explicit MovePtrDeleterBase(Deleter deleter)
     : m_Allocator(0)
    { m_DeleteFunction = deleter; }

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    deleter オブジェクトを削除するためのデリータです。
    //! @param[in]    allocator オブジェクトを削除するためのアロケータです。
    //!
    MovePtrDeleterBase(Deleter deleter, ut::IAllocator* allocator)
     : m_Allocator(allocator)
    { m_DeleteFunction = deleter; }

    //! @brief 削除を実行します。
    //!
    //! @param[in]    object 削除するオブジェクトです。
    //!
    void operator() (TObject* object)
    {
        void* memory = m_DeleteFunction(object);
        if (m_Allocator)
        {
            m_Allocator->Free(memory);
        }
    }

    static Deleter m_DeleteFunction;
    ut::IAllocator* m_Allocator;
};

template<class TObject>
typename MovePtrDeleterBase<TObject>::Deleter MovePtrDeleterBase<TObject>::m_DeleteFunction;

//---------------------------------------------------------------------------
//! @brief        MovePtr の単数オブジェクト用デリータークラスです。
//!
//! @tparam       TScalar 削除するオブジェクトの型です。
//---------------------------------------------------------------------------
template<typename TScalar>
class MovePtrScalarDeleter : public MovePtrDeleterBase<TScalar>
{
public:
    typedef MovePtrDeleterBase<TScalar> base;

    //! @brief        コンストラクタです。
    MovePtrScalarDeleter() : base(DoDelete) { }

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator オブジェクトを削除するためのアロケータです。
    //!
    explicit MovePtrScalarDeleter(ut::IAllocator* allocator) : base(DoDelete, allocator) { }

    //! @brief 削除を実行します。
    //!
    //! @param[in]    scalar 削除するオブジェクトです。
    //!
    static void* DoDelete(TScalar* scalar)
    {
        scalar->~TScalar();
        return scalar;
    }
};

//---------------------------------------------------------------------------
//! @brief        MovePtr の複数オブジェクト(配列)用デリータークラスです。
//!
//! この実装方法では、配列にしている要素の型によっては要素数が
//! 格納されていないことがあり、その基準はコンパイラ等に依存するので、
//! 配列用のデリーターは使用禁止とします。
//!
//! @tparam       TArray 削除する配列の型です。
//---------------------------------------------------------------------------
template<typename TArray>
class MovePtrArrayDeleter
    :  public MovePtrDeleterBase<typename remove_bounds<TArray>::type>
{
public:
    typedef typename remove_bounds<TArray>::type element_type;
    typedef MovePtrDeleterBase<element_type> base;

    //! @brief        コンストラクタです。
    MovePtrArrayDeleter() : base(DoDelete) { }

    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator オブジェクトを削除するためのアロケータです。
    //!
    explicit MovePtrArrayDeleter(ut::IAllocator* allocator) : base(DoDelete, allocator) { }

    //! @brief 削除を実行します。
    //!
    //! @param[in]    array 削除するオブジェクトです。
    //!
    static void* DoDelete(element_type* array)
    {
        // 先頭アドレスより前の 4 バイトに要素数が格納されている。
        size_t arraySize = *(reinterpret_cast<int*>(array) - 1);

        element_type* end = array + arraySize;
        for (element_type* i = array; i != end; ++i)
        {
            i->~element_type();
        }

        // 解放されるべきアドレスは先頭アドレスより 4 バイト前になる。
        return reinterpret_cast<char*>(array) - 4;
    }
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        MovePtr や Move に内部的に使用されるクラスです。
//!
//! @tparam       TPointer 移動させるポインタです。
//---------------------------------------------------------------------------
template<typename TPointer>
class MoveSource
{
public:
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    pointer 移動させるポインタです。
    //!
    explicit MoveSource(TPointer& pointer) : m_Pointer(pointer) {}

    //! @brief ポインタを返します。
    //! @return ポインタです。
    TPointer& Ptr() const { return m_Pointer; }

private:
    TPointer& m_Pointer;
    explicit MoveSource(const TPointer&);
};

} // namespace internal

//----------------------------------------
//! @name スマートポインタ操作関連
//@{

//! @brief MovePtr を移動させるための関数です。
//! @param[in] pointer 移動させるポインタです。
//! @return ポインタを MovePtr にしたものを返します。
template<typename TPointer>
internal::MoveSource<TPointer> Move(TPointer& pointer)
{
    return internal::MoveSource<TPointer>(pointer);
}

//@}

//---------------------------------------------------------------------------
//! @brief        MovePtr のデフォルトデリータークラスです。
//!
//! @tparam       TObject 削除するオブジェクトの型です。
//---------------------------------------------------------------------------
template<typename TObject>
class MovePtrDefaultDeleter : public If_< IsArray<TObject>,
                                           internal::MovePtrArrayDeleter<TObject>,
                                           internal::MovePtrScalarDeleter<TObject>
                                      >::type
{
public:
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator アロケータです。
    //!
    explicit MovePtrDefaultDeleter(ut::IAllocator* allocator = 0) { this->m_Allocator = allocator; }

    //! @brief        コンストラクタです。
    //!
    //! @tparam       TTObject 削除するオブジェクトの型です。
    //!
    //! @param[in]    object 削除するオブジェクトです。
    //!
    template<typename TTObject>
    explicit MovePtrDefaultDeleter(MovePtrDefaultDeleter<TTObject> object) { }
};

//---------------------------------------------------------------------------
//! @brief        所有権移動式スマートポインタです。
//!
//! 所有権を移動するように拡張した auto_ptr です。
//! 任意のデリータを指定することが可能です。
//!
//! @tparam       TObject オブジェクトの型です。
//! @tparam       TDeleter デリーターの型です。
//---------------------------------------------------------------------------
template<typename TObject, typename TDeleter = MovePtrDefaultDeleter<TObject> >
class MovePtr
{
    NW_STATIC_ASSERT(!IsArray<TObject>::value);

public:
    typedef typename remove_bounds<TObject>::type element_type; //!< 要素の型です。
    typedef TDeleter deleter_type; //!< オブジェクトを削除するデリータの型です。
    typedef TDeleter& deleter_reference; //!< オブジェクトを削除するデリータの参照です。
    typedef const TDeleter& deleter_const_reference; //!< オブジェクトを削除するデリータの const 参照です。

    //! @briefprivate
    struct SafeBoolHelper { int x; };
    typedef int SafeBoolHelper::* SafeBool; //!< @briefprivate

    //! @brief コンストラクタです。
    MovePtr() : m_Object(0), m_Deleter()
    {
    }

    //! @brief コピーコンストラクタです。コピー元から要素を移動します。
    //!
    //! @param[in]    pointer コピー元の MovePtr クラスです。
    //!
    MovePtr(const MovePtr& pointer)
    : m_Object(const_cast<element_type*>(pointer.Get())), m_Deleter(pointer.GetDeleter())
    {
        const_cast<MovePtr&>(pointer).Release();
    }

    //! @brief コンストラクタです。
    //!
    //! @tparam TTObject 格納するポインタの型です。
    //!
    //! @param[in] pointer 格納するポインタです。
    //!
    template<typename TTObject>
    explicit MovePtr(TTObject* pointer)
    : m_Object(pointer), m_Deleter()
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @tparam TTObject 格納するポインタの型です。
    //!
    //! @param[in] pointer 格納するポインタです。
    //! @param[in] allocator 解放時に使用するアロケータです。
    //!
    template<typename TTObject>
    MovePtr(TTObject* pointer, ut::IAllocator* allocator)
    : m_Object(pointer), m_Deleter(allocator)
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @tparam TTObject 格納するポインタの型です。
    //! @tparam TTDeleter 解放時に使用するデリータの型です。
    //!
    //! @param[in] pointer 格納するポインタです。
    //! @param[in] deleter 解放時に使用するデリーターです。
    //!
    template<typename TTObject, typename TTDeleter>
    MovePtr(TTObject* pointer, TTDeleter deleter)
    : m_Object(pointer), m_Deleter(deleter)
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @tparam TTObject 格納するポインタの型です。
    //! @tparam TTDeleter 解放時に使用するデリータの型です。
    //!
    //! @param[in] source 所有権移動元の MovePtr です。
    //!
    template<typename TTObject, typename TTDeleter>
    explicit MovePtr(internal::MoveSource<MovePtr<TTObject, TTDeleter> > source)
    : m_Object(source.Ptr().Get()), m_Deleter(source.Ptr().GetDeleter())
    {
        source.Ptr().Release();
    }

    //! @brief デストラクタです。
    ~MovePtr()
    {
        if (this->Ptr())
        {
            this->GetDeleter()(this->Ptr());
        }
    }

    //! @brief        MovePtr を代入します。
    //!
    //! @param[in]    rhs 代入する MovePtr です。
    //!               代入元の MovePtr は破棄されます。
    //! @return       代入後の MovePtr を返します。
    MovePtr& operator=(MovePtr rhs)
    {
        rhs.Swap(*this);
        return *this;
    }

    //! @brief        格納しているオブジェクトのポインタを取得します。
    //!
    //! @return       格納しているオブジェクトのポインタです。
    //!
    element_type* Get() const { return Ptr(); }

    //! @brief        格納しているオブジェクトの参照を返します。
    //! @return       オブジェクトの参照です。
    element_type& operator*() const
    {
        NW_STATIC_ASSERT(!IsArray<TObject>::value);
        return *Ptr();
    }

    //! @brief        格納しているオブジェクトを操作します。
    //! @return       オブジェクトの参照です。
    element_type* operator->() const
    {
        NW_STATIC_ASSERT(!IsArray<TObject>::value);
        return Ptr();
    }

    //! @brief        インデックスを指定して格納している配列の要素の参照を取得します。
    //! @param[in]    i インデックスです。
    //! @return       インデックスで指定された配列の要素の参照です。
    element_type& operator[](std::size_t i) const
    {
        NW_STATIC_ASSERT(IsArray<TObject>::value);
        return Ptr()[i];
    }

    //! @brief        格納しているオブジェクトの管理を放棄します。
    //! @return       管理を放棄したオブジェクトです。
    element_type* Release()
    {
        element_type* result = Ptr();
        Ptr() = 0;
        return result;
    }

    //! @brief        格納しているオブジェクトを削除して、状態をリセットします。
    void Reset()
    {
        if (Ptr())
        {
            GetDeleter()(Ptr());
        }
        Ptr() = 0;
    }

    //! @brief        格納しているオブジェクトを削除して、オブジェクトを入れ替えます。
    //!
    //! @tparam       TTObject 入れ替えを行うオブジェクトの型です。
    //!
    //! @param[in]    object 入れ替えを行うオブジェクトです。
    //!
    template<typename TTObject>
    void Reset(TTObject* object)
    {
        MovePtr(object).Swap(*this);
    }

    //! @brief        格納しているインスタンスを削除して、オブジェクトを入れ替えます。
    //!
    //! @tparam       TTObject 入れ替えを行うオブジェクトの型です。
    //! @tparam       TTDeleter　オブジェクトを削除するためのデリータの型です。
    //!
    //! @param[in]    object 入れ替えを行うインスタンスです。
    //! @param[in]    deleter オブジェクトを削除するためのデリータです。
    //!
    template<typename TTObject, typename TTDeleter>
    void Reset(TTObject* object, TTDeleter deleter)
    {
        MovePtr(object, deleter).Swap(*this);
    }

    //! @brief        格納しているインスタンスを削除して、オブジェクトを入れ替えます。
    //!
    //! @tparam       TTObject 入れ替えを行うオブジェクトの型です。
    //!
    //! @param[in]    object 入れ替えを行うインスタンスです。
    //! @param[in]    allocator オブジェクトを解放するためのアロケータです。
    //!
    template<typename TTObject>
    void Reset(TTObject* object, ut::IAllocator* allocator)
    {
        MovePtr(object, TDeleter(allocator)).Swap(*this);
    }

    //! @brief オブジェクトを所有を確認します。
    //!
    //! 望まない式評価を行われないような安全な真偽評価を行います。
    //!
    //! @return オブジェクトを所有する場合に true を返します。
    //!
    operator SafeBool() const { return Ptr() ? &SafeBoolHelper::x : 0; }

    //! @brief        MovePtr を入れ替えます。
    //!
    //! @param[in]    pointer 入れ替える MovePtr です。
    //!
    void Swap(MovePtr& pointer)
    {
        if (&pointer == this)
        {
            return;
        }

        element_type* object = this->m_Object;
        deleter_type deleter = this->m_Deleter;

        this->m_Object = pointer.m_Object;
        this->m_Deleter = pointer.m_Deleter;

        pointer.m_Object = object;
        pointer.m_Deleter = deleter;
    }

    //! @brief        デリータを取得します。
    //!
    //! @return       オブジェクトを削除するためのデリータです。
    //!
    deleter_reference GetDeleter() { return m_Deleter; }

    //! @brief        デリータを取得します。
    //!
    //! @return       オブジェクトを削除するためのデリータです。
    //!
    deleter_const_reference GetDeleter() const { return m_Deleter; }

private:
    template<typename Pointer> struct CantMoveFromConst;
    template<typename TTObject, typename TTDeleter>
    struct CantMoveFromConst<const MovePtr<TTObject, TTDeleter> >
    {
        typedef typename MovePtr<TTObject, TTDeleter>::error type;
    };

    template<typename Pointer>
    MovePtr(Pointer&, typename CantMoveFromConst<Pointer>::type = 0);

    element_type*& Ptr() { return m_Object; }
    element_type* Ptr() const { return m_Object; }

    element_type* m_Object;
    deleter_type m_Deleter;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_MOVEPTR_H_
