﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_LIST_H_
#define NW_UT_LIST_H_

#include <nw/types.h>

namespace nw {
namespace ut {

//----------------------------------------
//! @name リスト操作関連
//@{


//--------------------------------------------------------------------------
//! @brief        双方向リンクリストのノード構造体です。この構造体をリスト構造
//!               で繋ぎたい構造体のメンバとして格納します。
//--------------------------------------------------------------------------
typedef struct
{
    void*       prevObject;     // 前に繋がれているオブジェクトへのポインタ。
    void*       nextObject;     // 次に繋がれているオブジェクトへのポインタ。

} Link;


//--------------------------------------------------------------------------
//! @brief        双方向リンクリスト構造体です。
//--------------------------------------------------------------------------
typedef struct
{
    void*       headObject;     // 先頭に繋がれているオブジェクトへのポインタ。
    void*       tailObject;     // 後尾に繋がれているオブジェクトへのポインタ。
    u16         numObjects;     // リストに繋がれているオブジェクトの個数。
    u16         offset;         // NNSFndLink型の構造体メンバのオフセット。

} List;


//--------------------------------------------------------------------------
//! @brief      リスト構造体を初期化するためのマクロです。実際の初期化には、
//!             List_Init()関数により行われます。
//!
//!             このマクロでは、指定された構造体名とLink型メンバ変数名から、
//!             offsetofマクロを使ってオフセットを求め、List_Init関数に渡して
//!             います。
//!
//! @param[in]  list       リンク構造体へのポインタ。
//! @param[in]  structName リストに繋げたいオブジェクトの構造体名。
//! @param[in]  linkName   このオブジェクトのリンクに使用されるLink型のメンバ変数名。
//--------------------------------------------------------------------------
#define NW_UT_LIST_INIT(list, structName, linkName)                           \
        do {                                                                    \
            NW_COMPILER_ASSERT( offsetof(structName, linkName) < 0x10000 );   \
            ::nw::ut::List_Init(list, u16(offsetof(structName, linkName)));   \
        } while( (void)0, 0 )

#define NW_UT_INIT_LIST   NW_UT_LIST_INIT

/*---------------------------------------------------------------------------*
    関数プロトタイプ。

 *---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//! @brief        リスト構造体を初期化します。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[in]    offset  リストに繋げたい構造体の中に存在する、Link 型の
//!                       メンバ変数の構造体内でのオフセットを指定します。
//!                       stddef.h で定義されている offsetof マクロを使用すると、
//!                       便利です。
//---------------------------------------------------------------------------
void    List_Init(
                List*       list,
                u16         offset);

//---------------------------------------------------------------------------
//! @brief        オブジェクトをリストの最後に追加します。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[out]   object  リストに繋げたいオブジェクトへのポインタ。
//---------------------------------------------------------------------------
void    List_Append(
                List*       list,
                void*       object);

//---------------------------------------------------------------------------
//! @brief        オブジェクトをリストの先頭に挿入します。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[out]   object  リストに繋げたいオブジェクトへのポインタ。
//---------------------------------------------------------------------------
void    List_Prepend(
                List*       list,
                void*       object);

//---------------------------------------------------------------------------
//! @brief        オブジェクトを指定された位置に挿入します。オブジェクトは、
//!               targetで指定されたオブジェクトの前に挿入されます。挿入先が指
//!               定されていない場合（targetがNULLの場合）、オブジェクトはリス
//!               トの最後に追加されます。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[out]   target  挿入したい位置にあるオブジェクトへのポインタ。
//! @param[out]   object  リストに繋げたいオブジェクトへのポインタ。
//---------------------------------------------------------------------------
void    List_Insert(
                List*       list,
                void*       target,
                void*       object);

//---------------------------------------------------------------------------
//! @brief        オブジェクトをリストから削除します。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[out]   object  リストから削除したいオブジェクトへのポインタ。
//---------------------------------------------------------------------------
void    List_Remove(
                List*       list,
                void*       object);

//---------------------------------------------------------------------------
//! @brief        objectで指定されたオブジェクトの次に繋がれているオブジェクト
//!               を返します。objectにNULLが指定されていた場合には、リストの先
//!               頭に繋がれているオブジェクトを返します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    object  リスト中のオブジェクトへのポインタ。
//!
//! @return       指定されたオブジェクトの次のオブジェクトへのポインタを返します。
//!               もし、次のオブジェクトが無ければ、NULLを返します。
//---------------------------------------------------------------------------
void*   List_GetNext(
                const List* list,
                const void* object);

//---------------------------------------------------------------------------
//! @brief        objectで指定されたオブジェクトの前に繋がれているオブジェクト
//!               を返します。objectにNULLが指定されていた場合には、リストの後
//!               尾に繋がれているオブジェクトを返します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    object  リスト中のオブジェクトへのポインタ。
//!
//! @return       指定されたオブジェクトの前のオブジェクトへのポインタを返します。
//!               もし、前のオブジェクトが無ければ、NULL を返します。
//---------------------------------------------------------------------------
void*   List_GetPrev(
                const List* list,
                const void* object);

//---------------------------------------------------------------------------
//! @brief        リストのＮ番目に繋がれているオブジェクトへのポインタを返します。
//!               先頭から順番にリストをたどる為、リストの後部に繋がれてい
//!               オブジェクト程、時間がかかります。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    index   オブジェクトのインデックス。
//!
//! @return       オブジェクトへのポインタを返します。もし、指定されたインデッ
//!               クスのオブジェクトが無かった場合には、NULLが返ります。
//---------------------------------------------------------------------------
void*   List_GetNth(
                const List* list,
                u16         index);


//---------------------------------------------------------------------------
//  const版
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//! @brief        objectで指定されたオブジェクトの次に繋がれているオブジェクト
//!               を返します (const 版)。objectにNULLが指定されていた場合には、リストの先
//!               頭に繋がれているオブジェクトを返します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    object  リスト中のオブジェクトへのポインタ。
//!
//! @return       指定されたオブジェクトの次のオブジェクトへのポインタを返します。
//!               もし、次のオブジェクトが無ければ、NULLを返します。
//---------------------------------------------------------------------------
inline const void*
List_GetNextConst(
    const List*     list,
    const void*     object
)
{
    return List_GetNext( list, object );
}

//---------------------------------------------------------------------------
//! @brief        objectで指定されたオブジェクトの前に繋がれているオブジェクト
//!               を返します (const 版)。objectにNULLが指定されていた場合には、リストの後
//!               尾に繋がれているオブジェクトを返します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    object  リスト中のオブジェクトへのポインタ。
//!
//! @return       指定されたオブジェクトの前のオブジェクトへのポインタを返します。
//!               もし、前のオブジェクトが無ければ、NULL を返します。
//---------------------------------------------------------------------------
inline const void*
List_GetPrevConst(
    const List*     list,
    const void*     object
)
{
    return List_GetPrev( list, object );
}

//---------------------------------------------------------------------------
//! @brief        リストのＮ番目に繋がれているオブジェクトへのポインタを返します (const 版)。
//!               先頭から順番にリストをたどる為、リストの後部に繋がれてい
//!               オブジェクト程、時間がかかります。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//! @param[in]    index   オブジェクトのインデックス。
//!
//! @return       オブジェクトへのポインタを返します。もし、指定されたインデッ
//!               クスのオブジェクトが無かった場合には、NULLが返ります。
//---------------------------------------------------------------------------
inline const void*
List_GetNthConst(
    const List*     list,
    const u16       index
)
{
    return List_GetNth( list, index );
}

//---------------------------------------------------------------------------
//! @brief        リストの先頭のオブジェクトを取得します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//!
//! @return       リストの先頭のオブジェクトへのポインタを返します。
//!               もしリスト中にオブジェクトが無ければ NULL を返します。
//---------------------------------------------------------------------------
inline void*
List_GetFirst( const List* list )
{
    return List_GetNext( list, NULL );
}

//---------------------------------------------------------------------------
//! @brief        リストの先頭のオブジェクトを取得します (const 版)。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//!
//! @return       リストの先頭のオブジェクトへのポインタを返します。
//!               もしリスト中にオブジェクトが無ければNULLを返します。
//---------------------------------------------------------------------------
inline const void*
List_GetFirstConst( const List* list )
{
    return List_GetNextConst( list, NULL );
}

//---------------------------------------------------------------------------
//! @brief        リストの末尾のオブジェクトを取得します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//!
//! @return       リストの末尾のオブジェクトへのポインタを返します。
//!               もしリスト中にオブジェクトが無ければNULLを返します。
//---------------------------------------------------------------------------
inline void*
List_GetLast( const List* list )
{
    return List_GetPrev( list, NULL );
}

//---------------------------------------------------------------------------
//! @brief        リストの末尾のオブジェクトを取得します (const 版)。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//!
//! @return       リストの末尾のオブジェクトへのポインタを返します。
//!               もしリスト中にオブジェクトが無ければNULLを返します。
//---------------------------------------------------------------------------
inline const void*
List_GetLastConst( const List* list )
{
    return List_GetPrevConst( list, NULL );
}


//---------------------------------------------------------------------------
//! @brief        リストに繋がれているオブジェクトの個数を取得します。
//!
//! @param[in]    list    リスト構造体へのポインタ。
//!
//! @return       リストに繋がれているオブジェクトの個数を返します。
//---------------------------------------------------------------------------
inline u16
List_GetSize( const List* list )
{
    NW_NULL_ASSERT( list  );
    return list->numObjects;
}

//@}


} // namespace ut
} // namespace nw

/* NW_UT_LIST_H_ */
#endif
