﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_COLOR_H_
#define NW_UT_COLOR_H_

//#include <twl/gx/GXStruct.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/math/math_Types.h>

namespace nw {
namespace ut {

struct Color4f;

//--------------------------------------------------------------------------
//!  @brief       整数カラーを表す構造体です。
//---------------------------------------------------------------------------
struct Color4u8
{
public:
    typedef Color4u8  SelfType;   //!< @briefprivate

    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const int ELEMENT_MIN         = 0;           //!< カラー要素の最小値です。
    static const int ELEMENT_MAX         = 255;         //!< カラー要素の最大値です。
    static const int ALPHA_MIN           = ELEMENT_MIN; //!< αの最小値です。
    static const int ALPHA_MAX           = ELEMENT_MAX; //!< αの最大値です。
    static const int ALPHA_OPACITY       = ALPHA_MAX;   //!< 不透明のα値です。
    static const int ALPHA_TRANSPARENT   = ALPHA_MIN;   //!< 透明のα値です。

    static const u32 RED                    = 0xFF0000FF; //!< RED 色の定数です。
    static const u32 GREEN                  = 0x00FF00FF; //!< GREEN 色の定数です。
    static const u32 BLUE                   = 0x0000FFFF; //!< BLUE 色の定数です。
    static const u32 CYAN                   = 0x00FFFFFF; //!< CYAN 色の定数です。
    static const u32 YELLOW                 = 0xFFFF00FF; //!< YELLOW 色の定数です。
    static const u32 MAGENTA                = 0xFF00FFFF; //!< MAGENTA 色の定数です。
    static const u32 WHITE                  = 0xFFFFFFFF; //!< WHITE 色の定数です。
    static const u32 BLACK                  = 0x000000FF; //!< BLACK 色の定数です。
    static const u32 GRAY                   = 0x808080FF; //!< GRAY 色の定数です。

    static const u32 X_INDIAN_RED             = 0xCD5C5CFF; //!< INDIAN_RED 色の定数です。
    static const u32 X_LIGHT_CORAL            = 0xF08080FF; //!< LIGHT_CORAL 色の定数です。
    static const u32 X_SALMON                 = 0xFA8072FF; //!< SALMON 色の定数です。
    static const u32 X_DARK_SALMON            = 0xE9967AFF; //!< DARK_SALMON 色の定数です。
    static const u32 X_LIGHT_SALMON           = 0xFFA07AFF; //!< LIGHT_SALMON 色の定数です。
    static const u32 X_CRIMSON                = 0xDC143CFF; //!< CRIMSON 色の定数です。
    static const u32 X_RED                    = 0xFF0000FF; //!< RED 色の定数です。
    static const u32 X_FIRE_BRICK             = 0xB22222FF; //!< FIRE_BRICK 色の定数です。
    static const u32 X_DARK_RED               = 0x8B0000FF; //!< DARK_RED 色の定数です。
    static const u32 X_PINK                   = 0xFFC0CBFF; //!< PINK 色の定数です。
    static const u32 X_LIGHT_PINK             = 0xFFB6C1FF; //!< LIGHT_PINK 色の定数です。
    static const u32 X_HOT_PINK               = 0xFF69B4FF; //!< HOT_PINK 色の定数です。
    static const u32 X_DEEP_PINK              = 0xFF1493FF; //!< DEEP_PINK 色の定数です。
    static const u32 X_MEDIUM_VIOLET_RED      = 0xC71585FF; //!< MEDIUM_VIOLET_RED 色の定数です。
    static const u32 X_PALE_VIOLET_RED        = 0xDB7093FF; //!< PALE_VIOLET_RED 色の定数です。
    static const u32 X_CORAL                  = 0xFF7F50FF; //!< CORAL 色の定数です。
    static const u32 X_TOMATO                 = 0xFF6347FF; //!< TOMATO 色の定数です。
    static const u32 X_ORANGE_RED             = 0xFF4500FF; //!< ORANGE_RED 色の定数です。
    static const u32 X_DARK_ORANGE            = 0xFF8C00FF; //!< DARK_ORANGE 色の定数です。
    static const u32 X_ORANGE                 = 0xFFA500FF; //!< ORANGE 色の定数です。
    static const u32 X_GOLD                   = 0xFFD700FF; //!< GOLD 色の定数です。
    static const u32 X_YELLOW                 = 0xFFFF00FF; //!< YELLOW 色の定数です。
    static const u32 X_LIGHT_YELLOW           = 0xFFFFE0FF; //!< LIGHT_YELLOW 色の定数です。
    static const u32 X_LEMON_CHIFFON          = 0xFFFACDFF; //!< LEMON_CHIFFON 色の定数です。
    static const u32 X_LIGHT_GOLDENROD_YELLOW = 0xFAFAD2FF; //!< LIGHT_GOLDENROD_YELLOW 色の定数です。
    static const u32 X_PAPAYA_WHIP            = 0xFFEFD5FF; //!< PAPAYA_WHIP 色の定数です。
    static const u32 X_MOCCASIN               = 0xFFE4B5FF; //!< MOCCASIN 色の定数です。
    static const u32 X_PEACH_PUFF             = 0xFFDAB9FF; //!< PEACH_PUFF 色の定数です。
    static const u32 X_PALE_GOLDENROD         = 0xEEE8AAFF; //!< PALE_GOLDENROD 色の定数です。
    static const u32 X_KHAKI                  = 0xF0E68CFF; //!< KHAKI 色の定数です。
    static const u32 X_DARK_KHAKI             = 0xBDB76BFF; //!< DARK_KHAKI 色の定数です。
    static const u32 X_LAVENDER               = 0xE6E6FAFF; //!< LAVENDER 色の定数です。
    static const u32 X_THISTLE                = 0xD8BFD8FF; //!< THISTLE 色の定数です。
    static const u32 X_PLUM                   = 0xDDA0DDFF; //!< PLUM 色の定数です。
    static const u32 X_VIOLET                 = 0xEE82EEFF; //!< VIOLET 色の定数です。
    static const u32 X_ORCHID                 = 0xDA70D6FF; //!< ORCHID 色の定数です。
    static const u32 X_FUCHSIA                = 0xFF00FFFF; //!< FUCHSIA 色の定数です。
    static const u32 X_MAGENTA                = 0xFF00FFFF; //!< MAGENTA 色の定数です。
    static const u32 X_MEDIUM_ORCHID          = 0xBA55D3FF; //!< MEDIUM_ORCHID 色の定数です。
    static const u32 X_MEDIUM_PURPLE          = 0x9370DBFF; //!< MEDIUM_PURPLE 色の定数です。
    static const u32 X_AMETHYST               = 0x9966CCFF; //!< AMETHYST 色の定数です。
    static const u32 X_BLUE_VIOLET            = 0x8A2BE2FF; //!< BLUE_VIOLET 色の定数です。
    static const u32 X_DARK_VIOLET            = 0x9400D3FF; //!< DARK_VIOLET 色の定数です。
    static const u32 X_DARK_ORCHID            = 0x9932CCFF; //!< DARK_ORCHID 色の定数です。
    static const u32 X_DARK_MAGENTA           = 0x8B008BFF; //!< DARK_MAGENTA 色の定数です。
    static const u32 X_PURPLE                 = 0x800080FF; //!< PURPLE 色の定数です。
    static const u32 X_INDIGO                 = 0x4B0082FF; //!< INDIGO 色の定数です。
    static const u32 X_SLAT_EBLUE             = 0x6A5ACDFF; //!< SLATE_BLUE 色の定数です。
    static const u32 X_DARK_SLATE_BLUE        = 0x483D8BFF; //!< DARK_SLATE_BLUE 色の定数です。
    static const u32 X_MEDIUM_SLATE_BLUE      = 0x7B68EEFF; //!< MEDIUM_SLATE_BLUE 色の定数です。
    static const u32 X_GREEN_YELLOW           = 0xADFF2FFF; //!< GREEN_YELLOW 色の定数です。
    static const u32 X_CHARTREUSE             = 0x7FFF00FF; //!< CHARTREUSE 色の定数です。
    static const u32 X_LAWN_GREEN             = 0x7CFC00FF; //!< LAWN_GREEN 色の定数です。
    static const u32 X_LIME                   = 0x00FF00FF; //!< LIME 色の定数です。
    static const u32 X_LIME_GREEN             = 0x32CD32FF; //!< LIME_GREEN 色の定数です。
    static const u32 X_PALE_GREEN             = 0x98FB98FF; //!< PALE_GREEN 色の定数です。
    static const u32 X_LIGHT_GREEN            = 0x90EE90FF; //!< LIGHT_GREEN 色の定数です。
    static const u32 X_MEDIUM_SPRING_GREEN    = 0x00FA9AFF; //!< MEDIUM_SPRING_GREEN 色の定数です。
    static const u32 X_SPRING_GREEN           = 0x00FF7FFF; //!< SPRING_GREEN 色の定数です。
    static const u32 X_MEDIUM_SEA_GREEN       = 0x3CB371FF; //!< MEDIUM_SEA_GREEN 色の定数です。
    static const u32 X_SEA_GREEN              = 0x2E8B57FF; //!< SEA_GREEN 色の定数です。
    static const u32 X_FOREST_GREEN           = 0x228B22FF; //!< FOREST_GREEN 色の定数です。
    static const u32 X_GREEN                  = 0x008000FF; //!< GREEN 色の定数です。
    static const u32 X_DARK_GREEN             = 0x006400FF; //!< DARK_GREEN 色の定数です。
    static const u32 X_YELLOW_GREEN           = 0x9ACD32FF; //!< YELLOW_GREEN 色の定数です。
    static const u32 X_OLIVE_DRAB             = 0x6B8E23FF; //!< OLIVE_DRAB 色の定数です。
    static const u32 X_OLIVE                  = 0x808000FF; //!< OLIVE 色の定数です。
    static const u32 X_DARK_OLIVE_GREEN       = 0x556B2FFF; //!< DARK_OLIVE_GREEN 色の定数です。
    static const u32 X_MEDIUM_AQUAMARINE      = 0x66CDAAFF; //!< MEDIUM_AQUAMARINE 色の定数です。
    static const u32 X_DARK_SEA_GREEN         = 0x8FBC8FFF; //!< DARK_SEA_GREEN 色の定数です。
    static const u32 X_LIGHT_SEA_GREEN        = 0x20B2AAFF; //!< LIGHT_SEA_GREEN 色の定数です。
    static const u32 X_DARK_CYAN              = 0x008B8BFF; //!< DARK_CYAN 色の定数です。
    static const u32 X_TEAL                   = 0x008080FF; //!< TEAL 色の定数です。
    static const u32 X_AQUA                   = 0x00FFFFFF; //!< AQUA 色の定数です。
    static const u32 X_CYAN                   = 0x00FFFFFF; //!< CYAN 色の定数です。
    static const u32 X_LIGHT_CYAN             = 0xE0FFFFFF; //!< LIGHT_CYAN 色の定数です。
    static const u32 X_PALE_TURQUOISE         = 0xAFEEEEFF; //!< PALE_TURQUOISE 色の定数です。
    static const u32 X_AQUAMARINE             = 0x7FFFD4FF; //!< AQUAMARINE 色の定数です。
    static const u32 X_TURQUOISE              = 0x40E0D0FF; //!< TURQUOISE 色の定数です。
    static const u32 X_MEDIUM_TURQUOISE       = 0x48D1CCFF; //!< MEDIUM_TURQUOISE 色の定数です。
    static const u32 X_DARK_TURQUOISE         = 0x00CED1FF; //!< DARK_TURQUOISE 色の定数です。
    static const u32 X_CADET_BLUE             = 0x5F9EA0FF; //!< CADET_BLUE 色の定数です。
    static const u32 X_STEEL_BLUE             = 0x4682B4FF; //!< STEEL_BLUE 色の定数です。
    static const u32 X_LIGHT_STEEL_BLUE       = 0xB0C4DEFF; //!< LIGHT_STEEL_BLUE 色の定数です。
    static const u32 X_POWDER_BLUE            = 0xB0E0E6FF; //!< POWDER_BLUE 色の定数です。
    static const u32 X_LIGHT_BLUE             = 0xADD8E6FF; //!< LIGHT_BLUE 色の定数です。
    static const u32 X_SKY_BLUE               = 0x87CEEBFF; //!< SKY_BLUE 色の定数です。
    static const u32 X_LIGHT_SKY_BLUE         = 0x87CEFAFF; //!< LIGHT_SKY_BLUE 色の定数です。
    static const u32 X_DEEP_SKY_BLUE          = 0x00BFFFFF; //!< DEEP_SKY_BLUE 色の定数です。
    static const u32 X_DODGER_BLUE            = 0x1E90FFFF; //!< DODGER_BLUE 色の定数です。
    static const u32 X_CORNFLOWER_BLUE        = 0x6495EDFF; //!< CORNFLOWER_BLUE 色の定数です。
    static const u32 X_ROYAL_BLUE             = 0x4169E1FF; //!< ROYAL_BLUE 色の定数です。
    static const u32 X_BLUE                   = 0x0000FFFF; //!< BLUE 色の定数です。
    static const u32 X_MEDIUM_BLUE            = 0x0000CDFF; //!< MEDIUM_BLUE 色の定数です。
    static const u32 X_DARK_BLUE              = 0x00008BFF; //!< DARK_BLUE 色の定数です。
    static const u32 X_NAVY                   = 0x000080FF; //!< NAVY 色の定数です。
    static const u32 X_MIDNIGHT_BLUE          = 0x191970FF; //!< MIDNIGHT_BLUE 色の定数です。
    static const u32 X_CORNSILK               = 0xFFF8DCFF; //!< CORNSILK 色の定数です。
    static const u32 X_BLANCHED_ALMOND        = 0xFFEBCDFF; //!< BLANCHED_ALMOND 色の定数です。
    static const u32 X_BISQUE                 = 0xFFE4C4FF; //!< BISQUE 色の定数です。
    static const u32 X_NAVAJO_WHITE           = 0xFFDEADFF; //!< NAVAJO_WHITE 色の定数です。
    static const u32 X_WHEAT                  = 0xF5DEB3FF; //!< WHEAT 色の定数です。
    static const u32 X_BURLY_WOOD             = 0xDEB887FF; //!< BURLY_WOOD 色の定数です。
    static const u32 X_TAN                    = 0xD2B48CFF; //!< TAN 色の定数です。
    static const u32 X_ROSY_BROWN             = 0xBC8F8FFF; //!< ROSY_BROWN 色の定数です。
    static const u32 X_SANDY_BROWN            = 0xF4A460FF; //!< SANDY_BROWN 色の定数です。
    static const u32 X_GOLDENROD              = 0xDAA520FF; //!< GOLDENROD 色の定数です。
    static const u32 X_DARK_GOLDENROD         = 0xB8860BFF; //!< DARK_GOLDENROD 色の定数です。
    static const u32 X_PERU                   = 0xCD853FFF; //!< PERU 色の定数です。
    static const u32 X_CHOCOLATE              = 0xD2691EFF; //!< CHOCOLATE 色の定数です。
    static const u32 X_SADDLE_BROWN           = 0x8B4513FF; //!< SADDLE_BROWN 色の定数です。
    static const u32 X_SIENNA                 = 0xA0522DFF; //!< SIENNA 色の定数です。
    static const u32 X_BROWN                  = 0xA52A2AFF; //!< BROWN 色の定数です。
    static const u32 X_MAROON                 = 0x800000FF; //!< MAROON 色の定数です。
    static const u32 X_WHITE                  = 0xFFFFFFFF; //!< WHITE 色の定数です。
    static const u32 X_SNOW                   = 0xFFFAFAFF; //!< SNOW 色の定数です。
    static const u32 X_HONEYDEW               = 0xF0FFF0FF; //!< HONEYDEW 色の定数です。
    static const u32 X_MINT_CREAM             = 0xF5FFFAFF; //!< MINT_CREAM 色の定数です。
    static const u32 X_AZURE                  = 0xF0FFFFFF; //!< AZURE 色の定数です。
    static const u32 X_ALICE_BLUE             = 0xF0F8FFFF; //!< ALICE_BLUE 色の定数です。
    static const u32 X_GHOST_WHITE            = 0xF8F8FFFF; //!< GHOST_WHITE 色の定数です。
    static const u32 X_WHITE_SMOKE            = 0xF5F5F5FF; //!< WHITE_SMOKE 色の定数です。
    static const u32 X_SEASHELL               = 0xFFF5EEFF; //!< SEASHELL 色の定数です。
    static const u32 X_BEIGE                  = 0xF5F5DCFF; //!< BEIGE 色の定数です。
    static const u32 X_OLD_LACE               = 0xFDF5E6FF; //!< OLD_LACE 色の定数です。
    static const u32 X_FLORAL_WHITE           = 0xFFFAF0FF; //!< FLORAL_WHITE 色の定数です。
    static const u32 X_IVORY                  = 0xFFFFF0FF; //!< IVORY 色の定数です。
    static const u32 X_ANTIQUE_WHITE          = 0xFAEBD7FF; //!< ANTIQUE_WHITE 色の定数です。
    static const u32 X_LINEN                  = 0xFAF0E6FF; //!< LINEN 色の定数です。
    static const u32 X_LAVENDER_BLUSH         = 0xFFF0F5FF; //!< LAVENDER_BLUSH 色の定数です。
    static const u32 X_MISTY_ROSE             = 0xFFE4E1FF; //!< MISTY_ROSE 色の定数です。
    static const u32 X_GAINSBORO              = 0xDCDCDCFF; //!< GAINSBORO 色の定数です。
    static const u32 X_LIGHT_GRAY             = 0xD3D3D3FF; //!< LIGHT_GRAY 色の定数です。
    static const u32 X_SILVER                 = 0xC0C0C0FF; //!< SILVER 色の定数です。
    static const u32 X_DARK_GRAY              = 0xA9A9A9FF; //!< DARK_GRAY 色の定数です。
    static const u32 X_GRAY                   = 0x808080FF; //!< GRAY 色の定数です。
    static const u32 X_DIM_GRAY               = 0x696969FF; //!< DIM_GRAY 色の定数です。
    static const u32 X_LIGHT_SLATE_GRAY       = 0x778899FF; //!< LIGHT_SLATE_GRAY 色の定数です。
    static const u32 X_SLATE_GRAY             = 0x708090FF; //!< SLATE_GRAY 色の定数です。
    static const u32 X_DARK_SLATE_GRAY        = 0x2F4F4FFF; //!< DARK_SLATE_GRAY 色の定数です。
    static const u32 X_BLACK                  = 0x000000FF; //!< BLACK 色の定数です。
    static const u32 X_TRANSPARENT            = 0x00000000; //!< TRANSPARENT 色の定数です。

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //--------------------------------------------------------------------------
    //! @brief  デフォルトコンストラクタです。
    //--------------------------------------------------------------------------
    /* ctor */ Color4u8() { this->SetU32(WHITE); }

    //--------------------------------------------------------------------------
    //! @brief  上位ビットから RGBA の並びの u32 の値を設定するコンストラクタです。
    //!
    //! @param[in] color 設定したい u32 の RGBA 値です。
    //--------------------------------------------------------------------------
    /* ctor */ /*implicit*/ Color4u8(u32 color) { this->SetU32BE(color); }

    //--------------------------------------------------------------------------
    //! @brief  RGBA 各要素を設定するコンストラクタです。
    //!
    //! @param[in] red   設定したい R 要素の値です。
    //! @param[in] green 設定したい G 要素の値です。
    //! @param[in] blue  設定したい B 要素の値です。
    //! @param[in] alpha 設定したい A 要素の値です。
    //--------------------------------------------------------------------------
    /* ctor */ /*implicit*/ Color4u8(s32 red, s32 green, s32 blue, s32 alpha)
     : r(static_cast<u8>(red)),
       g(static_cast<u8>(green)),
       b(static_cast<u8>(blue)),
       a(static_cast<u8>(alpha))
    {
    }

    //--------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //--------------------------------------------------------------------------
    /* dtor */ ~Color4u8() {}

    //@}

    //----------------------------------------
    //! @name 演算子オーバーロード
    //@{

    //--------------------------------------------------------------------------
    //!  @brief        u32 型を代入する代入演算子です。
    //!
    //!  @param[in]    color     上位ビットから RGBA の並びの u32 型インスタンスです。
    //!
    //!  @return       Color4u8 のインスタンスを返します。
    //---------------------------------------------------------------------------
    SelfType&      operator =(u32 color)
    {
        this->SetU32BE(color);
        return *this;
    }

    //--------------------------------------------------------------------------
    //!  @brief        Color4f 型を代入する代入演算子です。
    //!
    //!  @param[out]   color     Color4f 型のカラーインスタンスです。
    //!
    //!  @return       Color4u8 のインスタンスを返します。
    //---------------------------------------------------------------------------
    SelfType&      operator =(Color4f& color);

    //--------------------------------------------------------------------------
    //! @brief        u32 型へのキャスト演算子です。
    //!
    //! @return       上位ビットから RGBA の並びの u32 で表現した値を返します。
    //---------------------------------------------------------------------------
    operator u32() const { return ToU32BE(); }

    //---- 四則演算
    //--------------------------------------------------------------------------
    //! @brief        カラー同士の加算演算子です。
    //!
    //! @param[in]    right   加算をおこなうカラー値です。
    //!
    //! @return       加算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator +(const SelfType& right) const
    {
        SelfType color(
            Min(this->r + right.r, ELEMENT_MAX),
            Min(this->g + right.g, ELEMENT_MAX),
            Min(this->b + right.b, ELEMENT_MAX),
            Min(this->a + right.a, ELEMENT_MAX)
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の減算演算子です。
    //!
    //! @param[in]    right   減算をおこなうカラー値です。
    //!
    //! @return       減算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator -(const SelfType& right) const
    {
        SelfType color(
            Max(this->r - right.r, ELEMENT_MIN),
            Max(this->g - right.g, ELEMENT_MIN),
            Max(this->b - right.b, ELEMENT_MIN),
            Max(this->a - right.a, ELEMENT_MIN)
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の乗算演算子です。
    //!
    //! @param[in]    right   乗算をおこなうカラー値です。
    //!
    //! @return       乗算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator *(const SelfType& right) const
    {
        SelfType color(
            this->r * right.r / ELEMENT_MAX,
            this->g * right.g / ELEMENT_MAX,
            this->b * right.b / ELEMENT_MAX,
            this->a * right.a / ELEMENT_MAX
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の除算演算子です。
    //!
    //! @param[in]    right   除算をおこなうカラー値です。
    //!
    //! @return       除算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator /(const SelfType& right) const
    {
        SelfType color(
            (right.r != 0) ? Min(this->r * ELEMENT_MAX / right.r, ELEMENT_MAX): ELEMENT_MAX,
            (right.g != 0) ? Min(this->g * ELEMENT_MAX / right.g, ELEMENT_MAX): ELEMENT_MAX,
            (right.b != 0) ? Min(this->b * ELEMENT_MAX / right.b, ELEMENT_MAX): ELEMENT_MAX,
            (right.a != 0) ? Min(this->a * ELEMENT_MAX / right.a, ELEMENT_MAX): ELEMENT_MAX
        );
        return color;
    }

    //---- インクリメント、デクリメント
    //--------------------------------------------------------------------------
    //! @brief        前置のインクリメント演算子です。
    //!
    //! @return       各要素をインクリメントした値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator ++()
    {
        if (r < ELEMENT_MAX) { r++; }
        if (g < ELEMENT_MAX) { g++; }
        if (b < ELEMENT_MAX) { b++; }
        if (a < ELEMENT_MAX) { a++; }

        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        後置のインクリメント演算子です。
    //!
    //! @return       インクリメント前の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator ++(int)
    {
        SelfType old = *this;
        if (r < ELEMENT_MAX) { r++; }
        if (g < ELEMENT_MAX) { g++; }
        if (b < ELEMENT_MAX) { b++; }
        if (a < ELEMENT_MAX) { a++; }

        return old;
    }

    //--------------------------------------------------------------------------
    //! @brief        前置のデクリメント演算子です。
    //!
    //! @return       各要素をデクリメントした値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator --()
    {
        if (r > ELEMENT_MIN) { r--; }
        if (g > ELEMENT_MIN) { g--; }
        if (b > ELEMENT_MIN) { b--; }
        if (a > ELEMENT_MIN) { a--; }

        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        後置のデクリメント演算子です。
    //!
    //! @return       デクリメント前の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator --(int)
    {
        SelfType old = *this;
        if (r > ELEMENT_MIN) { r--; }
        if (g > ELEMENT_MIN) { g--; }
        if (b > ELEMENT_MIN) { b--; }
        if (a > ELEMENT_MIN) { a--; }

        return old;
    }

    //---- bitごとの和と積
    //--------------------------------------------------------------------------
    //! @brief        u32 としての ビット和を求めます。
    //!
    //! @param[in]    right ビット和をとるカラーです。
    //!
    //! @return       ビット和をとったカラーを返します。
    //---------------------------------------------------------------------------
    const SelfType operator |(const SelfType& right) const
    {
        return U32RawToColor(this->ToU32Raw() | right.ToU32Raw());
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 としての ビット積を求めます。
    //!
    //! @param[in]    right ビット積をとるカラーです。
    //!
    //! @return       ビット積をとったカラーを返します。
    //---------------------------------------------------------------------------
    const SelfType operator &(const SelfType& right) const
    {
        return U32RawToColor(this->ToU32Raw() & right.ToU32Raw());
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の加算代入演算子です。
    //!
    //! @param[in]    rhs   加算をおこなうカラー値です。
    //!
    //! @return       加算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator +=(const SelfType& rhs)
    {
        *this = *this + rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の減算代入演算子です。
    //!
    //! @param[in]    rhs   減算をおこなうカラー値です。
    //!
    //! @return       減算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator -=(const SelfType& rhs)
    {
        *this = *this - rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の乗算代入演算子です。
    //!
    //! @param[in]    rhs   乗算をおこなうカラー値です。
    //!
    //! @return       乗算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator *=(const SelfType& rhs)
    {
        *this = *this * rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の除算代入演算子です。
    //!
    //! @param[in]    rhs   除算をおこなうカラー値です。
    //!
    //! @return       除算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator /=(const SelfType& rhs)
    {
        *this = *this / rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして加算をおこなう演算子です。
    //!
    //! @param[in]    rhs   加算をおこなう u32 値です。
    //!
    //! @return       加算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator +(u32 rhs) const
    {
        const SelfType right = SelfType(rhs);
        SelfType color(
            Min(this->r + right.r, ELEMENT_MAX),
            Min(this->g + right.g, ELEMENT_MAX),
            Min(this->b + right.b, ELEMENT_MAX),
            Min(this->a + right.a, ELEMENT_MAX)
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして減算をおこなう演算子です。
    //!
    //! @param[in]    rhs   減算をおこなう u32 値です。
    //!
    //! @return       減算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator -(u32 rhs) const
    {
        const SelfType right = SelfType(rhs);
        SelfType color(
            Max(this->r - right.r, ELEMENT_MIN),
            Max(this->g - right.g, ELEMENT_MIN),
            Max(this->b - right.b, ELEMENT_MIN),
            Max(this->a - right.a, ELEMENT_MIN)
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして乗算をおこなう演算子です。
    //!
    //! @param[in]    rhs   乗算をおこなう u32 値です。
    //!
    //! @return       乗算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator *(u32 rhs) const
    {
        const SelfType right = SelfType(rhs);
        SelfType color(
            this->r * right.r / ELEMENT_MAX,
            this->g * right.g / ELEMENT_MAX,
            this->b * right.b / ELEMENT_MAX,
            this->a * right.a / ELEMENT_MAX
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして除算をおこなう演算子です。
    //!
    //! @param[in]    rhs   除算をおこなう u32 値です。
    //!
    //! @return       除算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator /(u32 rhs) const
    {
        const SelfType right = SelfType(rhs);
        SelfType color(
            (right.r != 0) ? Min(this->r * ELEMENT_MAX / right.r, ELEMENT_MAX): ELEMENT_MAX,
            (right.g != 0) ? Min(this->g * ELEMENT_MAX / right.g, ELEMENT_MAX): ELEMENT_MAX,
            (right.b != 0) ? Min(this->b * ELEMENT_MAX / right.b, ELEMENT_MAX): ELEMENT_MAX,
            (right.a != 0) ? Min(this->a * ELEMENT_MAX / right.a, ELEMENT_MAX): ELEMENT_MAX
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして論理和演算をおこなう演算子です。
    //!
    //! @param[in]    right   論理和演算をおこなう u32 値です。
    //!
    //! @return       論理和演算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator |(u32 right) const
    {
        return *this | SelfType( right );
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 をカラーとして論理積演算をおこなう演算子です。
    //!
    //! @param[in]    right   論理積演算をおこなう u32 値です。
    //!
    //! @return       論理積演算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator &(u32 right) const
    {
        return *this & SelfType( right );
    }

    //@}

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //---- アクセサ
    //--------------------------------------------------------------------------
    //! @brief        RGBA の値を設定します。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //! @param[in]    alpha   カラーの A 要素です。(デフォルト値: 255)
    //---------------------------------------------------------------------------
    void        Set(
                    s32 red,
                    s32 green,
                    s32 blue,
                    s32 alpha = ALPHA_OPACITY )
    {
        r = static_cast<u8>(red);
        g = static_cast<u8>(green);
        b = static_cast<u8>(blue);
        a = static_cast<u8>(alpha);
    }

    //--------------------------------------------------------------------------
    //! @brief        RGBA8 のカラー値を設定します。
    //!
    //! @param[in]    color   設定するカラー値です。
    //---------------------------------------------------------------------------
    void        Set(SelfType color)
    {
        operator =(color);
    }

    //--------------------------------------------------------------------------
    //! @brief        完全な透明色かどうかを返します。
    //!
    //! @return       透明色であれば true が返ります。
    //---------------------------------------------------------------------------
    bool        IsTransparent() { return a == ELEMENT_MIN; }

    //---- 明示的な型変換
    //--------------------------------------------------------------------------
    //! @brief        u32 としてカラー値を取得します。
    //!
    //! @return       上位ビットから RGBA の並びの u32 値を取得します。
    //---------------------------------------------------------------------------
    u32         ToU32() const
    {
        return this->ToU32BE();
    }

    //--------------------------------------------------------------------------
    //! @brief        u32 として値を設定します。
    //!
    //! @param[in]    value   上位ビットから RGBA の並びの u32 値です。
    //---------------------------------------------------------------------------
    void        SetU32(u32 value) { this->SetU32BE( value ); }

  #if defined( NW_LITTLE_ENDIAN )
    //--------------------------------------------------------------------------
    //! @brief        ABGR の u32 としてカラー値を取得します。
    //!
    //! @return       上位ビットから ABGR の並びの u32 値です。
    //---------------------------------------------------------------------------
    u32         ToU32LE() const
    {
        return this->ToU32RawRef();
    }

    //--------------------------------------------------------------------------
    //! @brief        RGBA の u32 としてカラー値を取得します。
    //!
    //! @return       上位ビットから RGBA の並びの u32 値です。
    //---------------------------------------------------------------------------
    u32         ToU32BE() const
    {
        return ReverseEndian( *reinterpret_cast<const u32*>(this) );
    }

    //--------------------------------------------------------------------------
    //! @brief        ABGR の u32 として値を設定します。
    //!
    //! @param[in]    value   u32 としてのカラー値です。
    //---------------------------------------------------------------------------
    void        SetU32LE(u32 value) { this->ToU32RawRef() = value; }

    //--------------------------------------------------------------------------
    //! @brief        RGBA の u32 として値を設定します。
    //!
    //! @param[in]    value   u32 としてのカラー値です。
    //---------------------------------------------------------------------------
    void        SetU32BE(u32 value) { this->ToU32RawRef() = ReverseEndian( value ); }

  #else

    //--------------------------------------------------------------------------
    //! @brief        ABGR の u32 としてカラー値を取得します。
    //!
    //! @return       カラーを u32 へ変換した値です。
    //---------------------------------------------------------------------------
    u32         ToU32LE() const
    {
        return ReverseEndian( *reinterpret_cast<const u32*>(this) );
    }

    //--------------------------------------------------------------------------
    //! @brief        RGBA の u32 としてカラー値を取得します。
    //!
    //! @return       カラーを u32 へ変換した値です。
    //---------------------------------------------------------------------------
    u32         ToU32BE() const
    {
        return this->ToU32RawRef();
    }

    //--------------------------------------------------------------------------
    //! @brief        ABGR の u32 として値を設定します。
    //!
    //! @param[in]    value   u32 としてのカラー値です。
    //---------------------------------------------------------------------------
    void        SetU32LE(u32 value) { this->ToU32RawRef() = ReverseEndian( value ); }

    //--------------------------------------------------------------------------
    //! @brief        RGBA の u32 として値を設定します。
    //!
    //! @param[in]    value   u32 としてのカラー値です。
    //---------------------------------------------------------------------------
    void        SetU32BE(u32 value) { this->ToU32RawRef() = value; }

  #endif

    //@}

protected:

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //! @briefprivate
    //! @param[in] value u32 としてのカラー値です。
    //! @return u32 をカラーに変換したものです。
    static SelfType U32RawToColor(u32 value)
    {
        return *reinterpret_cast<SelfType*>(&value);
    }

    //! @briefprivate
    //! @return カラーを u32 に変換した値です。
    u32&        ToU32RawRef()
    {
        return *reinterpret_cast<u32*>(this);
    }

    //! @briefprivate
    //! @return カラーを u32 に変換した値です。
    const u32&  ToU32RawRef() const
    {
        return *reinterpret_cast<const u32*>(this);
    }

    //! @briefprivate
    //! @return カラーを u32 に変換した値です。
    u32         ToU32Raw() const
    {
        return *reinterpret_cast<const u32*>(this);
    }

public:
    u8 r; //!< カラーの R 要素です。 0 ～ 255 の範囲で値を保持します。
    u8 g; //!< カラーの G 要素です。 0 ～ 255 の範囲で値を保持します。
    u8 b; //!< カラーの B 要素です。 0 ～ 255 の範囲で値を保持します。
    u8 a; //!< カラーの A 要素です。 0 ～ 255 の範囲で値を保持します。
};


//! @briefprivate
#define NW_UT_DEFINE_CONST_COLOR4F(name) \
    static const Color4f& name() {       \
        Color4u8 color( nw::ut::Color4u8::name ); \
        static const Color4f s_ConstColor( color ); \
        return s_ConstColor;             \
    }

//--------------------------------------------------------------------------
//!  @brief       浮動小数カラーを表す構造体です。
//---------------------------------------------------------------------------
struct Color4f
{
public:
    typedef Color4f SelfType;    //!< @briefprivate

    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const int ELEMENT_MIN         = 0;   //!< カラー要素の最小値です。
    static const int ELEMENT_MAX         = 1;   //!< カラー要素の最大値です。
    static const int ALPHA_MIN           = ELEMENT_MIN; //!< αの最小値です。
    static const int ALPHA_MAX           = ELEMENT_MAX; //!< αの最小値です。
    static const int ALPHA_OPACITY       = ALPHA_MAX;   //!< 不透明のα値です。
    static const int ALPHA_TRANSPARENT   = ALPHA_MIN;   //!< 透明のα値です。

    //! @return RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( RED )
    //! @return GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( GREEN )
    //! @return BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( BLUE )
    //! @return CYAN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( CYAN )
    //! @return YELLOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( YELLOW )
    //! @return MAGENTA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( MAGENTA )
    //! @return WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( WHITE )
    //! @return BLACK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( BLACK )
    //! @return GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( GRAY )

    //! @return X11カラー定義の INDIAN_RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_INDIAN_RED )
    //! @return X11カラー定義の LIGHT_CORAL 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_CORAL )
    //! @return X11カラー定義の SALMON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SALMON )
    //! @return X11カラー定義の DARK_SALMON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_SALMON )
    //! @return X11カラー定義の LIGHT_SALMON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_SALMON )
    //! @return X11カラー定義の CRIMSON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CRIMSON )
    //! @return X11カラー定義の RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_RED )
    //! @return X11カラー定義の FIRE_BRICK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_FIRE_BRICK )
    //! @return X11カラー定義の DARK_RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_RED )
    //! @return X11カラー定義の PINK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PINK )
    //! @return X11カラー定義の LIGHT_PINK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_PINK )
    //! @return X11カラー定義の HOT_PINK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_HOT_PINK )
    //! @return X11カラー定義の DEEP_PINK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DEEP_PINK )
    //! @return X11カラー定義の MEDIUM_VIOLET_RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_VIOLET_RED )
    //! @return X11カラー定義の PALE_VIOLET_RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PALE_VIOLET_RED )
    //! @return X11カラー定義の CORAL 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CORAL )
    //! @return X11カラー定義の TOMATO 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_TOMATO )
    //! @return X11カラー定義の ORANGE_RED 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ORANGE_RED )
    //! @return X11カラー定義の DARK_ORANGE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_ORANGE )
    //! @return X11カラー定義の ORANGE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ORANGE )
    //! @return X11カラー定義の GOLD 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GOLD )
    //! @return X11カラー定義の YELLOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_YELLOW )
    //! @return X11カラー定義の LIGHT_YELLOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_YELLOW )
    //! @return X11カラー定義の LEMON_CHIFFON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LEMON_CHIFFON )
    //! @return X11カラー定義の LIGHT_GOLDENROD_YELLOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_GOLDENROD_YELLOW )
    //! @return X11カラー定義の PAPAYA_WHIP 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PAPAYA_WHIP )
    //! @return X11カラー定義の MOCCASIN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MOCCASIN )
    //! @return X11カラー定義の PEACH_PUFF 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PEACH_PUFF )
    //! @return X11カラー定義の PALE_GOLDENROD 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PALE_GOLDENROD )
    //! @return X11カラー定義の KHAKI 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_KHAKI )
    //! @return X11カラー定義の DARK_KHAKI 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_KHAKI )
    //! @return X11カラー定義の LAVENDER 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LAVENDER )
    //! @return X11カラー定義の THISTLE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_THISTLE )
    //! @return X11カラー定義の PLUM 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PLUM )
    //! @return X11カラー定義の VIOLET 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_VIOLET )
    //! @return X11カラー定義の ORCHID 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ORCHID )
    //! @return X11カラー定義の FUCHSIA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_FUCHSIA )
    //! @return X11カラー定義の MAGENTA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MAGENTA )
    //! @return X11カラー定義の MEDIUM_ORCHID 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_ORCHID )
    //! @return X11カラー定義の MEDIUM_PURPLE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_PURPLE )
    //! @return X11カラー定義の AMETHYST 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_AMETHYST )
    //! @return X11カラー定義の BLUE_VIOLET 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BLUE_VIOLET )
    //! @return X11カラー定義の DARK_VIOLET 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_VIOLET )
    //! @return X11カラー定義の DARK_ORCHID 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_ORCHID )
    //! @return X11カラー定義の DARK_MAGENTA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_MAGENTA )
    //! @return X11カラー定義の PURPLE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PURPLE )
    //! @return X11カラー定義の INDIGO 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_INDIGO )
    //! @return X11カラー定義の SLATE_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SLAT_EBLUE )
    //! @return X11カラー定義の DARK_SLATE_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_SLATE_BLUE )
    //! @return X11カラー定義の MEDIUM_SLATE_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_SLATE_BLUE )
    //! @return X11カラー定義の GREEN_YELLOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GREEN_YELLOW )
    //! @return X11カラー定義の CHARTREUSE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CHARTREUSE )
    //! @return X11カラー定義の LAWN_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LAWN_GREEN )
    //! @return X11カラー定義の LIME 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIME )
    //! @return X11カラー定義の LIME_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIME_GREEN )
    //! @return X11カラー定義の PALE_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PALE_GREEN )
    //! @return X11カラー定義の LIGHT_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_GREEN )
    //! @return X11カラー定義の MEDIUM_SPRING_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_SPRING_GREEN )
    //! @return X11カラー定義の SPRING_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SPRING_GREEN )
    //! @return X11カラー定義の MEDIUM_SEA_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_SEA_GREEN )
    //! @return X11カラー定義の SEA_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SEA_GREEN )
    //! @return X11カラー定義の FOREST_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_FOREST_GREEN )
    //! @return X11カラー定義の GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GREEN )
    //! @return X11カラー定義の DARK_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_GREEN )
    //! @return X11カラー定義の YELLOW_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_YELLOW_GREEN )
    //! @return X11カラー定義の OLIVE_DRAB 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_OLIVE_DRAB )
    //! @return X11カラー定義の OLIVE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_OLIVE )
    //! @return X11カラー定義の DARK_OLIVE_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_OLIVE_GREEN )
    //! @return X11カラー定義の MEDIUM_AQUAMARINE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_AQUAMARINE )
    //! @return X11カラー定義の DARK_SEA_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_SEA_GREEN )
    //! @return X11カラー定義の LIGHT_SEA_GREEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_SEA_GREEN )
    //! @return X11カラー定義の DARK_CYAN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_CYAN )
    //! @return X11カラー定義の TEAL 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_TEAL )
    //! @return X11カラー定義の AQUA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_AQUA )
    //! @return X11カラー定義の CYAN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CYAN )
    //! @return X11カラー定義の LIGHT_CYAN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_CYAN )
    //! @return X11カラー定義の PALE_TURQUOISE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PALE_TURQUOISE )
    //! @return X11カラー定義の AQUAMARINE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_AQUAMARINE )
    //! @return X11カラー定義の TURQUOISE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_TURQUOISE )
    //! @return X11カラー定義の MEDIUM_TURQUOISE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_TURQUOISE )
    //! @return X11カラー定義の DARK_TURQUOISE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_TURQUOISE )
    //! @return X11カラー定義の CADET_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CADET_BLUE )
    //! @return X11カラー定義の STEEL_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_STEEL_BLUE )
    //! @return X11カラー定義の LIGHT_STEEL_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_STEEL_BLUE )
    //! @return X11カラー定義の POWDER_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_POWDER_BLUE )
    //! @return X11カラー定義の LIGHT_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_BLUE )
    //! @return X11カラー定義の SKY_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SKY_BLUE )
    //! @return X11カラー定義の LIGHT_SKY_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_SKY_BLUE )
    //! @return X11カラー定義の DEEP_SKY_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DEEP_SKY_BLUE )
    //! @return X11カラー定義の DODGER_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DODGER_BLUE )
    //! @return X11カラー定義の CORNFLOWER_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CORNFLOWER_BLUE )
    //! @return X11カラー定義の ROYAL_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ROYAL_BLUE )
    //! @return X11カラー定義の BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BLUE )
    //! @return X11カラー定義の MEDIUM_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MEDIUM_BLUE )
    //! @return X11カラー定義の DARK_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_BLUE )
    //! @return X11カラー定義の NAVY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_NAVY )
    //! @return X11カラー定義の MIDNIGHT_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MIDNIGHT_BLUE )
    //! @return X11カラー定義の CORNSILK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CORNSILK )
    //! @return X11カラー定義の BLANCHED_ALMOND 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BLANCHED_ALMOND )
    //! @return X11カラー定義の BISQUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BISQUE )
    //! @return X11カラー定義の NAVAJO_WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_NAVAJO_WHITE )
    //! @return X11カラー定義の WHEAT 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_WHEAT )
    //! @return X11カラー定義の BURLY_WOOD 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BURLY_WOOD )
    //! @return X11カラー定義の TAN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_TAN )
    //! @return X11カラー定義の ROSY_BROWN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ROSY_BROWN )
    //! @return X11カラー定義の SANDY_BROWN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SANDY_BROWN )
    //! @return X11カラー定義の GOLDENROD 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GOLDENROD )
    //! @return X11カラー定義の DARK_GOLDENROD 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_GOLDENROD )
    //! @return X11カラー定義の PERU 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_PERU )
    //! @return X11カラー定義の CHOCOLATE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_CHOCOLATE )
    //! @return X11カラー定義の SADDLE_BROWN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SADDLE_BROWN )
    //! @return X11カラー定義の SIENNA 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SIENNA )
    //! @return X11カラー定義の BROWN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BROWN )
    //! @return X11カラー定義の MAROON 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MAROON )
    //! @return X11カラー定義の WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_WHITE )
    //! @return X11カラー定義の SNOW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SNOW )
    //! @return X11カラー定義の HONEYDEW 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_HONEYDEW )
    //! @return X11カラー定義の MINT_CREAM 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MINT_CREAM )
    //! @return X11カラー定義の AZURE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_AZURE )
    //! @return X11カラー定義の ALICE_BLUE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ALICE_BLUE )
    //! @return X11カラー定義の GHOST_WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GHOST_WHITE )
    //! @return X11カラー定義の WHITE_SMOKE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_WHITE_SMOKE )
    //! @return X11カラー定義の SEASHELL 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SEASHELL )
    //! @return X11カラー定義の BEIGE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BEIGE )
    //! @return X11カラー定義の OLD_LACE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_OLD_LACE )
    //! @return X11カラー定義の FLORAL_WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_FLORAL_WHITE )
    //! @return X11カラー定義の IVORY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_IVORY )
    //! @return X11カラー定義の ANTIQUE_WHITE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_ANTIQUE_WHITE )
    //! @return X11カラー定義の LINEN 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LINEN )
    //! @return X11カラー定義の LAVENDER_BLUSH 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LAVENDER_BLUSH )
    //! @return X11カラー定義の MISTY_ROSE 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_MISTY_ROSE )
    //! @return X11カラー定義の GAINSBORO 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GAINSBORO )
    //! @return X11カラー定義の LIGHT_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_GRAY )
    //! @return X11カラー定義の SILVER 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SILVER )
    //! @return X11カラー定義の DARK_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_GRAY )
    //! @return X11カラー定義の GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_GRAY )
    //! @return X11カラー定義の DIM_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DIM_GRAY )
    //! @return X11カラー定義の LIGHT_SLATE_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_LIGHT_SLATE_GRAY )
    //! @return X11カラー定義の SLATE_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_SLATE_GRAY )
    //! @return X11カラー定義の DARK_SLATE_GRAY 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_DARK_SLATE_GRAY )
    //! @return X11カラー定義の BLACK 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_BLACK )
    //! @return X11カラー定義の TRANSPARENT 色の定数です。
    NW_UT_DEFINE_CONST_COLOR4F( X_TRANSPARENT )

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //--------------------------------------------------------------------------
    //! @brief  デフォルトコンストラクタです。
    //--------------------------------------------------------------------------
    /* ctor */ Color4f()
    : r(0.0f), g(0.0f), b(0.0f), a(1.0f)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief     VEC4を引数にとるコンストラクタです。
    //!
    //! @param[in] vec Color4f に変換するベクトルです。
    //--------------------------------------------------------------------------
    /* ctor */ explicit Color4f(const nw::math::VEC4& vec)
     : r(vec.x), g(vec.y), b(vec.z), a(vec.w)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        Color4u8 から Color4f に変換するコンストラクタです。
    //!
    //! @param[in]    color   RGBA8 のカラーです。
    //--------------------------------------------------------------------------
    /* ctor */ explicit Color4f(Color4u8 color) { operator =(color); }

    //--------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    color   Color4f です。
    //--------------------------------------------------------------------------
    /* ctor */ Color4f(const SelfType& color)
    : r(color.r), g(color.g), b(color.b), a(color.a)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        指定カラーから A だけ変更するコンストラクタです。
    //!
    //! @param[in]    color   Color4f です。
    //! @param[in]    alpha   カラーの A 要素です。
    //--------------------------------------------------------------------------
    /* ctor */ /* implicit */  Color4f(const SelfType& color, f32 alpha)
    : r(color.r), g(color.g), b(color.b), a(alpha)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        RGBA をそれぞれ指定したコンストクタです。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //! @param[in]    alpha   カラーの A 要素です。
    //--------------------------------------------------------------------------
    /* ctor */ Color4f(f32 red, f32 green, f32 blue, f32 alpha)
    : r(red), g(green), b(blue), a(alpha)
    {
    }

    //--------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //--------------------------------------------------------------------------
    /* dtor */ ~Color4f() {}

    //@}

    //----------------------------------------
    //! @name 演算子オーバーロード
    //@{

    //--------------------------------------------------------------------------
    //! @brief        RGBA8 のカラーを Color4f に設定する代入演算子です。
    //!
    //! @param[in]    color   RGBA8 のカラーです。
    //!
    //! @return       Color4f のインスタンスを返します。
    //--------------------------------------------------------------------------
    SelfType&      operator =(Color4u8 color)
    {
        this->Set(
            static_cast<s32>(color.r),
            static_cast<s32>(color.g),
            static_cast<s32>(color.b),
            static_cast<s32>(color.a) );
        return *this;
    }

    //--------------------------------------------------------------------------
    //!  @brief        VEC4 型を代入する代入演算子です。
    //!
    //!  @param[out]   vec     VEC4 型のインスタンスです。
    //!
    //!  @return       Color4f のインスタンスを返します。
    //---------------------------------------------------------------------------
    SelfType&      operator =(const nw::math::VEC4& vec)
    {
        this->Set( vec.x, vec.y, vec.z, vec.w );
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        f32 配列へのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &r; }

    //--------------------------------------------------------------------------
    //! @brief        f32 配列へのキャスト演算子 const 版です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &r; }

    //--------------------------------------------------------------------------
    //! @brief        RGBA8 カラーへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator Color4u8() const
    {
        return Color4u8( static_cast<u8>( 0.5f + nw::ut::Clamp(r, 0.0f, 1.0f) * 255.f ),
                       static_cast<u8>( 0.5f + nw::ut::Clamp(g, 0.0f, 1.0f) * 255.f ),
                       static_cast<u8>( 0.5f + nw::ut::Clamp(b, 0.0f, 1.0f) * 255.f ),
                       static_cast<u8>( 0.5f + nw::ut::Clamp(a, 0.0f, 1.0f) * 255.f ));
    }

#if 0 // NOTE: f32* 経由でキャストされるので無効します。
    //--------------------------------------------------------------------------
    //! @brief        VEC4へのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator nw::math::VEC4&()
    {
        return this->ToVEC4();
    }

    //--------------------------------------------------------------------------
    //! @brief        VEC4へのキャスト演算子 const 版です。
    //---------------------------------------------------------------------------
    operator const nw::math::VEC4&() const
    {
        return this->ToVEC4();
    }
#endif

    //---- 四則演算
    //--------------------------------------------------------------------------
    //! @brief        カラー同士の加算演算子です。
    //!
    //! @param[in]    right   加算をおこなうカラー値です。
    //!
    //! @return       加算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator +(const SelfType& right) const
    {
        SelfType color(
            this->r + right.r,
            this->g + right.g,
            this->b + right.b,
            this->a + right.a
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の減算演算子です。
    //!
    //! @param[in]    right   減算をおこなうカラー値です。
    //!
    //! @return       減算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator -(const SelfType& right) const
    {
        SelfType color(
            this->r - right.r,
            this->g - right.g,
            this->b - right.b,
            this->a - right.a
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の乗算演算子です。
    //!
    //! @param[in]    right   乗算をおこなうカラー値です。
    //!
    //! @return       乗算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator *(const SelfType& right) const
    {
        SelfType color(
            this->r * right.r,
            this->g * right.g,
            this->b * right.b,
            this->a * right.a
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の除算演算子です。
    //!
    //! @param[in]    right   除算をおこなうカラー値です。
    //!
    //! @return       除算後の値を返します。
    //---------------------------------------------------------------------------
    const SelfType operator /(const SelfType& right) const
    {
        SelfType color(
            (right.r != 0) ? (this->r / right.r) : ELEMENT_MAX,
            (right.g != 0) ? (this->g / right.g) : ELEMENT_MAX,
            (right.b != 0) ? (this->b / right.b) : ELEMENT_MAX,
            (right.a != 0) ? (this->a / right.a) : ELEMENT_MAX
        );
        return color;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の加算代入演算子です。
    //!
    //! @param[in]    rhs   加算をおこなうカラー値です。
    //!
    //! @return       加算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator +=(const SelfType& rhs)
    {
        *this = *this + rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の減算代入演算子です。
    //!
    //! @param[in]    rhs   減算をおこなうカラー値です。
    //!
    //! @return       減算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator -=(const SelfType& rhs)
    {
        *this = *this - rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の乗算代入演算子です。
    //!
    //! @param[in]    rhs   乗算をおこなうカラー値です。
    //!
    //! @return       乗算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator *=(const SelfType& rhs)
    {
        *this = *this * rhs;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー同士の除算代入演算子です。
    //!
    //! @param[in]    rhs   除算をおこなうカラー値です。
    //!
    //! @return       除算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator /=(const SelfType& rhs)
    {
        *this = *this / rhs;
        return *this;
    }

    //---- 四則演算
    //--------------------------------------------------------------------------
    //! @brief        カラーへ実数を加算する代入演算子です。
    //!
    //! @param[in]    right   RGBAそれぞれに加算する値です。
    //!
    //! @return       加算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator +=(f32 right)
    {
        this->r += right;
        this->g += right;
        this->b += right;
        this->a += right;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラーから実数を減算する代入演算子です。
    //!
    //! @param[in]    right   RGBAそれぞれに減算する値です。
    //!
    //! @return       減算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator -=(f32 right)
    {
        this->r -= right;
        this->g -= right;
        this->b -= right;
        this->a -= right;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラーへ実数を乗算する代入演算子です。
    //!
    //! @param[in]    right   RGBAそれぞれに乗算する値です。
    //!
    //! @return       乗算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator *=(f32 right)
    {
        this->r *= right;
        this->g *= right;
        this->b *= right;
        this->a *= right;
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラーへ実数を減算する代入演算子です。
    //!
    //! @param[in]    right   RGBAそれぞれに減算する値です。
    //!
    //! @return       減算後のオブジェクトへの参照を返します。
    //---------------------------------------------------------------------------
    SelfType& operator /=(f32 right)
    {
        if (right != 0.0f)
        {
            this->r /= right;
            this->g /= right;
            this->b /= right;
            this->a /= right;
        }
        else
        {
            this->r = ELEMENT_MAX;
            this->g = ELEMENT_MAX;
            this->b = ELEMENT_MAX;
            this->a = ELEMENT_MAX;
        }
        return *this;
    }

    //--------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs     比較する値です。
    //!
    //! @return       等しい場合は true、そうでない場合は false を返します。
    //---------------------------------------------------------------------------
    bool        operator ==(const SelfType& rhs) const
    {
        return (r == rhs.r && g == rhs.g && b == rhs.b && a == rhs.a);
    }

    //--------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs     比較する値です。
    //!
    //! @return       等しい場合は false、そうでない場合は true を返します。
    //---------------------------------------------------------------------------
    bool        operator !=(const SelfType& rhs) const
    {
        return !(*this == rhs);
    }

    //@}

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //---- アクセサ
    //--------------------------------------------------------------------------
    //! @brief        カラーに値を設定します。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //! @param[in]    alpha   カラーの A 要素です。(デフォルト値: 1.0)
    //---------------------------------------------------------------------------
    void        Set(
                    f32 red,
                    f32 green,
                    f32 blue,
                    f32 alpha = ALPHA_OPACITY )
    {
        r = red;
        g = green;
        b = blue;
        a = alpha;
    }

    //---- アクセサ
    //--------------------------------------------------------------------------
    //! @brief        カラーに値を設定します。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //---------------------------------------------------------------------------
    void        SetColor(
                    f32 red,
                    f32 green,
                    f32 blue )
    {
        r = red;
        g = green;
        b = blue;
    }

    //--------------------------------------------------------------------------
    //! @brief        アルファに値を設定します。
    //!
    //! @param[in]    alpha   カラーの A 要素です。
    //---------------------------------------------------------------------------
    void        SetAlpha( f32 alpha )
    {
        a = alpha;
    }

    //--------------------------------------------------------------------------
    //! @brief        0～255 の整数値でカラーを設定します。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //! @param[in]    alpha   カラーの A 要素です。(デフォルト値: 255)
    //---------------------------------------------------------------------------
    void        Set(
                    s32 red,
                    s32 green,
                    s32 blue,
                    s32 alpha = Color4u8::ALPHA_OPACITY )
    {
        r = static_cast<f32>( red ) / 255.f;
        g = static_cast<f32>( green ) / 255.f;
        b = static_cast<f32>( blue ) / 255.f;
        a = static_cast<f32>( alpha ) / 255.f;
    }

    //--------------------------------------------------------------------------
    //! @brief        0～255 の整数値でカラーを設定します。
    //!
    //! @param[in]    red     カラーの R 要素です。
    //! @param[in]    green   カラーの G 要素です。
    //! @param[in]    blue    カラーの B 要素です。
    //---------------------------------------------------------------------------
    void        SetColor(
                    s32 red,
                    s32 green,
                    s32 blue)
    {

        r = static_cast<f32>( red ) / 255.f;
        g = static_cast<f32>( green ) / 255.f;
        b = static_cast<f32>( blue ) / 255.f;
    }

    //--------------------------------------------------------------------------
    //! @brief        0～255 の整数値でアルファを設定します。
    //!
    //! @param[in]    alpha   カラーの A 要素です。
    //---------------------------------------------------------------------------
    void        SetAlpha(
        s32 alpha)
    {
        a = static_cast<f32>( alpha ) / 255.f;
    }

    //--------------------------------------------------------------------------
    //! @brief        カラー型の値を設定します。
    //!
    //! @param[in]    color 設定する色です。
    //---------------------------------------------------------------------------
    void        Set(const SelfType& color) { operator =(color); }

    //--------------------------------------------------------------------------
    //! @brief        完全な透明色かどうかを返します。
    //!
    //! @return       透明色であれば true が返ります。
    //---------------------------------------------------------------------------
    bool        IsTransparent() { return a <= 0.0f; }

    //--------------------------------------------------------------------------
    //! @brief        各要素を 0.0 ～ 1.0 の範囲にクランプします。
    //---------------------------------------------------------------------------
    void        Clamp() { this->Clamp( 0.0f, 1.0f ); }

    //--------------------------------------------------------------------------
    //! @brief        各要素を指定の範囲にクランプします。
    //! @param[in]    low     下限値です。
    //! @param[in]    high    上限値です。
    //---------------------------------------------------------------------------
    void        Clamp( f32 low, f32 high )
    {
        r = nw::ut::Clamp( r, low, high );
        g = nw::ut::Clamp( g, low, high );
        b = nw::ut::Clamp( b, low, high );
        a = nw::ut::Clamp( a, low, high );
    }

    //--------------------------------------------------------------------------
    //! @brief        浮動小数型の配列へ変換します。
    //!
    //! @return       浮動小数点型の配列の先頭です。
    //---------------------------------------------------------------------------
    f32*        ToArray()       { return reinterpret_cast<f32*>( this ); }

    //--------------------------------------------------------------------------
    //! @brief        浮動小数型の const 配列へ変換します。
    //!
    //! @return       浮動小数点型の const 配列の先頭です。
    //---------------------------------------------------------------------------
    const f32*  ToArray() const { return reinterpret_cast<const f32*>( this ); }

    //@}

protected:
    //! @briefprivate
    //! @details VEC4 型へ変換します。
    //! @return VEC4 型です。
    nw::math::VEC4&        ToVEC4()
    {
        return *reinterpret_cast<nw::math::VEC4*>(this);
    }

    //! @briefprivate
    //! @details const の VEC4 型へ変換します。
    //! @return const の VEC4 型です。
    const nw::math::VEC4&  ToVEC4() const
    {
        return *reinterpret_cast<const nw::math::VEC4*>(this);
    }

public:
    f32 r; //!< カラーの R 要素です。
    f32 g; //!< カラーの G 要素です。
    f32 b; //!< カラーの B 要素です。
    f32 a; //!< カラーの A 要素です。

};


//! @briefprivate
typedef Color4f  FloatColor;
//! @briefprivate
typedef Color4u8 Color8;

} /* namespace ut */
} /* namespace nw */

#endif //  NW_UT_COLOR_H_
