﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_BINARYREADER_H_
#define NW_UT_BINARYREADER_H_

#include <nw/types.h>
#include <nw/ut/res/ut_ResTypes.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        メモリ上のバイト列を扱うStreamクラスです。
//---------------------------------------------------------------------------
class MemoryStream
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    start バイト列の開始アドレスを指定します。
    //! @param[in]    end バイト列の終了アドレスを指定します。
    //---------------------------------------------------------------------------
    /* ctor */ MemoryStream( void* start = NULL, void* end = NULL )
      : m_StartAddress( start ),
        m_EndAddress( end ),
        m_CurrentAddress( start )
    {
    }

    //! @brief 現在のアドレスをセットします
    //!
    //! @param[in] address 現在のアドレスです。
    void SetCurrentAddress( void* address ) { m_CurrentAddress = address; }

    //! @brief 現在のアドレスを返します
    //!
    //! @return 現在のアドレスです。
    void* GetCurrentAddress() const { return m_CurrentAddress; }

    // [TODO] ASSERT で止める？
    //! @brief アドレスを移動します
    //!
    //! @param[in] size 移動するサイズを指定します
    //! @return 移動が成功したら ture, 失敗したら false を返します
    bool MoveAddress( int size )
    {
        void* tmp = (void*)( (int)m_CurrentAddress + size );
        if ( tmp < m_StartAddress ) { return false; }
        else if ( tmp > m_EndAddress ) { return false; }
        else
        {
            m_CurrentAddress = tmp;
            return true;
        }
    }

private:
    void* m_StartAddress;
    void* m_EndAddress;
    void* m_CurrentAddress;
};


#define NW_UT_DEFINE_READ_NORMAL_FUNC(TYPE,NAME)                        \
    TYPE Read##NAME( const TYPE value ) const                           \
    {                                                                   \
        return value;                                                   \
    }                                                                   \
    TYPE Read##NAME()                                                   \
    {                                                                   \
        TYPE* x = static_cast<TYPE*>( m_pStream->GetCurrentAddress() );  \
        m_pStream->MoveAddress( sizeof(TYPE) );                          \
        return *x;                                                      \
    }

#define NW_UT_DEFINE_READ_REVERSE_FUNC(TYPE,NAME)                       \
    TYPE Read##NAME( const TYPE value ) const                           \
    {                                                                   \
        return EndianSwap::BSwap(value);                                    \
    }                                                                   \
    TYPE Read##NAME()                                                   \
    {                                                                   \
        TYPE* x = static_cast<TYPE*>( m_pStream->GetCurrentAddress() );  \
        m_pStream->MoveAddress( sizeof(TYPE) );                          \
        return EndianSwap::BSwap(*x);                                       \
    }

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        バイナリデータをそのまま読むリーダークラスです。
//---------------------------------------------------------------------------
class BinaryNormalReader
{
public:
#if defined( NW_ENABLE_FLOAT64 )
    NW_UT_DEFINE_READ_NORMAL_FUNC(f64,F64)
#endif
    NW_UT_DEFINE_READ_NORMAL_FUNC(s64,S64)
    NW_UT_DEFINE_READ_NORMAL_FUNC(u64,U64)
    NW_UT_DEFINE_READ_NORMAL_FUNC(f32,F32)
    NW_UT_DEFINE_READ_NORMAL_FUNC(s32,S32)
    NW_UT_DEFINE_READ_NORMAL_FUNC(u32,U32)
    NW_UT_DEFINE_READ_NORMAL_FUNC(s16,S16)
    NW_UT_DEFINE_READ_NORMAL_FUNC(u16,U16)
    NW_UT_DEFINE_READ_NORMAL_FUNC(s8,S8)
    NW_UT_DEFINE_READ_NORMAL_FUNC(u8,U8)

    /* ctor */ explicit BinaryNormalReader( MemoryStream* stream = NULL )
      : m_pStream( stream ) {}

    void SetStream( MemoryStream* stream ) { m_pStream = stream; }

private:
    MemoryStream* m_pStream;
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        バイナリデータを反転して読むリーダークラスです。
//---------------------------------------------------------------------------
class BinaryReverseReader
{
public:

#if defined( NW_ENABLE_FLOAT64 )
    NW_UT_DEFINE_READ_REVERSE_FUNC(f64,F64)
#endif
    NW_UT_DEFINE_READ_REVERSE_FUNC(s64,S64)
    NW_UT_DEFINE_READ_REVERSE_FUNC(u64,U64)
    NW_UT_DEFINE_READ_REVERSE_FUNC(f32,F32)
    NW_UT_DEFINE_READ_REVERSE_FUNC(s32,S32)
    NW_UT_DEFINE_READ_REVERSE_FUNC(u32,U32)
    NW_UT_DEFINE_READ_REVERSE_FUNC(s16,S16)
    NW_UT_DEFINE_READ_REVERSE_FUNC(u16,U16)
    NW_UT_DEFINE_READ_NORMAL_FUNC(s8,S8)
    NW_UT_DEFINE_READ_NORMAL_FUNC(u8,U8)

    /* ctor */ explicit BinaryReverseReader( MemoryStream* stream = NULL )
      : m_pStream( stream ) {}

    void SetStream( MemoryStream* stream ) { m_pStream = stream; }

private:
    MemoryStream* m_pStream;
};

}   // namespace nw::ut
}   // namespace nw


#endif  // NW_UT_BINARYREADER_H_

