﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_RESTYPES_H_
#define NW_UT_RESTYPES_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/math/math_Types.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        エンディアンスワップユーティリティ用の名前空間
//---------------------------------------------------------------------------
namespace EndianSwap {
    typedef union
    {
        u64 UInt64;
        s64 SInt64;
#if defined( NW_ENABLE_FLOAT64 )
        f64 Float64;
#endif
    } Type64;

    typedef union
    {
        u32 UInt32;
        s32 SInt32;
        f32 Float32;
    } Type32;

    typedef union
    {
        u16 UInt16;
        s16 SInt16;
    } Type16;

    //---------------------------------------------------------------------------
    //! @brief        u64 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline u64 BSwap( u64 val )
    {
        const u64 MASK  = 0xFF00FF00FF00FF00ULL;
        const u64 MASK2 = 0xFFFF0000FFFF0000ULL;
        val = ( (val & MASK) >> 8 ) | ( (val << 8) & MASK );
        val = ( (val & MASK2) >> 16 ) | ( (val << 16) & MASK2 );
        return (val >> 32) | (val << 32);
    }

    //---------------------------------------------------------------------------
    //! @brief        s64 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline s64 BSwap( s64 val )
    {
        Type64 data;
        data.SInt64 = val;
        data.UInt64 = BSwap( data.UInt64 );
        return data.SInt64;
    }

#if defined( NW_ENABLE_FLOAT64 )
    //---------------------------------------------------------------------------
    //! @brief        f64 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline f64 BSwap( f64 val )
    {
        Type64 data;
        data.Float64 = val;
        data.UInt64 = BSwap( data.UInt64 );
        return data.Float64;
    }
#endif

    //---------------------------------------------------------------------------
    //! @brief        u32 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline u32 BSwap( u32 val )
    {
        const u32 MASK = 0xFF00FF00;
        val = ( (val & MASK) >> 8 ) | ( (val << 8) & MASK );
        return (val >> 16) | (val << 16);
    }

    //---------------------------------------------------------------------------
    //! @brief        s32 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline s32 BSwap( s32 val )
    {
        Type32 data;
        data.SInt32 = val;
        data.UInt32 = BSwap( data.UInt32 );
        return data.SInt32;
    }

    //---------------------------------------------------------------------------
    //! @brief        f32 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline f32 BSwap( f32 val )
    {
        Type32 data;
        data.Float32 = val;
        data.UInt32 = BSwap( data.UInt32 );
        return data.Float32;
    }

    //---------------------------------------------------------------------------
    //! @brief        u16 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline u16 BSwap( u16 val )
    {
        return (u16)( (val >> 8) | (val << 8) );
    }

    //---------------------------------------------------------------------------
    //! @brief        s16 のエンディアンを変換します。
    //!
    //! @param[in]    val     エンディアン変換する値です。
    //!
    //! @return       変換結果を返します。
    //---------------------------------------------------------------------------
    inline s16 BSwap( s16 val )
    {
        return (s16)( ((u16)val >> 8) | ((u16)val << 8) );
    }
} /* namespace EndianSwap */

#if !defined( NW_UT_RES_SWAP_ENDIAN )
    typedef u8  ResU8;          //!< @briefprivate
    typedef s8  ResS8;          //!< @briefprivate
    typedef u16 ResU16;         //!< @briefprivate
    typedef s16 ResS16;         //!< @briefprivate
    typedef u32 ResU32;         //!< @briefprivate
    typedef s32 ResS32;         //!< @briefprivate
    typedef f32 ResF32;         //!< @briefprivate
    typedef u32 ResU64;         //!< @briefprivate
    typedef s32 ResS64;         //!< @briefprivate
    typedef f32 ResF64;         //!< @briefprivate
#else // if defined( NW_UT_RES_SWAP_ENDIAN )
    template <typename T>
    class ResNum
    {
    public:
        /* ctor */ ResNum() {}
        /* ctor */ ResNum(const ResNum& other) : m_Bits(other.m_Bits) {}
        /* ctor */ explicit ResNum(const T val ) : m_Bits( EndianSwap::BSwap( val ) ) {}

        void    operator = (const ResNum& other) { m_Bits = other.m_Bits; }
        /* T */ operator T () const { return EndianSwap::BSwap( m_Bits ); }
        void    operator = (T val) { m_Bits = EndianSwap::BSwap( val ); }

        ResNum  operator+=(T num) { m_Bits = EndianSwap::BSwap( static_cast<T>(*this) + num ); return ResNum(*this); }
        ResNum  operator-=(T num) { m_Bits = EndianSwap::BSwap( static_cast<T>(*this) - num ); return ResNum(*this); }

    private:
        T   m_Bits;
    };

    typedef u8          ResU8;      //!< @briefprivate
    typedef s8          ResS8;      //!< @briefprivate
    typedef ResNum<u16> ResU16;     //!< @briefprivate
    typedef ResNum<s16> ResS16;     //!< @briefprivate
    typedef ResNum<u32> ResU32;     //!< @briefprivate
    typedef ResNum<s32> ResS32;     //!< @briefprivate
    typedef ResNum<f32> ResF32;     //!< @briefprivate
    typedef ResNum<u64> ResU64;     //!< @briefprivate
    typedef ResNum<s64> ResS64;     //!< @briefprivate
    typedef ResNum<f64> ResF64;     //!< @briefprivate
#endif // define( NW_UT_RES_SWAP_ENDIAN )

//! @briefprivate
typedef struct BinString
{
    ResS32      offset;

    const char* to_ptr() const
    {
        const u8* p = reinterpret_cast<const u8*>(this);
        if ( offset != 0 ) { return reinterpret_cast<const char*>(p + offset); }
        else { return NULL; }
    }

    void  set_ptr(const char* ptr)
    {
        if (ptr == NULL) { offset = 0; }
        else { offset = GetOffsetFromPtr(this, ptr); }
    }

    operator const char*() const { return to_ptr(); }

} BinString;


//! @briefprivate
typedef struct Offset
{
    ResS32      offset;

//    共用体に入れるために、コンストラクタを実装しない。
//    /* ctor */  Offset() : offset(0) {}
//    /* ctor */  Offset(s32 ofs) : offset(ofs) {}
    static Offset s32_to_ofs(s32 ofs) { return *reinterpret_cast<Offset*>(&ofs); }

    Offset      operator=(s32 ofs) { offset = ofs; return *this; }
    operator s32() const { return offset; }
    Offset      operator+(s32 ofs) const { return  s32_to_ofs(this->offset + ofs); }
    Offset      operator-(s32 ofs) const { return  s32_to_ofs(this->offset - ofs); }
    Offset      operator+=(s32 ofs) { this->offset += ofs; return *this; }
    Offset      operator-=(s32 ofs) { this->offset -= ofs; return *this; }

    void  set_ptr(const void* ptr)
    {
        if (ptr == NULL) { offset = 0; }
        else { offset = GetOffsetFromPtr(this, ptr); }
    }

    void* to_ptr()
    {
        u8* p = reinterpret_cast<u8*>(this);
        if ( offset != 0 ) { return p + offset; }
        else { return NULL; }
    }

    const void* to_ptr() const
    {
        const u8* p = reinterpret_cast<const u8*>(this);
        if ( offset != 0 ) { return p + offset; }
        else { return NULL; }
    }

    template<typename T>
    const T* to_ptr() const
    {
        return static_cast<const T*>( to_ptr() );
    }

    template<typename T>
    T* to_ptr()
    {
        return static_cast<T*>( to_ptr() );
    }

  #if 0
    u32   table_num() const
    {
        return *(static_cast<const u32*>( this ) - 1);
    }
  #endif

    void* to_table_ptr()
    {
        return to_ptr();
    }

    const void* to_table_ptr() const
    {
        return to_ptr();
    }
} Offset;


typedef ResU32 Size;            //!< @briefprivate
typedef ResU32 Length;          //!< @briefprivate
typedef ResU32 ResTypeInfo;     //!< @briefprivate

//! @briefprivate
typedef struct ResBool
{
    ResS8   value;

    operator bool() const { return (value != 0)? true : false; }
    bool operator = (bool rhs) { value = rhs; return bool(*this); }
} ResBool;


#if !defined( NW_UT_RES_SWAP_ENDIAN )
    //! @briefprivate
    typedef nw::math::VEC2        ResVec2; //!< バイナリリソース上の Vector2 型です。
    //! @briefprivate
    typedef nw::math::VEC3        ResVec3; //!< バイナリリソース上の Vector3 型です。
    //! @briefprivate
    typedef nw::math::VEC4        ResVec4; //!< バイナリリソース上の Vector4 型です。
    //! @briefprivate
    typedef nw::math::Transform2  ResTransform2; //!< バイナリリソース上の Transform2 型です。
    //! @briefprivate
    typedef nw::math::Transform3  ResTransform3; //!< バイナリリソース上の Transform3 型です。
    //! @briefprivate
    typedef nw::math::QUAT        ResQuaternion; //!< バイナリリソース上の Quaternion 型です。

    //! @briefprivate
    typedef nw::math::MTX22       ResMtx22; //!< バイナリリソース上の Matrix22 型です。
    //! @briefprivate
    typedef nw::math::MTX23       ResMtx23; //!< バイナリリソース上の Matrix23 型です。
    //! @briefprivate
    typedef nw::math::MTX33       ResMtx33; //!< バイナリリソース上の Matrix33 型です。
    //! @briefprivate
    typedef nw::math::MTX34       ResMtx34; //!< バイナリリソース上の Matrix34 型です。
    //! @briefprivate
    typedef nw::math::MTX44       ResMtx44; //!< バイナリリソース上の Matrix44 型です。

    // TODO: ResVec3とResMtx34が共用体のメンバにできないため、
    //       暫定で追加している。将来的に統合する予定。
    //! @briefprivate
    typedef nw::math::VEC3_       ResVec3_;  //!< @briefprivate
    //! @briefprivate
    typedef nw::math::MTX34_      ResMtx34_; //!< @briefprivate
#else // if defined( NW_UT_RES_SWAP_ENDIAN )

    //! @briefprivate
    typedef struct
    {
        nw::ut::ResF32 x;
        nw::ut::ResF32 y;

        operator nw::math::VEC2() const { nw::math::VEC2 vec; vec.x = x; vec.y = y; return vec; }
        void operator = (const math::VEC2& val) { x = val.x; y = val.y; }
    } ResVec2;

    //! @briefprivate
    typedef struct
    {
        nw::ut::ResF32 x;
        nw::ut::ResF32 y;
        nw::ut::ResF32 z;

        operator nw::math::VEC3() const
        {
            nw::math::VEC3 vec; vec.x = x; vec.y = y; vec.z = z; return vec;
        }
        void operator = (const math::VEC3& val) { x = val.x; y = val.y; z = val.z; }

        void Set(f32 px, f32 py, f32 pz) { x = px; y = py; z = pz; }
    } ResVec3;

    //! @briefprivate
    typedef struct
    {
        nw::ut::ResF32 x;
        nw::ut::ResF32 y;
        nw::ut::ResF32 z;
        nw::ut::ResF32 w;

        operator nw::math::VEC4() const
        {
            nw::math::VEC4 vec; vec.x = x; vec.y = y; vec.z = z; vec.w = w; return vec;
        }
        void operator = (const nw::math::VEC4& val) { x = val.x; y = val.y; z = val.z; w = val.w; }
    } ResVec4;

    //! @briefprivate
    typedef struct
    {
        ResVec2     scale;
        ResVec2     rotate;
        ResVec2     translate;

        operator nw::math::Transform2() const
        {
            nw::math::Transform2 transform;
            transform.scale     = this->scale;
            transform.rotate    = this->rotate;
            transform.translate = this->translate;
            return transform;
        }
    } ResTransform2;

    //! @briefprivate
    typedef struct
    {
        ResVec3     scale;
        ResVec3     rotate;
        ResVec3     translate;

        operator nw::math::Transform3() const
        {
            nw::math::Transform3 transform;
            transform.scale     = this->scale;
            transform.rotate    = this->rotate;
            transform.translate = this->translate;
            return transform;
        }
    } ResTransform3;


    //! @briefprivate
    typedef struct
    {
        ResF32 x;
        ResF32 y;
        ResF32 z;
        ResF32 w;

        operator nw::math::QUAT() const
        {
            nw::math::QUAT quat;
            quat.x = x; quat.y = y; quat.z = z; quat.w = w;
            return quat;
        }
    } ResQuaternion;


    //! @briefprivate
    typedef struct
    {
        ResF32 a[4];

        operator nw::math::MTX22() const
        {
            nw::math::MTX22 m;
            for (int i = 0; i < 4; ++i ) { m.a[i] = this->a[i]; }
            return m;
        }
    } ResMtx22;

    //! @briefprivate
    typedef struct
    {
        ResF32 a[6];

        operator nw::math::MTX23() const
        {
            nw::math::MTX23 m;
            for (int i = 0; i < 6; ++i ) { m.a[i] = this->a[i]; }
            return m;
        }
    } ResMtx23;

    //! @briefprivate
    typedef struct
    {
        ResF32 a[9];

        operator nw::math::MTX33() const
        {
            nw::math::MTX33 m;
            for (int i = 0; i < 9; ++i ) { m.a[i] = this->a[i]; }
            return m;
        }
    } ResMtx33;

    //! @briefprivate
    typedef struct
    {
        ResF32 a[12];

        operator nw::math::MTX34() const
        {
            nw::math::MTX34 m;
            for (int i = 0; i < 12; ++i ) { m.a[i] = this->a[i]; }
            return m;
        }
    } ResMtx34;

    //! @briefprivate
    typedef struct
    {
        ResF32 a[16];

        operator nw::math::MTX44() const
        {
            nw::math::MTX44 m;
            for (int i = 0; i < 16; ++i ) { m.a[i] = this->a[i]; }
            return m;
        }
    } ResMtx44;
#endif

} /* namespace ut */
} /* namespace nw */

#endif /* NW_UT_RESTYPES_H_ */
