﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_RESPRIMITIVE_H_
#define NW_UT_RESPRIMITIVE_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/res/ut_ResTypes.h>
#include <nw/ut/res/ut_ResUtil.h>
#include <nw/ut/res/ut_ResDeclMacros.h>

namespace nw {
namespace ut {
namespace internal {

//! @briefprivate
struct ResBoundingVolumeData
{
    ResTypeInfo typeInfo;
};

//! @briefprivate
struct ResOrientedBoundingBoxData : public ResBoundingVolumeData
{
    ResVec3   m_CenterPosition;
    ResMtx33  m_OrientationMatrix;
    ResVec3   m_Size;
};

//! @briefprivate
struct ResAxisAlignedBoundingBoxData : public ResBoundingVolumeData
{
    ResVec3   m_CenterPosition;
    ResVec3   m_Size;
};


//--------------------------------------------------------------------------
//! @brief        バウンディング情報です。
//---------------------------------------------------------------------------
class ResBoundingVolume : public ResCommon<ResBoundingVolumeData>
{
public:
    enum { TYPE_INFO = NW_UT_RES_TYPE_INFO(ResBoundingVolume) };

    NW_RES_CTOR( ResBoundingVolume )

    //---------------------------------------------------------------------------
    //! @brief        インスタンスの型情報を取得します。
    //!
    //! @return       型情報です。
    //---------------------------------------------------------------------------
    ResTypeInfo     GetTypeInfo() const { return ref().typeInfo; }
};

//--------------------------------------------------------------------------
//! @brief        OBB の情報です。
//---------------------------------------------------------------------------
class ResOrientedBoundingBox : public ResBoundingVolume
{
public:
    enum { TYPE_INFO = NW_UT_RES_TYPE_INFO(ResOrientedBoundingBox) };

    NW_RES_CTOR_INHERIT( ResOrientedBoundingBox, ResBoundingVolume )


    //---------------------------------------------------------------------------
    //! @brief        OBB の中心座標を設定します。
    //! @param[in]    x 中心座標のxの値です。
    //! @param[in]    y 中心座標のyの値です。
    //! @param[in]    z 中心座標のzの値です。
    //---------------------------------------------------------------------------
    void  SetCenterPosition( f32 x, f32 y, f32 z ) { ref().m_CenterPosition.Set(x, y, z); }
    //---------------------------------------------------------------------------
    //! @brief        OBB の中心座標を取得します。
    //! @return       中心座標です。
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetCenterPosition() const { return ref().m_CenterPosition; }
    //---------------------------------------------------------------------------
    //! @brief        OBB の中心座標を設定します。
    //! @param[in]    value 中心座標です。
    //---------------------------------------------------------------------------
    void  SetCenterPosition(const nw::math::VEC3& value) { ref().m_CenterPosition = value; }

    //---------------------------------------------------------------------------
    //! @brief        姿勢行列を取得します。
    //! @return       姿勢行列です。
    //---------------------------------------------------------------------------
    nw::math::MTX33& GetOrientationMatrix() { return ref().m_OrientationMatrix; }
    //---------------------------------------------------------------------------
    //! @brief        姿勢行列を取得します。
    //! @return       姿勢行列です。
    //---------------------------------------------------------------------------
    const nw::math::MTX33& GetOrientationMatrix() const { return ref().m_OrientationMatrix; }
    //---------------------------------------------------------------------------
    //! @brief        姿勢行列を設定します。
    //! @param[in]    value 設定する姿勢行列です。
    //---------------------------------------------------------------------------
    void SetOrientationMatrix(const nw::math::MTX33& value) { ref().m_OrientationMatrix = value; }


    //---------------------------------------------------------------------------
    //! @brief         姿勢行列座標空間におけるサイズを設定します。
    //! @param[in]     x OBB の x 軸方向のサイズです。
    //! @param[in]     y OBB の y 軸方向のサイズです。
    //! @param[in]     z OBB の z 軸方向のサイズです。
    //---------------------------------------------------------------------------
    void SetSize( f32 x, f32 y, f32 z ) { ref().m_Size.Set(x, y, z); }
    //---------------------------------------------------------------------------
    //! @brief        姿勢行列座標空間におけるサイズを取得します。
    //! @return       サイズです。
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetSize() const { return ref().m_Size; }
    //---------------------------------------------------------------------------
    //! @brief         姿勢行列座標空間におけるサイズを設定します。
    //! @param[in]     value OBB のサイズです。
    //---------------------------------------------------------------------------
    void SetSize(const nw::math::VEC3& value) { ref().m_Size = value; }
};

//--------------------------------------------------------------------------
//! @brief        AABB の情報です。
//---------------------------------------------------------------------------
class ResAxisAlignedBoundingBox : public ResBoundingVolume
{
public:
    enum { TYPE_INFO = NW_UT_RES_TYPE_INFO(ResAxisAlignedBoundingBox) };

    NW_RES_CTOR_INHERIT( ResAxisAlignedBoundingBox, ResBoundingVolume )

    //---------------------------------------------------------------------------
    //! @brief        AABB の中心座標を設定します。
    //! @param[in]    x 中心座標のxの値です。
    //! @param[in]    y 中心座標のyの値です。
    //! @param[in]    z 中心座標のzの値です。
    //---------------------------------------------------------------------------
    void SetCenterPosition( f32 x, f32 y, f32 z ) { ref().m_CenterPosition.Set(x, y, z); }
    //---------------------------------------------------------------------------
    //! @brief        AABB の中心座標を取得します。
    //! @return       中心座標です。
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetCenterPosition() const { return ref().m_CenterPosition; }
    //---------------------------------------------------------------------------
    //! @brief        AABB の中心座標を設定します。
    //! @param[in]    value 中心座標です。
    //---------------------------------------------------------------------------
    void SetCenterPosition(const nw::math::VEC3& value) { ref().m_CenterPosition = value; }

    //---------------------------------------------------------------------------
    //! @brief        軸平行座標空間におけるサイズを設定します。
    //! @param[in]    x AABB の x 軸方向のサイズです。
    //! @param[in]    y AABB の y 軸方向のサイズです。
    //! @param[in]    z AABB の z 軸方向のサイズです。
    //---------------------------------------------------------------------------
    void  SetSize( f32 x, f32 y, f32 z ) { ref().m_Size.Set(x, y, z); }
    //---------------------------------------------------------------------------
    //! @brief        軸平行座標空間におけるサイズを取得します。
    //! @return       サイズです。
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetSize() const { return ref().m_Size; }
    //---------------------------------------------------------------------------
    //! @brief        軸平行座標空間におけるサイズを設定します。
    //! @param[in]    value AABB のサイズです。
    //---------------------------------------------------------------------------
    void  SetSize(const nw::math::VEC3& value) { ref().m_Size = value; }
};

} /* namespace internal */
} /* namespace ut */
} /* namespace nw */

#endif /* NW_UT_RESPRIMITIVE_H_ */
