﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MUTEX_H_
#define NW_UT_MUTEX_H_

#include <nw/types.h>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/os/OSMutex.h>
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    #include <pthread.h>
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief    Mutex を使用したロックオブジェクトの具象クラスです。
//---------------------------------------------------------------------------
class Mutex
{
public:
    /* ctor */ Mutex() : m_IsIntialized(false) {}
    /* dtor */ ~Mutex() {}

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief    初期化処理をおこないます。
    //---------------------------------------------------------------------------
    void Initialize()
    {
        if ( ! m_IsIntialized )
        {
            m_IsIntialized = true;
            OSInitMutex(&m_Lock);
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    終了処理をおこないます。
    //---------------------------------------------------------------------------
    void Finalize()
    {
        m_IsIntialized = false;
    }

    //---------------------------------------------------------------------------
    //! @brief    ロック処理です。
    //---------------------------------------------------------------------------
    void Lock()
    {
        NW_ASSERT(m_IsIntialized);
        OSLockMutex(&m_Lock);
    }

    //---------------------------------------------------------------------------
    //! @brief    ウェイトせずに失敗を返すロック処理です。
    //---------------------------------------------------------------------------
    bool TryLock()
    {
        NW_ASSERT(m_IsIntialized);
        return (OSTryLockMutex( &m_Lock ) != FALSE);
    }

    //---------------------------------------------------------------------------
    //! @brief    ロックの解放処理です。
    //---------------------------------------------------------------------------
    void Unlock()
    {
        NW_ASSERT(m_IsIntialized);
        OSUnlockMutex(&m_Lock);
    }

    //---------------------------------------------------------------------------
    //! @brief    ロックの解放処理です。
    //!
    //! @details  Cafe 版のみ定義しています。
    //---------------------------------------------------------------------------
    void WaitCond(OSCond& cond)
    {
        NW_ASSERT(m_IsIntialized);
        OSWaitCond(&cond, &m_Lock);
    }

#else

    void Initialize();
    void Finalize();
    void Lock();
    bool TryLock();
    void Unlock();

#endif

private:
  #if defined(NW_PLATFORM_CAFE)
    OSMutex m_Lock;
  #elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    static const u32 CRITICAL_SECTION_SIZE = 24;

    u32 m_CriticalSection[ CRITICAL_SECTION_SIZE / sizeof(u32) ];
  #elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    pthread_mutex_t m_CriticalSection;
  #endif

    bool    m_IsIntialized;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_MUTEX_H_

