﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/sndutil_MidiManager.h
 *
 * @file sndutil_MidiManager.h
 */

#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_NoteOnCallback.h>
#include <nw/snd/snd_MidiStreamParser.h>
#include <nw/snd/snd_MidiSequencePlayer.h>
#include <nw/snd/snd_MidiSequenceTrack.h>
#include <nw/snd/snd_MidiSequenceTrackAllocator.h>
#include <nw/ut/os/ut_Thread.h>
#include <nw/ut/os/ut_CriticalSection.h>

namespace nw {
namespace snd {

class SoundDataManager;

namespace util {

//! @brief  MIDI 信号を受診して、指定したバンクで再生させるためのクラスです。
//!
//!         デバッグ以外の用途で使うことは禁止です。
//!
//!         また、試験的な実装のため今後インターフェイスが変わったり、
//!         廃止されたりする可能性があります。
//!
//!         詳しくは、Demo/snd/midi デモをご覧ください
//!
//!         Debug 版、Develop 版で利用可能です。
//!         Release 版ではリンクすることは可能ですが、各関数内で何も処理されません。
//!
//! @date 2012/06/18 ビルドターゲット別の説明を追加
//! @date 2012/04/06 初版
class MidiManager :
    public nw::snd::internal::driver::SoundThread::SoundFrameCallback,
    public nw::snd::internal::driver::NoteOnCallback,
    public nw::ut::ThreadHandler
{
public:
    //! @name コンストラクタ
    //@{
    //! @brief コンストラクタです。
    //! @date 2012/04/06 初版
    MidiManager();
    //@}

    //! @name 初期化・終了処理
    //@{

    //! @brief  初期化に必要なメモリサイズを取得します。
    //! @param[in] stackSize 内部で生成される MIDI 受信スレッドのスタックサイズです。
    //! @return 初期化に必要なメモリサイズを返します。
    //! @date   2012/04/06 初版
    size_t GetRequiredMemSize( size_t stackSize = STACK_SIZE ) const;

    //! @brief  MIDI マネージャの初期化を行います。
    //!
    //!         引数の内容を元に、内部で MIDI 受信用のスレッドを起動します。
    //!         MIDI 受信スレッドは、本関数を呼び出したコア上で起動されます。
    //!
    //!
    //!
    //! @param[in] threadPriority   MIDI 受信スレッドの優先度です。
    //! @param[in] stackBase        MIDI 受信スレッドのスタックとして利用されるメモリ領域です。
    //!                             @ref GetRequiredMemSize
    //!                             で取得できたサイズのメモリ領域を渡す必要があります。
    //! @param[in] stackSize        MIDI 受信スレッドのスタックサイズです。
    //!
    //! @return 初期化に成功あるいは初期化済みの場合は true を返します。
    //!         初期化に失敗すると false を返します。
    //!
    //! @see GetRequiredMemSize
    //! @see Finalize
    //! @date 2012/04/06 初版
    bool Initialize( s32 threadPriority, void* stackBase, size_t stackSize );

    //! @brief  MIDI マネージャを破棄します。
    //!
    //!         内部で起動されたスレッドも破棄されます。
    //!
    //! @see Initialize
    //! @date 2012/04/06 初版
    void Finalize();
    //@}

    //! @name データ設定
    //@{
    //! @brief  MIDI 再生に使用されるバンクデータを設定します。
    //!
    //!         本関数では、バンク ID のみ設定されますが、
    //!         指定するバンクおよびバンクに関連付けられた波形アーカイブは、
    //!         ユーザー側でロードしておく必要があります。
    //!
    //! @param[in] bankIds  バンク ID です。最大 4 つまで設定することができます。
    //! @param[in] arc      サウンドアーカイブです。
    //! @param[in] mgr      サウンドデータマネージャです。
    //!                     このサウンドデータマネージャに関連付けられたバンクや波形アーカイブを使って
    //!                     MIDI 再生が行われます。
    //!
    //! @date 2012/04/06 初版
    void Prepare(
        SoundArchive::ItemId bankIds[],
        const SoundArchive& arc,
        const SoundDataManager& mgr);
    //@}

    //! @name MIDI 通信状況確認
    //@{
    //! @brief  MIDI 信号を受信したかどうかを取得します。
    //! @param[in] portIndex   ポート番号です。
    //! @return MIDI 信号を受信したかどうかを返します。
    //! @see    ResetMidiRecieveFlag
    //! @date   2012/04/06 初版
    bool GetMidiRecieveFlag( int portIndex = 0 ) const
    {
        return m_MidiPort[portIndex].m_RecieveFlag;
    }

    //! @brief  MIDI 信号の受信状態をリセットします。
    //! @param[in] portIndex   ポート番号です。
    //! @see    GetMidiRecieveFlag
    //! @date   2012/04/06 初版
    void ResetMidiRecieveFlag( int portIndex = 0 )
    {
        m_MidiPort[portIndex].m_RecieveFlag = false;
    }
    //@}

    //! @name パラメータ設定・取得
    //@{
    //! @brief  MIDI 再生の音量を設定します。
    //! @param[in] volume   MIDI 再生の音量です。
    //! @see    GetVolume
    //! @date   2012/04/06 初版
    void SetVolume( f32 volume ) { m_SequencePlayer.SetVolume( volume ); }

    //! @brief  MIDI 再生の音量を取得します。
    //! @return MIDI 再生の音量を返します。
    //! @see    SetVolume
    //! @date   2012/04/06 初版
    f32 GetVolume() const { return m_SequencePlayer.GetVolume(); }

    //! @brief MIDI 再生の出力先を設定します。
    //!
    //!        @ref SoundHandle::SetOutputLine と同じ引数を渡します。
    //! @param[in] lineFlag 出力先のビットフラグです。
    //! @see   SoundHandle::SetOutputLine
    //! @date  2012/05/08 初版
    void SetOutputLine( u32 lineFlag ) { m_SequencePlayer.SetOutputLine( lineFlag ); }
    //@}

    //! @briefprivate
    //! @param status :private
    //! @param data1 :private
    //! @param data2 :private
    //! @param arg :private
    static void MidiCallback( u8 status, u8 data1, u8 data2, void* arg );

    //! @brief  MIDI 受信スレッドのデフォルトスタックサイズです。
    //!
    //!         @ref GetRequiredMemSize のデフォルト引数として利用されます。
    //!
    //! @date   2012/04/06 初版
    //! @see    GetRequiredMemSize
    static const size_t STACK_SIZE = 16 * 1024; // 16KB
private:
    static const int MIDI_PORT_CHECK_INTERVAL_US = 3000;
    static const int BANK_INDEX_MAX = 4;
    static const int MIDI_PORT_MAX = 4;
    static const int MIDI_BUFFER_SIZE = 1024;   // CTR では nn::midi::MIDI_RECV_BUFFER_MAX

    struct MidiBuffer
    {
        u8 buffer[MIDI_BUFFER_SIZE];
        u32 size;
    };

    struct MidiPort
    {
        nw::snd::internal::MidiStreamParser m_Parser;
        MidiBuffer m_MidiBuffer[2];
        MidiBuffer* m_pReadBuffer;
        MidiBuffer* m_pWriteBuffer;
        bool m_RecieveFlag;

        MidiPort() : m_RecieveFlag( false )
        {
            m_pReadBuffer = &m_MidiBuffer[0];
            m_pWriteBuffer = &m_MidiBuffer[1];
        }
    };

    void RegisterSoundThreadCallback();
    void UnregisterSoundThreadCallback();

    void Prepare( const void* banks[],
            const void* warcs[],
            bool warcIsIndividuals[] )
    {
        NW_NULL_ASSERT( banks );
        NW_NULL_ASSERT( warcs );
        NW_NULL_ASSERT( warcIsIndividuals );
        m_SequencePlayer.Prepare( banks, warcs, warcIsIndividuals );
    }

    virtual void ThreadHandlerProc();

    virtual nw::snd::internal::driver::Channel* NoteOn(
            nw::snd::internal::driver::SequenceSoundPlayer* pSequencePlayer,
            u8 bankIndex,
            const nw::snd::internal::driver::NoteOnInfo& noteOnInfo
            );
    virtual void OnBeginSoundFrame();

    nw::ut::Thread m_RecieveThread;
    MidiPort m_MidiPort[MIDI_PORT_MAX];
    nw::snd::internal::driver::MidiSequencePlayer m_SequencePlayer;
    nw::snd::internal::driver::MidiSequenceTrack m_SequenceTrack[16];
    nw::snd::internal::driver::MidiSequenceTrackAllocator m_SequenceTrackAllocator;

    const nw::snd::SoundArchivePlayer* m_pSoundArchivePlayer;
    nw::ut::CriticalSection m_CriticalSection;
    volatile bool m_FinalizeFlag;
    bool m_InitializeThreadFlag;
};

}}} // namespace nw::snd::util

