﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_NW_SND_SPY_H_
#define NW_SND_SPY_NW_SND_SPY_H_

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

#include <nw/snd.h>
#include <nw/snd/spy/sndspy_SpyData.h>
#include <nw/snd/spy/sndspy_SpyDataInfo.h>
#include <nw/snd/spy/modules/sndspy_SpyModule.h>

namespace nw {
namespace snd {
namespace spy {

class SpyController;

class AxFinalOutModule : public SpySimpleModule
{
public:
    AxFinalOutModule() : SpySimpleModule("FinalOut", FinalOutPacketData::VERSION) { }
};

class AxProfileModule : public SpySimpleModule
{
public:
    AxProfileModule() : SpySimpleModule("AxProfile", ProfilePacketData::VERSION) { }
};

class NwSoundDataInfoModule : public SpySimpleModule
{
public:
    NwSoundDataInfoModule()
        : SpySimpleModule("NwSoundDataInfo", SoundDataInfoPacketData::VERSION)
        , m_RequestCount(0)
        , m_LastRequestCount(0)
    {
    }

    //! @brief データ要求があるか調べます。
    //!
    //! 呼び出しのたびにデータ要求はリセットされます。
    //!
    //! IsDataRequested() は常に同じスレッドから呼ばれる必要があります。
    //! OnRequested() と IsDataRequested() は別のスレッドから呼び出されてもかまいません。
    //!
    //! @return
    //! データ要求があった場合は true を返します。
    //!
    bool IsDataRequested()
    {
        u32 count = m_RequestCount;
        if (m_LastRequestCount != count)
        {
            m_LastRequestCount = count;
            return true;
        }
        else
        {
            return false;
        }
    }

protected:
    virtual void OnRequested(bool bRequested)
    {
        SpyModule::OnRequested(bRequested);

        if (bRequested)
        {
            ++m_RequestCount;
        }
    }

private:
    volatile u32 m_RequestCount;
    u32 m_LastRequestCount;
};

class NwSoundStatusInfoModule : public SpySimpleModule
{
public:
    NwSoundStatusInfoModule() : SpySimpleModule("NwSoundStatusInfo", SoundStatusInfoPacketData::VERSION) { }
};

class AxVoiceInfoModule : public SpySimpleModule
{
public:
    AxVoiceInfoModule() : SpySimpleModule("AxVoiceInfo", VoiceInfoPacketData::VERSION) { }
};

//! @briefprivate
//!
class NwSndSpy
{
public:
    static NwSndSpy& GetInstance() {
        static NwSndSpy instance;
        return instance;
    }
    void Initialize( SpyController* controller );
    void Finalize();

    bool IsInitialized() const { return m_pController != NULL; }

    void AttachSoundArchive( const nw::snd::SoundArchive& soundArchive );
    void DetachSoundArchive( const nw::snd::SoundArchive& soundArchive );

    void EnableLog()
    {
        m_IsEnableLog = true;
        InitializeLog();
    }

    void DisableLog()
    {
        m_IsEnableLog = false;
        FinalizeLog();
    }

    void EnableProfile()
    {
        m_IsEnableProfile = true;
        InitializeProfile();
    }

    void DisableProfile()
    {
        m_IsEnableProfile = false;
        FinalizeProfile();
    }

    void EnableVoiceInfo()
    {
        m_IsEnableVoiceInfo = true;
    }

    void DisableVoiceInfo()
    {
        m_IsEnableVoiceInfo = false;
    }

    bool SendSoundArchivePlayerInfo( const nw::snd::SoundArchivePlayer& soundArchivePlayer );

private:
    //! @briefprivate
    class FrameCallback : public nw::snd::internal::driver::SoundThread::SoundFrameCallback
    {
    public:
        FrameCallback( NwSndSpy& spy ) : m_Spy(spy) {}
        virtual ~FrameCallback() {}
        virtual void OnBeginSoundFrame();
        virtual void OnEndSoundFrame();
    private:
        NwSndSpy& m_Spy;
    };
    friend class FrameCallback;

    //! @briefprivate
    struct VoiceInfoCondition
    {
        bool dirty;
        VoiceInfoPacketData::VoiceInfo packet;
    };

    static void SpyLog( const char* message );
    static void AxFrameCallbackFunc();
    static void ConnectionChangedCallbackFunc( void* arg );

    NwSndSpy();
    void WriteSoundInfoToPacket(
        SoundStatusInfoPacketData* packet,
        const nw::snd::internal::BasicSound* sound );
    void SetCurrentAudioFrame();
    bool SendFinalOutPacket();
    void ConnectionChangedCallback();

    bool SendSoundDataInfo();
    bool SendSoundDataSoundInfo( const nw::snd::SoundArchive& soundArchive );
    bool SendSoundDataPlayerInfo( const nw::snd::SoundArchive& soundArchive );

    void InitializeLog();
    void FinalizeLog();
    void InitializeProfile();
    void FinalizeProfile();
    void InitializeDataInfo();
    void FinalizeDataInfo();

private:
    SpyController* m_pController;

    AxFinalOutModule m_AxFinalOutModule;
    AxProfileModule m_AxProfileModule;
    AxVoiceInfoModule m_AxVoiceInfoModule;
    NwSoundDataInfoModule m_NwSoundDataInfoModule;
    NwSoundStatusInfoModule m_NwSoundStatusInfoModule;

    FrameCallback m_FrameCallback;
    nw::snd::ProfileReader m_ProfileReader;

    const nw::snd::SoundArchive* m_pSoundArchive;

    bool m_IsEnableLog;
    bool m_IsEnableProfile;
    bool m_IsEnableVoiceInfo;
    u8   m_Padding;

    bool m_IsLogFuncHooked;
    bool m_IsProfileReaderRegistered;
    bool m_IsDataInfoInitialized;
    u8   m_Padding2[1];

    VoiceInfoCondition m_VoiceInfoCondition[VoiceInfoPacketData::VOICE_COUNT_MAX];
};

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_NW_SND_SPY_H_
