﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_WaveSoundHandle.h
 *
 * @file snd_WaveSoundHandle.h
 */

#ifndef NW_SND_WAVE_SOUND_HANDLE_H_
#define NW_SND_WAVE_SOUND_HANDLE_H_

#include <nw/ut/ut_PreProcessor.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_WaveSound.h>

namespace nw {
namespace snd {

class SoundHandle;

//---------------------------------------------------------------------------
//! @brief    再生したウェーブサウンドの操作を行うためのハンドルクラスです。
//!
//!           サウンドは再生時に汎用サウンドハンドル
//!           @ref SoundHandle と関連付けられますが、
//!           ウェーブサウンドに固有の処理は汎用サウンドハンドルには実装されていません。
//!
//!           ウェーブサウンドに固有の処理を行いたい場合は、
//!           汎用サウンドハンドルを引数にとるコンストラクタを用いて、
//!           ウェーブサウンドハンドルを生成し、使用してください。
//!
//!           ハンドルの使い方は、汎用サウンドハンドル @ref SoundHandle と同じです。
//!
//! @see      SoundHandle クラス
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class WaveSoundHandle
{
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           引数の無いコンストラクタは、
    //!           どのサウンドとも関連付けられていないハンドルを構築します。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    WaveSoundHandle() : m_pSound( NULL ) { }

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           引数付きコンストラクタは、
    //!           引数 handle で指定したサウンドハンドルが参照しているサウンドを用いて、
    //!           ウェーブサウンドハンドルを構築します。
    //!           ウェーブサウンド以外のサウンドがサウンドハンドルに関連付けられているときは、
    //!           引数の無いコンストラクタと同様に構築します。
    //!
    //!           引数付きコンストラクタで構築した時、
    //!           サウンドが既に別のウェーブサウンドハンドルと関連付けられている場合は、
    //!           その関連付けが切られる点に注意してください。
    //!           ただし、引数に渡したサウンドハンドルとの関連付けは切れません。
    //!
    //! @param[in] handle ウェーブサウンドと関連付けられたサウンドハンドル。
    //!
    //! @see      SoundHandle クラス
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    explicit WaveSoundHandle( SoundHandle* handle );

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    ~WaveSoundHandle() { DetachSound(); }

    //@}


    //! @name 再生/停止/一時停止/再開
    //@{

    //---------------------------------------------------------------------------
    //! @brief    再生準備が完了したサウンドを再生します。
    //!
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           サウンドの再生を開始するためには、
    //!           @ref SoundArchivePlayer::PrepareSound を呼び出した後、
    //!           再生準備が完了していなければなりません。
    //!           この関数は、再生準備が完了したサウンドの再生を開始します。
    //!           再生準備が完了していないサウンドは、完了するまで待ってから再生を開始します。
    //!
    //! @see nw::snd::SoundArchivePlayer::PrepareSound
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void StartPrepared()
    {
        if ( IsAttachedSound() ) m_pSound->StartPrepared();
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを停止します。
    //!
    //!           ハンドルに関連付けられたサウンドを停止します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトさせることができます。
    //!           0 を指定した場合はフェードアウトを行わず、
    //!           エンベロープのリリースを発音し全ての減衰が完了した後にサウンドが停止します。
    //!
    //!           フェードアウトの音量制御は、フェードインと共有されます。
    //!           フェードアウトにかかるフレーム数は、
    //!           最大音量から音が消えるまでにかかる変化速度を表しますので、
    //!           フェードイン中にフェードアウトを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードアウトが完了する可能性があります。
    //!
    //! @param[in] fadeFrames フェードアウトにかけるフレーム数です。
    //!                       フレーム数は @ref SoundArchivePlayer::Update
    //!                       の呼び出し回数で換算されます。
    //!                       fadeFrames には 0 以上の値を指定してください。
    //!                       負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Stop( int fadeFrames )
    {
        if ( IsAttachedSound() ) m_pSound->Stop( fadeFrames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを一時停止または再開します。
    //!
    //!           ハンドルに関連付けられたサウンドを一時停止または再開します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           fadeFrames で指定したフレーム数をかけてフェードアウトしながら一時停止、
    //!           またはフェードインしながら再開させることができます。
    //!           0 を指定した場合は、即座に一時停止または再開します。
    //!
    //!           一時停止・再開時のフェードは、再生開始時のフェードイン、
    //!           停止時のフェードアウトとは独立してはたらきます。
    //!           フェードにかかるフレーム数は、最大音量から音が消えるまで、
    //!           あるいは、
    //!           発音していない状態から最大音量に達するまでにかかる変化速度を表しますので、
    //!           フェード中にさらにフェードを指定した時などは、
    //!           指定したフレーム数よりも短い時間でフェードが完了する可能性があります。
    //!
    //! @param[in]    flag        true なら一時停止、false なら再開します。
    //! @param[in]    fadeFrames  フェードイン・フェードアウトにかけるフレーム数です。
    //!                           フレーム数は @ref SoundArchivePlayer::Update
    //!                           の呼び出し回数で換算されます。
    //!                           fadeFrames には 0 以上の値を指定してください。
    //!                           負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see  IsPause
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Pause( bool flag, int fadeFrames )
    {
        if ( IsAttachedSound() ) m_pSound->Pause( flag, fadeFrames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの再生準備が完了しているかどうかを調べます。
    //!
    //!           @ref SoundArchivePlayer::PrepareSound を呼び出した後、
    //!           そのサウンドの再生準備が完了しているかどうかを調べます。
    //!           再生準備が完了したサウンドは、@ref StartPrepared を呼び出した際に、
    //!           すぐに再生を始めることができます。
    //!
    //! @return   サウンドの再生準備が完了していれば true を返します。
    //!
    //! @see  SoundArchivePlayer::PrepareSound
    //! @see  StartPrepared
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPrepared() const { return IsAttachedSound() && m_pSound->IsPrepared(); }

    //---------------------------------------------------------------------------
    //! @brief    サウンドが一時停止中かどうかを調べます。
    //!
    //! @return   サウンドが一時停止状態であれば true を返します。
    //!
    //! @see Pause
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsPause() const { return IsAttachedSound() && m_pSound->IsPause(); }

    //---------------------------------------------------------------------------
    //! @brief    サウンドを再生開始時にフェードインさせます。
    //!
    //!           この関数でフェードインの指定を行うと、
    //!           @ref SoundArchivePlayer::StartSound または
    //!           @ref StartPrepared を呼び出して再生を開始してから、
    //!           最初の @ref SoundArchivePlayer::Update
    //!           が呼び出されたときにフェードインが設定されます。
    //!
    //!           サウンドの再生が開始され、最初の @ref SoundArchivePlayer::Update
    //!           が呼ばれた後に、この関数を呼び出しても効果がありません。
    //!
    //!           フェードインの音量制御は、停止時のフェードアウトと共有されます。
    //!
    //! @param[in] frames フェードインにかけるフレーム数です。
    //!                   フレーム数は @ref SoundArchivePlayer::Update
    //!                   の呼び出し回数で換算されます。
    //!                   frames には 0 以上の値を指定してください。
    //!                   負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @see  SoundArchivePlayer::StartSound
    //! @see  StartPrepared
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void FadeIn( int frames )
    {
        if ( IsAttachedSound() ) m_pSound->FadeIn( frames );
    }

    //@}


    //! @name パラメータ変更
    //@{

    //---------------------------------------------------------------------------
    //! @brief    サウンドの音量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの音量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //!           音量の変化は frames で指定したフレーム数をかけて行われます。
    //!           音量の変化途中にさらにこの関数を呼び出した場合は、
    //!           その時点での変化途中の音量値を基点として、
    //!           新しく指定したフレーム数をかけて音量を変化させます。
    //!
    //! @param[in] volume 変更する音量の倍率 (0.0～) です。
    //! @param[in] frames 音量変化にかけるフレーム数です。
    //!                   フレーム数は @ref SoundArchivePlayer::Update
    //!                   の呼び出し回数で換算されます。
    //!                   frames には 0 以上の値を指定してください。
    //!                   負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 frames の範囲について追記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVolume( f32 volume, int frames = 0 )
    {
        if ( IsAttachedSound() ) m_pSound->SetVolume( volume, frames );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの音程を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの音程を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどの音程パラメータとも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           音程 pitch は、周波数の比率で指定します。
    //!           すなわち、1.0 を指定すると音程に影響を与えません。
    //!           2.0 を指定すると再生される周波数が 2 倍になり、
    //!           1 オクターブ高い音程になります。
    //!           0.5 を指定すると 1 オクターブ低い音程になります。
    //!           デフォルト値は 1.0 です。
    //!
    //! @param[in] pitch  変更する音程の周波数比率です。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPitch( f32 pitch )
    {
        if ( IsAttachedSound() ) m_pSound->SetPitch( pitch );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドのローパスフィルタカットオフ値を変更します。
    //!
    //!           ハンドルに関連付けられたローパスフィルタカットオフ値を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、
    //!           他のどのローパスフィルタカットオフパラメータとも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           lpfFreq は、カットオフの相対変化の値を指定します。
    //!           0.0 を指定するとカットオフの値を変更しません。
    //!           -1.0 を指定すると、フィルタがかかっていない状態から、
    //!           もっともフィルタがかかっている状態 (カットオフ周波数が下がる方向)
    //!           に変更します。
    //!
    //! @param[in] lpfFreq    0.0 を基準としたフィルタカットオフの相対変化の値です。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetLpfFreq( f32 lpfFreq )
    {
        if ( IsAttachedSound() ) m_pSound->SetLpfFreq( lpfFreq );
    }

    //---------------------------------------------------------------------------
    //! @brief    プレイヤープライオリティを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのプレイヤープライオリティを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数は、サウンドアーカイブ中のデータで指定されている
    //!           プレイヤープライオリティの値を変更します。
    //!
    //!           priority の値の範囲は 0～127 で、大きいほど優先度が高くなります。
    //!
    //! @param[in] priority   プレイヤープライオリティの値です。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPlayerPriority( int priority )
    {
        if ( IsAttachedSound() ) m_pSound->SetPlayerPriority( priority );
    }

    //---------------------------------------------------------------------------
    //! @brief    発音プライオリティを変更します。
    //!
    //!           ハンドルに関連付けられたサウンドの発音プライオリティを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数は、
    //!           サウンドアーカイブ中のデータで指定されている、
    //!           発音プライオリティの値を変更します。
    //!
    //!           priority の値の範囲は 0～127 で、大きいほど優先度が高くなります。
    //!
    //! @param[in] priority   発音プライオリティの値です。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetChannelPriority( int priority )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetChannelPriority( priority );
        }
    }
    //@}

    //----------------------------------------
    //! @name メイン (TV) 出力用パラメータ
    //@{

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するパン (左右の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのパンパラメータとも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           pan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!           -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] pan    0.0 を基準としたパンの相対変化の値です。
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPan( f32 pan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetPan( pan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するサラウンドパン (前後の定位) を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するサラウンドパンを変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのサラウンドパンパラメータとも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           surroundPan は、定位の相対変化の値を設定します。
    //!           0.0 を指定するとデータで設定されたパンの値から変化しません。
    //!           1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!           2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!           前方へ定位を移動させたい場合は負の値を指定してください。
    //!           デフォルト値は 0.0 です。
    //!
    //! @param[in] surroundPan    0.0 を基準としたサラウンドパンの相対変化の値です。
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetSurroundPan( f32 surroundPan )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetSurroundPan( surroundPan );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するメインセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するメインセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのメインセンド量とも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           メインセンドは、
    //!           出力に送るサウンドの音量をエフェクトセンドの後で調節するパラメータです。
    //!           主に、エフェクトのドライ・ウェット成分のバランスを調整するために使用されます。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが、
    //!           メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see SetFxSend
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetMainSend( f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetMainSend( send );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    メイン (TV) 出力に対するエフェクトセンド量を変更します。
    //!
    //!           ハンドルに関連付けられたサウンドのメイン出力に対するエフェクトセンド量を変更します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           この関数で指定する値は、他のどのエフェクトセンド量とも独立して動作し、
    //!           それらは全て重ね合わされます。
    //!
    //!           センド量 send は、相対変化の値を指定します。
    //!           すなわち、0.0 を指定するとセンド量を変更しません。
    //!           1.0 を指定すると AUX バスに送られていなかったサウンドが、
    //!           最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] bus    センド量を設定する AUX のバスです。
    //! @param[in] send   0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see AuxBus
    //! @see SetMainSend
    //!
    //! @date 2011/10/31 メイン (TV) 出力に対するパラメータ変更である旨、明記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetFxSend( AuxBus bus, f32 send )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetFxSend( bus, send );
        }
    }
    //@}



    //! @name ハンドル操作情報設定 / 取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    ハンドルにサウンドが関連付けられているかどうかを調べます。
    //!
    //! @return   ハンドルに関連付けられているサウンドがあれば true を返します。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsAttachedSound() const { return m_pSound != NULL; }

    //---------------------------------------------------------------------------
    //! @brief    ハンドルからサウンドを解放します。
    //!
    //!           ハンドルから開放されたサウンドは、
    //!           その後ハンドルを通して操作できないようになります。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void DetachSound();

    //@}


    //! @name 情報設定 / 取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    サウンドの ID を設定します。
    //!
    //!           ハンドルに関連付けられたサウンドの ID を設定します。
    //!           ハンドルが無効の場合は、何もしません。
    //!
    //!           @ref SoundArchivePlayer でサウンドを再生すると、
    //!           再生開始時にサウンド ID が自動的に設定されます。
    //!           この関数を呼び出すと、ID を上書きして変更します。
    //!
    //!           設定した ID を取得するためには @ref GetId を呼び出します。
    //!
    //! @param[in] id サウンドを識別する ID です。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see GetId
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetId( u32 id )
    {
        if ( IsAttachedSound() ) m_pSound->SetId( id );
    }

    //---------------------------------------------------------------------------
    //! @brief    サウンドの ID を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドに設定されている ID を取得します。
    //!           ハンドルが無効の場合は 0xffffffff を返します。
    //!
    //!           この関数で取得できる ID は @ref SetId で設定された ID です。
    //!
    //! @return   サウンドに設定されている ID を返します。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetId() const
    {
        if ( IsAttachedSound() ) return m_pSound->GetId();
        return internal::BasicSound::INVALID_ID;
    }

    //---------------------------------------------------------------------------
    //! @brief    アンビエントパラメータの値を取得します。
    //!
    //!           3D サウンドを利用している際は、
    //!           3D サウンドで計算された結果がアンビエントパラメータに格納されますので、
    //!           アンビエントパラメータを取得して 3D
    //!           サウンドによるサウンドパラメータの値を知ることができます。
    //!
    //! @return   現在のアンビエントパラメータの値を返します。
    //!           ハンドルが無効の場合は NULL を返します。
    //!
    //! @see      nw::snd::SoundParam 構造体
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const SoundParam* GetAmbientParam() const
    {
        if ( ! IsAttachedSound() ) return NULL;
        return &m_pSound->GetAmbientParam();
    }

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータの情報を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドで再生している、
    //!           ウェーブサウンドデータの情報を取得します。
    //!           ハンドルが無効の場合は false を返します。
    //!
    //!           ウェーブサウンドデータの情報は、
    //!           ウェーブサウンドのプリペア処理が完了するまでは取得することが出来ず、
    //!           その場合は false を返します。
    //!           プリペア処理が完了しているかどうかは @ref IsPrepared で取得できます。
    //!
    //! @param[out] info 結果を格納するウェーブサウンドデータ情報構造体
    //!
    //! @return   ウェーブサウンドデータの情報を取得に成功したら true を、
    //!           取得に失敗したら false を返します。
    //!
    //! @see WaveSoundDataInfo 構造体
    //! @see IsPrepared
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadWaveSoundDataInfo( WaveSoundDataInfo* info ) const
    {
        if ( ! IsAttachedSound() ) return false;
        return m_pSound->ReadWaveSoundDataInfo( info );
    }

    //---------------------------------------------------------------------------
    //! @brief    現在再生中のウェーブサウンドデータの再生位置を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドで再生している、
    //!           ウェーブサウンドデータの再生位置をサンプル数で取得します。
    //!           ハンドルが無効の場合は、負の値を返します。
    //!
    //!           NW4F 1.14.0 以前は、ハードウェアの制約の為に補正され出力された
    //!           バイナリファイルのサンプル位置を返していましたが、
    //!           NW4F 1.14.0 から、元波形ファイルのどの部分を再生しているかを返すようになりました。
    //!
    //!           NW4F 1.14.0 以前の値を取得しなければいけない場合には、
    //!           @ref GetCompatiblePlaySamplePosition() を使用してください。
    //!
    //! @return   ウェーブサウンドデータの再生位置をサンプル数で返します。
    //!
    //! @date 2014/08/21 取得できる値の変更についての説明を追加
    //! @date 2011/12/15 正常に動作するようになったため、下記の記述を削除
    //! @date 2011/08/19 NW4F 1.0.0 RC 現在、正常に動作しない旨追記
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    s32 GetPlaySamplePosition() const
    {
        if ( ! IsAttachedSound() ) return -1;
        return m_pSound->GetPlaySamplePosition(true);
    }

    //---------------------------------------------------------------------------
    //! @brief    現在再生中のウェーブサウンドデータの再生位置を取得します。
    //!
    //!           ハンドルに関連付けられたサウンドで再生している、
    //!           ウェーブサウンドデータの再生位置をサンプル数で取得します。
    //!           ハンドルが無効の場合は、負の値を返します。
    //!
    //!           この関数は非推奨です。代わりに @ref GetPlaySamplePosition() を使用してください。
    //!           この関数との違いなどについては、@ref GetPlaySamplePosition() のリファレンスを参照してください。
    //!
    //! @return   ウェーブサウンドデータの再生位置をサンプル数で返します。
    //!
    //! @date 2014/08/21 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( s32 GetCompatiblePlaySamplePosition() const )
    {
        if ( ! IsAttachedSound() ) return -1;
        return m_pSound->GetPlaySamplePosition(false);
    }

    //@}

    // -----------------------------------------------------------------
    // 非公開関数

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドのサラウンド振り分け方法を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンド振り分け方法を変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         mixMode には MIX_MODE_PAN か MIX_MODE_MIXPARAMETER を設定します。
    //!         MIX_MODE_PAN を指定した場合は、パンとサラウンドパンの値を設定する事で
    //!         サラウンドの振り分けを行います。
    //!         @ref SetChannelMixParameter でサラウンドボリュームを設定していても無視されます。
    //!
    //!         MIX_MODE_MIXPARAMETER を指定した場合は、
    //!         サラウンドボリューム(5.1chの各スピーカーの音量)を指定する必要があります。
    //!         パンやサラウンドパンに値を設定していても無視されます。
    //!
    //! @param[in] mixMode     サラウンド振り分け方法の種類です。
    //!
    //! @see MixMode
    //! @see SetPan
    //! @see SetSurroundPan
    //! @see SetChannelMixParameter
    //! @see SetOutputChannelMixParameter
    //---------------------------------------------------------------------------
    void SetMixMode( MixMode mixMode )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetMixMode( mixMode );
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を変更します。
    //!
    //!         ハンドルに関連付けられたサウンドのサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         この関数でサラウンドボリュームを指定すると、TV, DRC の両方の出力に適用されます。
    //!         出力先別に設定したい場合は、@ref SetOutputChannelMixParameter を利用してください。
    //!
    //! @param[in]    srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in]    param   サラウンドボリュームです。
    //!
    //! @see SetOutputChannelMixParameter
    //---------------------------------------------------------------------------
    void SetChannelMixParameter( u32 srcChNo, const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
            m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    波形の有効なチャンネルに、同一のサラウンドボリュームを設定します。
    //! @param param :private
    void SetChannelMixParameter( const MixParameter& param )
    {
        if( IsAttachedSound() )
        {
            // 有効なチャンネルで同じ設定を使用
            u32 channelCount = m_pSound->GetChannelCount();
            for ( u32 srcChNo = 0; srcChNo < channelCount; srcChNo++ )
            {
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_MAIN, srcChNo, param );
                m_pSound->SetOutputChannelMixParameter( OUTPUT_DEVICE_DRC, srcChNo, param );
            }
        }
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  波形のチャンネル別に、サウンドのサラウンドボリューム(5.1chの各スピーカーの音量)を出力先別で変更します。
    //!
    //!         ハンドルに関連付けられたサウンドの
    //!         メイン (TV) もしくは DRC 出力に対するサラウンドボリュームを変更します。
    //!         ハンドルが無効の場合は、何もしません。
    //!
    //!         param は、サラウンドボリュームの値を設定します。
    //!         全てのチャンネルのサラウンドボリュームを指定する必要があります。
    //!
    //!         デフォルト値は fL, fR, rL, rR, fC, lfe ともに 1.0f です。
    //!         ただし、DRC の場合は fC, lfe の値は無視されます。
    //!
    //!         ミックスモードにパンが指定されている場合は
    //!         この値を指定しても無視されます。
    //!
    //!         TV, DRC の両方の出力にサラウンドボリュームを設定したい場合、@ref SetChannelMixParameter を利用してください。
    //!
    //! @param[in] device  出力先です。
    //! @param[in] srcChNo サラウンドボリュームを指定するチャンネルです。
    //! @param[in] param   サラウンドボリュームです。
    //!
    //! @see SetChannelMixParameter
    //---------------------------------------------------------------------------
    void SetOutputChannelMixParameter( OutputDevice device, u32 srcChNo, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
        }
    }
    //! @briefprivate
    //! @brief    出力先指定で、波形の有効なチャンネルに同一のサラウンドボリュームを設定します。
    //! @param device :private
    //! @param param :private
    void SetOutputChannelMixParameter( OutputDevice device, const MixParameter& param )
    {
        if ( IsAttachedSound() )
        {
            // 有効なチャンネルで同じ設定を使用
            u32 channelCount = m_pSound->GetChannelCount();
            for ( u32 srcChNo = 0; srcChNo < channelCount; srcChNo++ )
            {
                m_pSound->SetOutputChannelMixParameter( device, srcChNo, param );
            }
        }
    }
    //! @briefprivate
    //! @brief    現在再生中のウェーブサウンドデータのチャンネル数を取得します。
    //!           プレイヤーヒープ使用時は、Prepare が完了するまでは不正な値が返るためご注意ください。
    //! @return :private
    u32 GetChannelCount() const
    {
        if ( ! IsAttachedSound() ) return 0;
        return m_pSound->GetChannelCount();
    }

    // ハンドルをサウンドに関連付ける
    //! @briefprivate
    //! @param sound :private
    void detail_AttachSoundAsTempHandle( internal::WaveSound* sound );

    //! @briefprivate
    //! @return :private
    internal::WaveSound* detail_GetAttachedSound() { return m_pSound; }

    //! @briefprivate
    //! @return :private
    const internal::WaveSound* detail_GetAttachedSound() const { return m_pSound; }

private:
    NW_DISALLOW_COPY_AND_ASSIGN( WaveSoundHandle );

    internal::WaveSound* m_pSound;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_WAVE_SOUND_HANDLE_H_ */

