﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_WAVE_SOUND_FILE_H_
#define NW_SND_WAVE_SOUND_FILE_H_

#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_CurveAdshr.h>

namespace nw {
namespace snd {
namespace internal {

/*
    ウェーブサウンドファイル (.bcwsd) の構造

    「ウェーブサウンド」ファイルと呼んでいるが、実際には、
    「ウェーブサウンドセット」１つ分を格納するファイルとなっている。

    bcwsd
     |
     +-- FileHeader
     +-- InfoBlock
          |
          +-- ut::BinaryBlockHeader
          +-- InfoBlockBody
               |
               +-- Table<ref to WaveSoundData>
                    |
                    +-- u32 count
                    +-- ref to WaveSoundData[0] --+ ← このインデックスが、ウェー
                    +-- ref to WaveSoundData[1]   |    ブサウンドセット内でのイン
                    +-- ...                       |    デックスに相当する。
     +--------------------------------------------+
     |
     +-> WaveSoundData
          |
          +-- ref to WaveSoundInfo
          |    |
          |    +--> RawWaveSoundInfo
          |          |
          |          +-- optionParameter (パン、ピッチ、センド量、ADSR)
          |
          +-- ref to TrackInfoTable  <-- 各トラックの楽譜情報が入る
          |    |
          |    +--> Table<ref to TrackInfo>
          |          |
          |          +-- u32 count
          |          +-- ref to TrackInfo[0] --+
          |          +-- ref to TrackInfo[1]   |
          |          +-- ...                   |
          |    +-------------------------------+
          |    |
          |    +--> TrackInfo
          |          |
          |          +-- ref to Table<ref to NoteEvent>
          |               |
          |               +--> Table<ref to NoteEvent>
          |                     |
          |                     +-- u32 count
          |                     +-- ref to NoteEvent[0] --+
          |                     +-- ref to NoteEvent[1]   |
          |                     +-- ...                   |
          |   +-------------------------------------------+
          |   |
          |   +--> NoteEvent
          |         |
          |         +-- position
          |         +-- length
          |         +-- noteIndex    <-- NoteInfoTable のノートのインデックス
          |         +-- reserved
          |
          +-- ref to NoteInfoTable   <-- トラックで使われる全ノートの
               |                               情報が入る
               +--> Table<ref to NoteInfo>
                     |
                     +-- u32 count
                     +-- ref to NoteInfo[0] --+
                     +-- ref to NoteInfo[1]   |
                     +-- ...                     |
              +----------------------------------+
              |
              +--> NoteInfo
                    |
                    +-- waveArchiveId
                    +-- waveIndex
                    +-- optionParameter
                         |
                         +-- originalKey
                         +-- volume
                         +-- pan
                         +-- (surroundPan)
                         +-- pitch
                         +-- sendValue
                         +-- adshrCurve

*/

struct WaveSoundFile
{
    //
    // ヘッダー
    //
    struct InfoBlock;
    struct FileHeader : public Util::SoundFileHeader
    {
        // アクセサ
        const InfoBlock* GetInfoBlock() const;
    };

    //
    // INFO ブロック
    //

    struct WaveSoundData;

    struct InfoBlockBody
    {
        // データ
        Util::Reference toWaveIdTable;
        Util::Reference toWaveSoundDataReferenceTable;

        // アクセサ
        // (テーブル取得)
        const Util::WaveIdTable& GetWaveIdTable() const;
        const Util::ReferenceTable& GetWaveSoundDataReferenceTable() const;

        // (テーブルアイテム数取得)
        NW_INLINE u32 GetWaveIdCount() const
        {
            return GetWaveIdTable().GetCount();
        }
        NW_INLINE u32 GetWaveSoundCount() const
        {
            return GetWaveSoundDataReferenceTable().count;
        }

        // (テーブルアイテム取得)
        NW_INLINE const Util::WaveId* GetWaveId( u32 index ) const
        {
            return GetWaveIdTable().GetWaveId( index );
        }
        const WaveSoundData& GetWaveSoundData( u32 index ) const;
    };

    struct InfoBlock
    {
        ut::BinaryBlockHeader   header;
        InfoBlockBody           body;
    };


    struct WaveSoundInfo;
    struct TrackInfo;
    struct NoteInfo;

    // ウェーブサウンド 1 つ分のデータ
    struct WaveSoundData
    {
        // データ
        Util::Reference toWaveSoundInfo;
        Util::Reference toTrackInfoReferenceTable;
        Util::Reference toNoteInfoReferenceTable;

        // アクセサ
        const WaveSoundInfo& GetWaveSoundInfo() const;

        // (テーブル取得)
        const Util::ReferenceTable& GetTrackInfoReferenceTable() const;
        const Util::ReferenceTable& GetNoteInfoReferenceTable() const;

        // (テーブルアイテム数取得)
        NW_INLINE u32 GetTrackCount() const
        {
            return GetTrackInfoReferenceTable().count;
        }
        NW_INLINE u32 GetNoteCount() const
        {
            return GetNoteInfoReferenceTable().count;
        }

        // (テーブルアイテム取得)
        const TrackInfo& GetTrackInfo( u32 index ) const;
        const NoteInfo& GetNoteInfo( u32 index ) const;
    };

    struct WaveSoundInfo
    {
        // データ
        Util::BitFlag optionParameter;

        // アクセサ
        u8 GetPan() const;
        s8 GetSurroundPan() const;
        f32 GetPitch() const;
        void GetSendValue( u8* mainSend, u8* fxSend, u8 fxSendCount ) const;
        const AdshrCurve& GetAdshrCurve() const;
        u8 GetLpfFreq() const;
        u8 GetBiquadType() const;
        u8 GetBiquadValue() const;
    };

    struct NoteEvent;

    struct TrackInfo
    {
        // データ
        Util::Reference toNoteEventReferenceTable;

        // アクセサ
        // (テーブル取得)
        const Util::ReferenceTable& GetNoteEventReferenceTable() const;

        // (テーブルアイテム数取得)
        NW_INLINE u32 GetNoteEventCount() const
        {
            return GetNoteEventReferenceTable().count;
        }

        // (テーブルアイテム取得)
        const NoteEvent& GetNoteEvent( u32 index ) const;
    };

    struct NoteEvent
    {
#if defined(NW_PLATFORM_CAFE)
        ut::ResF32  position;
        ut::ResF32  length;
        ut::ResU32  noteIndex;
        ut::ResU32  reserved;
#else
        f32  position;
        f32  length;
        u32  noteIndex;
        u32  reserved;
#endif
    };

    struct NoteInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32  waveIdTableIndex;
#else
        u32  waveIdTableIndex;
#endif
        Util::BitFlag optionParameter;

        // アクセサ
        u8 GetOriginalKey() const;
        u8 GetVolume() const;
        u8 GetPan() const;
        u8 GetSurroundPan() const;
        f32 GetPitch() const;
        void GetSendValue( u8* mainSend, u8* fxSend[], u8 fxSendCount ) const;
        const AdshrCurve& GetAdshrCurve() const;
    };
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_WAVE_SOUND_FILE_H_ */

