﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_WAVE_FILE_
#define NW_SND_WAVE_FILE_

#if defined( NW_PLATFORM_CTR )
    #include <nn/types.h>
#elif defined( NW_PLATFORM_CAFE )
    #include <nw/types.h>
#endif

#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Util.h>
#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>      // ut::BinaryFileHeader など
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/ut/ut_BinaryReader.h>          // ResU32 など

namespace nw {
namespace snd {
namespace internal {

/*
    波形ファイル (.bcwav) の構造

    bcwav
     |
     +-- FileHeader
     +-- InfoBlock
     |    |
     |    +-- BinaryBlockHeader
     |    +-- InfoBlockBody
     |         |
     |         +-- u8 encoding
     |         +-- u8 isLoop
     |         +-- u32 sampleRate
     |         +-- u32 loopStart
     |         +-- u32 loopEnd
     |         +-- u32 originalLoopStart
     |         +-- Table< reference to ChannelInfo >
     |              |
     |              +-- s32 count
     |              +-- reference to ChannelInfo[0] ----------------+
     |              +-- reference to ChannelInfo[1]                 |
     |              +-- ... (2ch までしかサポートしていない？！)    |
     |                                                              |
     |    +---------------------------------------------------------+
     |    |
     |    +-> ChannelInfo
     |         |
     |         +-- reference to Samples (DataBlockBody を起点とするサンプルへのオフセット)
     |         +-- reference to AdpcmInfo --+
     |                                      |
     |    +---------------------------------+
     |    |
     |    +-> AdpcmInfo
     |         |
     |         +-- DspAdpcmParam (snd_Global.h)
     |         |    |
     |         |    +-- u16 coef[16]
     |         |    +-- u16 predScale
     |         |    +-- u16 yn1
     |         |    +-- u16 yn2
     |         |
     |         +-- DspAdpcmLoopParam (snd_Global.h)
     |              |
     |              +-- u16 loopPredScale
     |              +-- u16 loopYn1
     |              +-- u16 loopYn2
     |
     +-- DataBlock
          |
          +-- BinaryBlockHeader
          +-- DataBlockBody (サンプルデータ)
*/

struct WaveFile
{
    enum EncodeMethod
    {
        PCM8        = 0,
        PCM16,              // バイトオーダーはプラットフォームに依存
        DSP_ADPCM,
        IMA_ADPCM
    };

    //
    // 前方宣言
    //
    struct InfoBlock;
    struct ChannelInfo;
    struct DspAdpcmInfo;
    struct DataBlock;

    struct FileHeader : public Util::SoundFileHeader
    {
        const InfoBlock* GetInfoBlock() const;
        const DataBlock* GetDataBlock() const;
    };

    struct InfoBlockBody
    {
        u8              encoding;           // EncodeMethod が入る
        u8              isLoop;             // 0: ループなし, 1: ループあり
        u16             padding;
#if defined(NW_PLATFORM_CAFE)
        nw::ut::ResU32  sampleRate;         // サンプリングレート
        nw::ut::ResU32  loopStartFrame;     // ループ開始 (ループなしなら 0)
        nw::ut::ResU32  loopEndFrame;       // ループ終端 (ループなしなら 0)
        nw::ut::ResU32  originalLoopStartFrame;
#else
        u32  sampleRate;                    // サンプリングレート
        u32  loopStartFrame;                // ループ開始 (ループなしなら 0)
        u32  loopEndFrame;                  // ループ終端 (ループなしなら 0)
        u32  originalLoopStartFrame;
#endif
        Util::ReferenceTable channelInfoReferenceTable;

        inline s32 GetChannelCount() const { return channelInfoReferenceTable.count; }
        const ChannelInfo& GetChannelInfo( s32 channelIndex ) const;
    };

    struct InfoBlock
    {
        ut::BinaryBlockHeader   header;
        InfoBlockBody           body;
    };

    struct ChannelInfo
    {
        Util::Reference     referToSamples;     // 起点は DATA ブロックボディ
        Util::Reference     referToAdpcmInfo;
#if defined(NW_PLATFORM_CAFE)
        nw::ut::ResU32      reserved;
#else
        u32      reserved;
#endif

        const void* GetSamplesAddress( const void* dataBlockBodyAddress ) const;
        const DspAdpcmInfo& GetDspAdpcmInfo() const;
    };

    struct DspAdpcmInfo
    {
        DspAdpcmParam      adpcmParam;
        DspAdpcmLoopParam  adpcmLoopParam;
    };

    struct DataBlock
    {
        ut::BinaryBlockHeader   header;
        union
        {
            s8  pcm8[1];
            s16 pcm16[1];
            u8  byte[1];
        };
    };
};  // struct WaveFile

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_WAVE_FILE_ */
