﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_WAV_OUT_RECORDER_H_
#define NW_SND_WAV_OUT_RECORDER_H_

#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_DeviceOutRecorder.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_Util.h>


namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  波形出力を録音するクラスです。
//---------------------------------------------------------------------------
class WavOutRecorder : public DeviceOutRecorder
{
private:
    static const u32 MAX_CHANNELS     = 6;                               //!< 最大チャンネル数
    static const u32 BYTES_PER_SAMPLE  = 2;                              //!< サンプルビット数 : 16bit
    static const u32 MAX_FRAME_LENGTH = MAX_CHANNELS * BYTES_PER_SAMPLE; //!< 最大フレームサイズ
    static const u32 SAMPLES_PER_SEC   = 48000;                          //!< サンプル数 : FinalOutput は 48kHz 固定。


#ifdef NW_PLATFORM_CAFE
    // Cafe の場合、DMA バッファのサンプル格納順は FR, FL, FC, LFE, RR, RL
    static const u32 DMA_FR_INDEX = 0;
    static const u32 DMA_FL_INDEX = 1;
#else
    static const u32 DMA_FL_INDEX = 0;
    static const u32 DMA_FR_INDEX = 1;
#endif
    static const u32 DMA_FC_INDEX  = 2;
    static const u32 DMA_LFE_INDEX = 3;
#ifdef NW_PLATFORM_CAFE
    static const u32 DMA_RR_INDEX  = 4;
    static const u32 DMA_RL_INDEX  = 5;
#else
    static const u32 DMA_RL_INDEX  = 4;
    static const u32 DMA_RR_INDEX  = 5;
#endif

    static const u32 FL_INDEX  = 0;
    static const u32 FR_INDEX  = 1;
    static const u32 FC_INDEX  = 2;
    static const u32 LFE_INDEX = 3;
    static const u32 RL_INDEX  = 4;
    static const u32 RR_INDEX  = 5;

    static const char* DEVICE_NAME;

public:
    //---------------------------------------------------------------------------
    //! @brief         コンストラクタです。
    //---------------------------------------------------------------------------
    WavOutRecorder();

    //---------------------------------------------------------------------------
    //! @brief         デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~WavOutRecorder();

protected:
    virtual u32 GetMaxFrameLength() const { return MAX_FRAME_LENGTH; }

    virtual u32 GetSamplesPerSec() const { return SAMPLES_PER_SEC; }

    virtual u32 GetValidChannels(u32 /* channels */) const
    {
#if defined( NW_PLATFORM_WIN32 ) || defined ( NW_USE_NINTENDO_SDK )
        return 2;
#else
        // 無効なチャンネル数が指定されたら、ステレオ録音扱いとします。
        u32 result = 2;

        // SDK のモードを元にチャンネル数を設定します。
        u32 mode;
        AXPB_ERROR_CODE err = AXGetDeviceMode(AX_DEVICE_TV, &mode);
        NW_ASSERTMSG(err == AXPB_ERROR_NONE,
                "AXGetDeviceMode error(%s)\n", internal::Util::GetAxErrorCodeString(err));

        switch (mode)
        {
        case AX_MODE_STEREO:
        case AX_MODE_MONO:
            result = 2;
            break;
        case AX_MODE_6CHAN:
            result = 6;
            break;
        default:
            NW_ASSERT(mode == AX_MODE_STEREO
                   || mode == AX_MODE_MONO
                   || mode == AX_MODE_6CHAN);
            break;
        }

        return result;
#endif
    }

    virtual void OnStart();
    virtual void OnStop();

    virtual u32 OnProcessSamples(s16* sampleBuffer, u32 samples);

private:
#ifdef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    static void AxAppFrameCallbackFunc();
#endif

    static WavOutRecorder* s_pThis;      //!< AXコールバックから参照するための this ポインタ
};

} // snd
} // nw

#endif // NW_SND_WAV_OUT_RECORDER_H_
