﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut/ut_FileStream.h>

namespace nw {
namespace snd {
namespace internal {

//---------------------------------------------------------------------------
//! @brief  Wavバイナリを出力するファイルストリームクラスです。
//---------------------------------------------------------------------------
class WavOutFileStream : public ut::FileStream
{
private:
    static const u32 FILE_IO_BUFFER_ALIGNMENT;      //!< ファイルIOバッファアライメント

public:
    /* ctor */ WavOutFileStream();
    /* dtor */ virtual ~WavOutFileStream();

public:
    //---------------------------------------------------------------------------
    //! @brief       Wavバイナリ出力ストリームを開きます。
    //!
    //! @param[in]   stream             出力先のストリーム
    //! @param[in]   channels       波形のチャンネル数
    //! @param[in]   samplesPerSec  波形のサンプリングレート
    //!
    //! @return      成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool Open(ut::FileStream& stream, u32 channels, u32 samplesPerSec);

    //---------------------------------------------------------------------------
    //! @brief       Wavバイナリ出力ストリームを閉じます。
    //---------------------------------------------------------------------------
    virtual void Close();

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(同期版)
    //!
    //! @param[in]   buf         書き込み元バッファ
    //! @param[in]   length      書き込みデータのバイト数
    //!
    //! @return      実際に書き込まれたバイト数を返します。
    //!               書き込みに失敗した場合には負のエラーコードを返します。
    //---------------------------------------------------------------------------
    virtual s32 Write(const void* buf, u32 length);

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(非同期版)
    //!
    //! @param[in]   buf         書き込み元バッファへのポインタ
    //! @param[in]   length      書き込むデータ長
    //! @param[in]   callback    非同期処理完了時のコールバック
    //! @param[in]   arg         コールバックに渡されるユーザパラメータ
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool WriteAsync(const void* buf, u32 length, IoStreamCallback callback, void* arg)
    {
        NW_UNUSED_VARIABLE(buf);
        NW_UNUSED_VARIABLE(length);
        NW_UNUSED_VARIABLE(callback);
        NW_UNUSED_VARIABLE(arg);
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタを移動します。
    //!
    //! @param[in]    offset  ファイルポインタ移動値を指定します。
    //! @param[in]    origin  ファイルポインタ移動の基準点を指定します。
    //!                       FILE_STREAM_SEEK_BEGIN      - ファイルの先頭を基準
    //!                       FILE_STREAM_SEEK_CURRENT    - 現在の読み込み位置を基準
    //!                       FILE_STREAM_SEEK_END        - ファイルの終端を基準(offset <= 0で指定されるはずです)
    //!
    //! @return       ファイルポインタが移動できたときは true を返します。
    //---------------------------------------------------------------------------
    virtual bool Seek(s32 offset, u32 origin);

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(同期関数)
    //---------------------------------------------------------------------------
    virtual void Cancel() { m_Stream->Cancel(); }

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(非同期関数)
    //!
    //! @param[in]    callback  処理完了時のコールバック関数です。
    //! @param[out]   arg       コールバックで返されるユーザパラメータです。
    //!
    //! @return      コマンドが正常に発行された場合には true を返します。
    //!              発行されなかった場合には false を返します。
    //---------------------------------------------------------------------------
    virtual bool  CancelAsync(IoStreamCallback callback, void* arg)
    {
        return m_Stream->CancelAsync(callback, arg);
    }

    //---------------------------------------------------------------------------
    //! @brief  非同期処理の可否を取得します。
    //!
    //! @return 常に false を返します。
    //---------------------------------------------------------------------------
    virtual bool CanAsync() const { return false; }

    //---------------------------------------------------------------------------
    //! @brief  リード処理の可否を取得します。
    //!
    //! @return 常に false を返します。
    //---------------------------------------------------------------------------
    virtual bool CanRead() const { return false; }

    //---------------------------------------------------------------------------
    //! @brief  ライト処理の可否を取得します。
    //!
    //! @return ライト処理が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool CanWrite() const { return m_Stream->CanWrite(); }

    //---------------------------------------------------------------------------
    //! @brief  ファイルポインタをランダムに移動できるかどうかの属性を取得します。
    //!
    //! @return ファイルポインタの移動が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool CanSeek() const { return m_Stream->CanSeek(); }

    //---------------------------------------------------------------------------
    //! @brief  ファイルのリードをキャンセルできるかどうかの属性を取得します。
    //!
    //! @return キャンセル可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool CanCancel() const { return m_Stream->CanCancel(); }

    //---------------------------------------------------------------------------
    //! @brief   ストリームがオープンできているかを取得します。
    //!
    //! @return  ストリームがオープンできている場合には true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsAvailable() const
    {
        return m_Stream != NULL && m_Stream->IsAvailable();
    }

    //---------------------------------------------------------------------------
    //! @brief  ファイルサイズを取得します。
    //!
    //! @return ファイルサイズを返します。
    //---------------------------------------------------------------------------
    virtual u32  GetSize() const
    {
        return m_Stream->GetSize() + m_ValidBufferLength;
    }

    //---------------------------------------------------------------------------
    //! @brief  ファイルポインタの位置を取得します。
    //!
    //! @return ファイルポインタの位置（先頭からのバイト数）を返します。
    //---------------------------------------------------------------------------
    virtual u32 Tell() const { return m_Stream->Tell(); }

    //---------------------------------------------------------------------------
    //! @brief         波形データキャッシュ用のバッファを設定します。
    //!
    //! @param[in]     buf    バッファへのポインタ
    //! @param[in]     length バッファの長さ
    //---------------------------------------------------------------------------
    void SetCacheBuffer(u8* buf, u32 length);

private:
    //---------------------------------------------------------------------------
    //! @brief  ヘッダを出力します。
    //---------------------------------------------------------------------------
    bool WriteHeader(u32 channels, u32 samplesPerSec);

    //---------------------------------------------------------------------------
    //! @brief  RIFF チャンクサイズを更新します。
    //---------------------------------------------------------------------------
    bool UpdateRiffChunkSize();

    //---------------------------------------------------------------------------
    //! @brief  data チャンクサイズを更新します。
    //---------------------------------------------------------------------------
    bool UpdateDataChunkSize();

    //---------------------------------------------------------------------------
    //! @brief  RIFF チャンクサイズを計算します。
    //---------------------------------------------------------------------------
   u32 CalcRiffChunkSize(u32 dataSize);

   s32 WriteDirect(const void* buf, u32 length);

   s32 FlushBuffer();

private:
    ut::FileStream* m_Stream;
    u32  m_WaveDataSize;
    bool m_IsWaveDataSizeCalculating;

    u8* m_Buffer;
    u32 m_BufferLength;
    u32 m_ValidBufferLength;
};

} // internal
} // snd
} // nw
